/*
 * asn_int.C - methods for AsnInt (ASN.1 INTEGER) class
 *
 * MS 92/06/16
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#include "asn_config.h"
#include "asn_len.h"
#include "asn_tag.h"
#include "asn_type.h"
#include "asn_int.h"

// Encodes BER content of this AsnInt to the given buffer.
// Returns the number of octets written to the buffer.
AsnLen AsnInt::BEncContent( BUF_TYPE b)
{

    AsnLen            len;
    int               i; 
    unsigned long int   mask;
    unsigned long int   dataCpy;

#define INT_MASK (0x7f80 << ((sizeof(long int) - 2) * 8))

    dataCpy = value;

    /*
     * calculate encoded length of the integer (content)
     */
    mask = INT_MASK;
    if ((long int)dataCpy < 0)
        for (len = sizeof(long int) ; len > 1 ; --len)
        {                
            if ( (dataCpy & mask) == mask)
                mask >>= 8;  
            else
                break;
        }
    else
        for (len = sizeof(long int) ; len > 1 ; --len)
        {                
            if ( (dataCpy & mask) == 0)
                mask >>= 8;  
            else
                break;
        }

    /*
     * write the BER integer
     */
    for (i = 0; i < len; i++)
    {
        b.PutByteRvs( dataCpy);
        dataCpy >>= 8;
    }

    return(len);
}

// Decodes the content of a BER INTEGER from the given buffer.
// The value is placed in this object.  tagId is ignored.
// bytesDecoded is incremented by the number of bytes read for this
// integer value.
void AsnInt::BDecContent( BUF_TYPE b, AsnTag tagId, AsnLen elmtLen,
                               AsnLen& bytesDecoded, ENV_TYPE env)
{
    int   i;
    unsigned long int byte;

    if ( elmtLen > sizeof(long int))
    {
        Asn1Error("AsnInt::BDecContent: ERROR - integer is too big to decode.\n");
        longjmp(env, -7);
    }

    /* 
     * look at integer value 
     */
    byte =  (unsigned long int) b.GetByte();

    if (byte & 0x80)   /* top bit of first byte is sign bit */
        value = (-1 << 8) | byte;
    else
        value = byte;

    /*
     * write from buffer into long int
     */
    for (i = 1; i < elmtLen; i++)
        value = (value << 8) | (unsigned long int)(b.GetByte());

    bytesDecoded += elmtLen;

    if (b.ReadError())
    {
        Asn1Error( "AsnInt::BDecContent: ERROR - decoded past end of data.\n");
        longjmp(env, -8);
    }


}  /* AsnInt::BDecContent */





