/*
 *  asn_bits.C - AsnBits (ASN.1 BIT STRING) Type
 *
 *  Mike Sample
 *  92/07/02
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */


#include "asn_config.h"
#include "asn_len.h"
#include "asn_tag.h"
#include "asn_type.h"
#include "asn_bits.h"
#include "str_stk.h"

extern StrStk strStkG;
unsigned short int  strStkUnusedBitsG;

char numToHexCharTblG[16] = 
    { '0', '1', '2', '3', '4', '5', '6', '7', 
      '8', '9', 'a', 'b', 'c', 'd' ,'e', 'f'};


// Initializes the bits string with a bit string numBits in length.
// All bits are zeroed.
void AsnBits::Set(unsigned long int numBits)
{
    bitLen = numBits;
    bits = Asn1Alloc((bitLen-1)/8 + 1);
    memset(bits, '\0', (bitLen-1)/8 + 1); // init to zeros 
}

// initializes a BIT STRING with the given string and bit length
// Copies the bits from bitsOcts.
void AsnBits::Set(const char* bitOcts, unsigned long int numBits )
{
    unsigned long int octetLen;

    if (bitOcts != bits)
    {
        bitLen = numBits;
        octetLen = (bitLen-1)/8 +1;
        bits = new char[octetLen];
        memcpy(bits, bitOcts, octetLen);
    }
}

// initializes a BIT STRING by copying another BIT STRING's bits
void AsnBits::Set(const AsnBits& b)
{
    unsigned long int octetLen;
    if (&b != this)
    {
        bitLen = b.bitLen;
        octetLen = (bitLen-1)/8 +1;
        bits = new char[octetLen];
        memcpy(bits, b.bits, octetLen);
    }
}

// Initializes the bits string with a bit string numBits in length.
// All bits are zeroed.
void AsnBits::ReSet(unsigned long int numBits)
{
    delete bits;
    Set(numBits);
}

// frees old bits value and then re-initializes the
// BIT STRING with the given string and bit length
// Copies the bitOcts into bits.
void AsnBits::ReSet(const char* bitOcts, unsigned long int numBits )
{
    if (bitOcts != bits)
    {
        delete bits;
        Set(bitOcts, numBits);
    }
}

// frees old bits value and then re-initializes the
// BIT STRING by copying another BIT STRING's bits
void AsnBits::ReSet(const AsnBits& b)
{
    if (&b != this)  // avoid b = b; probs
    {
        delete bits;
        Set(b);
    }
}


// Prints the BIT STRING to the given ostream.
void AsnBits::Print(ostream& os)
{
    unsigned long int octetLen;

    // print bits in hex 
    if (bitLen == 0)
        octetLen = 0;
    else
        octetLen = (bitLen-1)/8 +1;

    os << "'" ;
    for (int i = 0; i < octetLen; i++)
        os << TO_HEX(bits[i] >> 4) << (TO_HEX(bits[i]));
    os << "'H  -- BIT STRING bitlen = " << bitLen << " --" ;

} /* AsnBits::Print */


// Returns TRUE if the given BIT STRING is the same as this one
int AsnBits::BitsEquiv(AsnBits& ab)
{
    unsigned long int octetsLessOne = (bitLen-1)/8;
    unsigned int octetBits = 7 - (bitLen % 8);

    if ( (bitLen == 0) && ab.bitLen == 0)
        return (TRUE);

    // trailing bits may not be significant 
    return ( (bitLen == ab.bitLen) &&
             (memcmp(bits, ab.bits, octetsLessOne) == 0) &&
             (( bits[octetsLessOne] & (0xFF << octetBits)) ==
              ( ab.bits[octetsLessOne] & (0xFF << octetBits))));
}  /* AsnBits::BitsEquiv */


// set given bit to 1. Most signif. bit is bit 0, least signif bit is
// bitLen-1
void AsnBits::SetBit(unsigned long int bit)
{
    unsigned long int octet;
    unsigned long int octetsBit;

    if (bit < bitLen)
    {
        octet = bit/8;
        octetsBit = 7 - (bit % 8); // bit zero is first/most sig bit in octet
        bits[octet] |= 1 << octetsBit;
    }
#ifdef DEBUG
    else
        cerr << "AsnBits::SetBit: ERROR - bit larger than bit string" << endl;
#endif
}  /* AsnBits::SetBit */

// Clr bit. Most signif. bit is bit 0, least signif bit is
// bitLen-1
void AsnBits::ClrBit(unsigned long int bit)
{
    unsigned long int octet;
    unsigned long int octetsBit;

    if (bit < bitLen)
    {
        octet = bit/8;
        octetsBit = 7 - (bit % 8); // bit zero is first/most sig bit in octet
        bits[octet] &= ~(1 << octetsBit);
    }
#ifdef DEBUG
    else
        cerr << "AsnBits::ClrBit: ERROR - bit larger than bit string" << endl;
#endif

}  /* AsnBits::ClrBit */

// returns given bit. Most signif. bit is bit 0, least signif bit is
// bitLen-1.  Returns false if the givnen bit index is out of range.
int  AsnBits::GetBit(unsigned long int bit)
{
    unsigned long int octet;
    unsigned long int octetsBit;

    if (bit < bitLen)
    {
        octet = bit/8;
        octetsBit = 7 - (bit % 8); // bit zero is first/most sig bit in octet
        return (bits[octet] & (1 << octetsBit));
    }
#ifdef DEBUG
    else
        cerr << "AsnBits::GetBit: ERROR - bit larger than bit string" << endl;

#endif
    return(0);    
}  /* AsnBits::GetBit */


// Encoded the content (included unused bits octet) of the BIT STRING
// to the given buffer.
AsnLen AsnBits::BEncContent( BUF_TYPE b)
{
    unsigned long int unusedBits;
    unsigned long int byteLen;

    if (bitLen == 0)
        byteLen = 0;
    else
        byteLen = ((bitLen-1)/8) + 1;

    b.PutSegRvs( bits, byteLen);
    unusedBits = (bitLen % 8);
    if (unusedBits != 0)
        unusedBits = 8 - unusedBits;
    b.PutByteRvs( unusedBits);
    return(byteLen + 1);

} /* AsnBits::BEncContent */


// Decodes a BER BIT STRING from the given buffer and stores
// the value in this object.
void AsnBits::BDecContent( BUF_TYPE b, AsnTag tagId, AsnLen elmtLen,
                               AsnLen& bytesDecoded, ENV_TYPE env)
{
    char* tmp;

    /*
     * tagId is encoded tag shifted into long int.
     * if CONS bit is set then constructed bit string
     */
    if (tagId & 0x20000000)
        BDecConsBits(b, elmtLen, bytesDecoded, env);

    else /* primitive octet string */
    {
        bytesDecoded += elmtLen;
        elmtLen--;
        bitLen = (elmtLen * 8) - (unsigned int)b.GetByte();
        bits =  Asn1Alloc(elmtLen);
        b.Copy( bits, elmtLen);
        if (b.ReadError())
        {
            Asn1Error("BDecBitString: ERROR - decoded past end of data\n");
            longjmp(env, -1);
        }
    }

} /* AsnBits::BDecContent /*


/*
 * Used to concatentate constructed bit strings when decoding.
 *
 * fills string stack with references to the pieces of a
 * construced bit string. sets strStkUnusedBitsG appropriately.
 * and strStkTotalByteLenG to bytelen needed to hold the bitstring
 */
void AsnBits::FillBitStringStk (BUF_TYPE b, AsnLen elmtLen0, 
                  AsnLen& bytesDecoded, ENV_TYPE env)
{
    unsigned long int refdLen;
    unsigned long int totalRefdLen;
    char* strPtr;
    AsnLen totalElmtsLen1 = 0;
    unsigned long int tagId1;
    AsnLen elmtLen1;
    unsigned long int lenToRef;
    unsigned long int unusedBits;

    for ( ; (totalElmtsLen1 < elmtLen0) || (elmtLen0 == INDEFINITE_LEN);)
    {
        tagId1 = BDecTag(b, totalElmtsLen1, env);

        if ( (tagId1 == EOC_TAG_ID) && (elmtLen0 == INDEFINITE_LEN))
        {
            BDEC_2ND_EOC_OCTET(b, totalElmtsLen1, env);
            break; 
        }

        elmtLen1 = BDecLen (b, totalElmtsLen1, env);
        if ( tagId1 == MAKE_TAG_ID( UNIV, PRIM, BITSTRING_TAG_CODE))
        {
            /*
             * primitive part of string, put references to piece(s) in
             * str stack
             */

            /*
             * get unused bits octet 
             */
            if (strStkUnusedBitsG != 0)
            {
                /*  
                 *  whoa - only allowed non-octed aligned bits on
                 *  on last piece of bits string
                 */
                Asn1Error("BDecConsBitString: ERROR - a component of a constructed BIT STRING that is not the last has non-zero unused bits\n");
                longjmp(env, -2);
            }

            if (elmtLen1 != 0)
                strStkUnusedBitsG = b.GetByte();

            totalRefdLen = 0;
            lenToRef =elmtLen1-1; /* remove one octet for the unused bits oct*/
            refdLen = lenToRef;
            while (1)
            {
                strPtr = b.GetSeg( &refdLen);

                strStkG.Push(strPtr, refdLen);
                totalRefdLen += refdLen;
                if (totalRefdLen == lenToRef)
                    break; /* exit this while loop */

                if (refdLen == 0) /* end of data */
                {
                    Asn1Error("BDecConsOctetString: ERROR - expecting more data\n");
                    longjmp(env, -3);
                }
                refdLen = lenToRef - totalRefdLen;
            }
            totalElmtsLen1 += elmtLen1;
        } 

    
        else if ( tagId1 == MAKE_TAG_ID( UNIV, CONS, BITSTRING_TAG_CODE))
        {
            /*
             * constructed octets string embedding in this constructed
             * octet string. decode it.
             */
            FillBitStringStk( b, elmtLen1, totalElmtsLen1, env);
        }
        else  /* wrong tag */
        {
            Asn1Error("BDecConsBitString: ERROR - decoded non-BIT STRING tag inside a constructed BIT STRING\n");
            longjmp(env, -4);
        }
    } /* end of for */

    bytesDecoded += totalElmtsLen1;

}  /* FillBitStringStk */


/*
 * decodes a seq of universally tagged bits until either EOC is 
 * encountered or the given len decoded.  Return them in a
 * single concatenated bit string
 */
void AsnBits::BDecConsBits (BUF_TYPE b, AsnLen elmtLen,
                                 AsnLen& bytesDecoded, ENV_TYPE env)
{
    strStkG.Reset();
    strStkUnusedBitsG = 0;
        
    /*
     * decode each piece of the octet string, puting
     * an entry in the octet/bit string stack for each 
     */
    FillBitStringStk(b, elmtLen, bytesDecoded, env);
    
    /* alloc single str long enough for combined bitstring */
    bitLen = strStkG.totalByteLen*8 - strStkUnusedBitsG;

    bits = Asn1Alloc(strStkG.totalByteLen);

    strStkG.Copy(bits);

}  /* BDecConsBits */




