/* 
 * asn_tag.h - stuff for dealing with tags
 *
 * MS 92
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#ifndef _asn_tag_h_
#define _asn_tag_h_

typedef unsigned long int AsnTag;

/*
 * The MAKE_TAG_ID macro generates the TAG_ID rep for the
 * the given class/form/code (rep'd in integer for)
 * if the class/form/code are constants the the compiler (should)
 * calculate the tag completely --> zero runtime overhead.
 * This is good for efficiently comparing tags in switch statements
 * (decoding) etc.  because run-time bit fiddling (eliminated) minimized
 */
/* Tag Id's byte len */
#define TB sizeof(AsnTag)

#define MAKE_TAG_ID( cl, fm, cd)\
( (cl << ((TB -1) * 8)) | ( fm << ((TB -1) * 8)) | (MAKE_TAG_ID_CODE(cd)))

#define MAKE_TAG_ID_CODE(cd)\
( (cd < 31) ?  (MAKE_TAG_ID_CODE1(cd)):\
      ((cd < 128)?  (MAKE_TAG_ID_CODE2(cd)):\
         ((cd < 16384)?  (MAKE_TAG_ID_CODE3(cd)):\
           (MAKE_TAG_ID_CODE4(cd)))))

#define MAKE_TAG_ID_CODE1(cd)  ((long int)cd << ((TB -1) * 8))
#define MAKE_TAG_ID_CODE2(cd)  ((31l << ((TB -1) * 8)) | (cd << ((TB-2) * 8)))
#define MAKE_TAG_ID_CODE3(cd)  ((31l << ((TB -1) * 8))\
                                | ((cd & 0x3f80) << 9)\
                                | ( 0x0080 << ((TB-2) * 8))\
                                | ((cd & 0x007F) << ((TB-3)* 8)) )

#define MAKE_TAG_ID_CODE4(cd)  ((31l << ((TB -1) * 8))\
                                | ((cd & 0x1fc000) << 2)\
                                | ( 0x0080 << ((TB-2) * 8))\
                                | ((cd & 0x3f80) << 1)\
                                | ( 0x0080 << ((TB-3) * 8))\
                                | ((cd & 0x007F) << ((TB-4)*8)) )
       

typedef enum BER_CLASS
{
    ANY_CLASS = -2,
    NULL_CLASS = -1,
    UNIV = 0,
    APPL = (1 << 6),
    CNTX = (2 << 6),
    PRIV = (3 << 6)
} BER_CLASS;

typedef enum BER_FORM
{
    ANY_FORM = -2,
    NULL_FORM = -1,
    PRIM = 0,
    CONS = (1 << 5)
} BER_FORM;


typedef enum BER_UNIV_CODE
{
    NO_TAG_CODE = 0,
    BOOLEAN_TAG_CODE = 1,
    INTEGER_TAG_CODE,
    BITSTRING_TAG_CODE,
    OCTETSTRING_TAG_CODE,
    NULLTYPE_TAG_CODE,
    OID_TAG_CODE,
    OD_TAG_CODE,
    EXTERNAL_TAG_CODE,
    REAL_TAG_CODE,
    ENUM_TAG_CODE,
    SEQ_TAG_CODE =  16,
    SET_TAG_CODE,
    NUMERICSTRING_TAG_CODE,
    PRINTABLESTRING_TAG_CODE,
    TELETEXSTRING_TAG_CODE,
    VIDEOTEXSTRING_TAG_CODE,
    IA5STRING_TAG_CODE,
    UTCTIME_TAG_CODE,
    GENERALIZEDTIME_TAG_CODE,
    GRAPHICSTRING_TAG_CODE,
    VISIBLESTRING_TAG_CODE,
    GENERALSTRING_TAG_CODE
} BER_UNIV_CODE;

/*
 * the TAG_ID_[CLASS/FORM/CODE] macros are not
 * super fast - try not to use during encoding/decoding
 */
#define TAG_ID_CLASS( tid) ( (tid & (0xC0 << ((TB-1) *8))) >> ((TB -1) * 8))
#define TAG_ID_FORM( tid)  ( (tid & (0x20 << ((TB-1) *8))) >> ((TB -1) * 8))

/*
 * TAG_IS_CONS evaluates to true if the given AsnTag type
 * tag has the constructed bit set.
 */
#define TAG_IS_CONS(tag) ((tag) & (CONS << ((TB-1) *8)))


#define EOC_TAG_ID 0



/*
 * tag encoders.  given constant exprs for class form & code in the 
 * source, these can be optimized by the compiler (eg 
 * do the shifts and bitwise ors etc)
 */

#define BEncTag1( b, class, form, code)\
    1;\
    b.PutByteRvs((class) | (form) | (code))

#define BEncTag2( b, class, form, code)\
    2;\
    b.PutByteRvs( code);\
    b.PutByteRvs( (class) | (form) | 31)

#define BEncTag3( b, class, form, code)\
    3;\
    b.PutByteRvs( (code) & 0x7F);\
    b.PutByteRvs( 0x80 | ((code) >> 7));\
    b.PutByteRvs( (class) | (form) | 31)

#define BEncTag4( b, class, form, code)\
    4;\
    b.PutByteRvs( (code) & 0x7F);\
    b.PutByteRvs( 0x80 | ((code) >> 7));\
    b.PutByteRvs( 0x80 | ((code) >> 14));\
    b.PutByteRvs( (class) | (form) | 31)

#define BEncTag5( b, class, form, code)\
    5;\
    b.PutByteRvs( (code) & 0x7F);\
    b.PutByteRvs( 0x80 | ((code) >> 7));\
    b.PutByteRvs( 0x80 | ((code) >> 14));\
    b.PutByteRvs( 0x80 | ((code) >> 21));\
    b.PutByteRvs( (class) | (form) | 31)


AsnTag BDecTag( BUF_TYPE b, AsnLen& bytesDecoded, ENV_TYPE env);

#endif /* conditional include */
