/*
 *  asn_oid.h  - ASN.1 OBJECT IDENTIFIER type
 *
 *  Mike Sample
 *  92/07/02
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#ifndef _asn_oid_h_
#define _asn_oid_h_

class AsnOid: public AsnType
{
  protected:
    char* oid;
    unsigned long int octetLen;

  public:
    AsnOid(): oid(NULL), octetLen(0) {}

    AsnOid(const char* encOid, unsigned long int len) { Set(encOid, len); }
    AsnOid(const AsnOid& o) { Set(o); }
    AsnOid(unsigned long int a1, unsigned long int a2, 
           long int a3 = -1, long int a4 = -1, long int a5 = -1,
           long int a6 = -1, long int a7 = -1, long int a8 = -1,
           long int a9 = -1, long int a10 = -1, long int a11 = -1)
    {
        Set(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, a11);
    }
    ~AsnOid() { delete oid; }

    AsnOid& operator=(const AsnOid& o) { ReSet(o); return *this;}

    // Set methods overwrite oid and octetLen values
    void Set(const char* encOid, const unsigned long int len);
    void Set(const AsnOid& o);

    // first two arc numbers are mandatory.  rest are optional
    // since negative arc nums are not allowed in the encodings,
    // use them to indicate the 'end of arc numbers' in the 
    // optional parameters
    void Set(unsigned long int a1, unsigned long int a2, 
        long int a3 = -1, long int a4 = -1, long int a5 = -1,
        long int a6 = -1, long int a7 = -1, long int a8 = -1,
        long int a9 = -1, long int a10 = -1, long int a11 = -1);


    // ReSet routines are like Set except the old oid value is freed
    void ReSet(const char* encOid, const unsigned long int len);
    void ReSet(const AsnOid& o);
    void ReSet(unsigned long int a1, unsigned long int a2, 
        long int a3 = -1, long int a4 = -1, long int a5 = -1,
        long int a6 = -1, long int a7 = -1, long int a8 = -1,
        long int a9 = -1, long int a10 = -1, long int a11 = -1);

    operator char*() { return oid; }
    unsigned long int Len() {return octetLen;}
    unsigned long int NumArcs();

    void Print(ostream& os);
    AsnType* Clone() { return new AsnOid; }

    operator==(AsnOid& o)  { return(OidEquiv(o)); }
    operator!=(AsnOid& o)  { return(!OidEquiv(o)); }


    AsnLen BEnc( BUF_TYPE b)
    {
        AsnLen l;
        l =  BEncContent(b); 
        l += BEncDefLen(b, l);
        l += BEncTag1(b, UNIV, PRIM, OID_TAG_CODE);
        return (l);
    }

    void BDec( BUF_TYPE b, AsnLen& bytesDecoded, ENV_TYPE env)
    {
        AsnLen elmtLen;
        if (BDecTag(b, bytesDecoded, env) != MAKE_TAG_ID(UNIV, PRIM, OID_TAG_CODE))
        {
            Asn1Error("AsnOid::BDec: ERROR tag on OBJECT IDENTIFIER is wrong.\n");
            longjmp(env,-57);
        }
        elmtLen = BDecLen(b, bytesDecoded, env);

        BDecContent(b, MAKE_TAG_ID(UNIV, PRIM, OID_TAG_CODE), elmtLen,
                         bytesDecoded, env);
    }


    AsnLen BEncContent( BUF_TYPE b)
    {
        b.PutSegRvs(oid, octetLen);
        return(octetLen);
    }


    void BDecContent( BUF_TYPE b, AsnTag tagId, AsnLen elmtLen,
                          AsnLen& bytesDecoded, ENV_TYPE env);

    PDU_MEMBER_MACROS

    private:
    int OidEquiv(AsnOid o)
    {
        return ( (o.octetLen == octetLen) && 
                 (strncmp(o.oid, oid, octetLen)==0));
    }
    
};

#endif /* conditional include */
