(* Copyright (C) 1994, Digital Equipment Corporation                         *)
(* Digital Internal Use Only                                                 *)
(* All rights reserved.                                                      *)
(*                                                                           *)
(* Last modified on Fri Sep 30 10:53:03 PDT 1994 by najork                   *)
(*       Created on Thu Feb 10 11:47:55 PST 1994 by najork                   *)

(* A camera is a special kind of geometric object. Like any other geometric 
   object, it can be a part of a group (and thus, by transitivity, of an 
   entire scene), properties can be attached to it, and it observes both 
   the properties attached to itself and to its ancestors in the scene. 
   In particular, cameras are affected by transformation properties: When 
   the transformation property of a camera or of one of its ancestors in 
   the scene changes, the camera changes its position. 

   Cameras are themselves invisible.

   "CameraGO.T" is the abstract class of all cameras. *)

INTERFACE CameraGO;

IMPORT GO, Point3, PointProp, RealProp;

TYPE
  T <: GO.T;
(* "CameraGO.T" is a subtype of \type{GO}{T}. No additional fiels or methods 
   are defined. *)

VAR
  From   : PointProp.Name;
  To     : PointProp.Name;
  Up     : PointProp.Name;
  Aspect : RealProp.Name;
(* In addition to the properties observed by all \type{GO}{T}'s, there are four
   additional properties that are observed by all "CameraGO.T"'s. 

   "From" is the name of a property that describes the position of the 
   camera. "To" is the name of a property that defines a point the camera 
   is looking at. "Up" is the name of a property that defines the up-vector 
   of the camera. "From", "To", and "Up" associate with property values of 
   type \type{PointProp}{Val}.

   "Aspect" is the name of a property that defines the aspect ration 
   (width / height) of the image generated by the camera. It associates with 
   a \type{RealProp}{Val}. Its default value is 1, which means that the 
   picture is not distorted
   (In the current implementation, the drawing area is always square). *)

(* The following four procedures make it more convenient to modify the
   "From", "To", "Up", and "Aspect" properties of a camera: *)

PROCEDURE SetFrom (o : GO.T; p: Point3.T);
(* The expression "SetFrom(o,p)" is equivalent to 
   "o.setProp(From.bind(PointProp.NewConst(p)))". *)

PROCEDURE SetTo (o : GO.T; p: Point3.T);
(* The expression "SetTo(o,p)" is equivalent to 
   "o.setProp(To.bind(PointProp.NewConst(p)))". *)

PROCEDURE SetUp (o : GO.T; p: Point3.T);
(* The expression "SetUp(o,p)" is equivalent to 
   "o.setProp(Up.bind(PointProp.NewConst(p)))". *)

PROCEDURE SetAspect (o : GO.T; r: REAL);
(* The expression "SetAspect(o,r)" is equivalent to 
   "o.setProp(Aspect.bind(RealProp.NewConst(r)))". *)

END CameraGO.
