(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(*  Last modified on Tue Jun 16 10:15:31 PDT 1992 by muller                  *)
(*       modified on Tue Nov 26 18:21:21 PST 1991 by meehan                  *)

MODULE FilterRd;

IMPORT Rd, RdClass, Thread;

REVEAL
  T = Public BRANDED "FilterRd.T" OBJECT
        rd        : Rd.T;
        closeChild: BOOLEAN
      OVERRIDES
        init   := Init;
        length := Length;
        seek   := Seek;
        close  := Close;
      END;

PROCEDURE Init (z: T; rd: Rd.T; closeChild := FALSE): T =
  BEGIN
    RdClass.Lock (rd);
    TRY
      z.rd := rd;
      z.closeChild := closeChild;
      CopyFields (from := rd, to := z);
      RETURN z
    FINALLY
      RdClass.Unlock (rd)
    END
  END Init;

PROCEDURE Seek (z: T; dontBlock: BOOLEAN): RdClass.SeekResult
  RAISES {Rd.Failure, Thread.Alerted} =
  BEGIN
    RdClass.Lock (z.rd);
    TRY
      CopyFields (from := z, to := z.rd);
      WITH result = z.rd.seek (dontBlock) DO
        CopyFields (from := z.rd, to := z);
        RETURN result
      END
    FINALLY
      RdClass.Unlock (z.rd)
    END
  END Seek;

PROCEDURE Length (z: T): CARDINAL RAISES {Rd.Failure, Thread.Alerted} =
  BEGIN
    RdClass.Lock (z.rd);
    TRY
      CopyFields (from := z, to := z.rd);
      WITH result = z.rd.length () DO
        CopyFields (from := z.rd, to := z);
        RETURN result
      END
    FINALLY
      RdClass.Unlock (z.rd)
    END
  END Length;

PROCEDURE Close (z: T) RAISES {Rd.Failure, Thread.Alerted} =
  BEGIN
    z.buff := NIL;
    z.closed := TRUE;
    IF z.closeChild THEN Rd.Close (z.rd) END
  END Close;

PROCEDURE CopyFields (to, from: Rd.T) =
  BEGIN
    to.buff := from.buff;
    to.st := from.st;
    to.lo := from.lo;
    to.cur := from.cur;
    to.hi := from.hi;
    to.closed := from.closed;
    to.seekable := from.seekable;
    to.intermittent := from.intermittent;
  END CopyFields;

BEGIN
END FilterRd.


