(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Sun May 14 20:21:12 1989               *)
(* Last modified on Tue Jun  2 15:56:54 PDT 1992 by stolfi     *)
(*      modified on Tue Feb 11 21:43:42 PST 1992 by muller     *)


MODULE ColorNameTest EXPORTS Main;

IMPORT 
  Wr, Fmt, RGB, RGBIO, RGBDist, List, Rd, Thread,
  ColorName, Text, Random;
FROM Stdio IMPORT stderr;

PROCEDURE Main() =

  PROCEDURE TestColor(name1: Text.T; rgb1: RGB.T) =
    <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
    BEGIN
      TRY
        WITH
          name2 = ColorName.FromRGB(rgb1),
          rgb2 = ColorName.ToRGB(name2)
        DO
          Wr.PutText(stderr, "    " & 
            Fmt.Pad(name1, 40, align := Fmt.Align.Left) & " = " & 
            RGBIO.ToText(rgb1) & "\n"
          );
          Wr.PutText(stderr, " -> " & 
            Fmt.Pad(name2, 40, align := Fmt.Align.Left) & " = " & 
            RGBIO.ToText(rgb2) & "\n"
          );
          Wr.PutText(stderr, "    ** distance = " &
            Fmt.Real(RGBDist.Perceptual(rgb1, rgb2), 4) & "\n"
          );
          Wr.PutText(stderr, "\n");
        END 
      EXCEPT
      | ColorName.NotFound => 
          Wr.PutText(stderr, " ** ColorName.NotFound -- " & name1 & "\n")
      END; 
    END TestColor;

  VAR basic, full: TEXT;
      p, names: List.T;
      modCode, fracCode: CARDINAL;
      c: RGB.T;
  <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted, ColorName.NotFound *>
  BEGIN
    Init();
    names := ColorName.NameList();
    p := names;
    WHILE p # NIL DO
      basic := NARROW(p^.first, TEXT);
      TestColor(basic, ColorName.ToRGB(basic));
      IF Random.Real(NIL) > 0.75 THEN
        (* Try some prefixes: *)
        REPEAT
          modCode := Random.Subrange(NIL, 0, 11);
          CASE modCode OF
          |  0 => full := basic;
          |  1 => full := "Dark" & basic;
          |  2 => full := "Dim" & basic;
          |  3 => full := "Drab" & basic;
          |  4 => full := "Dull" & basic;
          |  5 => full := "Weak" & basic;
          |  6 => full := "Pale" & basic;
          |  7 => full := "Light" & basic;
          |  8 => full := "Vivid" & basic;
          |  9 => full := "Strong" & basic;
          | 10 => full := "Bright" & basic;
          | 11 => full := "Medium" & basic;
          ELSE <* ASSERT FALSE *>
          END;
          IF (modCode # 0) AND (modCode # 11) THEN
            fracCode := Random.Subrange(NIL, 0, 9);
            CASE fracCode OF
            |  0 => full := full;
            |  1 => full := "VeryVerySlightly" & full;
            |  2 => full := "VerySlightly" & full;
            |  3 => full := "Slightly" & full;
            |  4 => full := "Somewhat" & full;
            |  5 => full := "Rather" & full;
            |  6 => full := "Quite" & full;
            |  7 => full := "Very" & full;
            |  8 => full := "VeryVery" & full;
            |  9 => full := "VeryVeryVery" & full;
            ELSE <* ASSERT FALSE *>
            END;
          ELSE
            fracCode := 0
          END;
          TestColor(full, ColorName.ToRGB(full));
        UNTIL Random.Real(NIL) <= 0.25;
      END;
      p := p^.tail
    END;
    
    (* Now some random RGB values: *)
    FOR i := 0 TO 100 DO 
      FOR i := 0 TO 2 DO c[i] := Random.Real(NIL) END;
      TestColor("Random", c)
    END;
    Wr.PutText(stderr, "Done.\n");
  END Main;

PROCEDURE Init() =
  BEGIN
  END Init;

BEGIN
  Main();
END ColorNameTest.

