(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(* Created by stolfi on Mon Mar 28 20:52:56 1988               *)
(* Last modified on Mon Jun 15 21:51:22 1992 by mhb        *)
(*      modified on Tue Jun  2 13:44:36 PDT 1992 by stolfi     *)
(*      modified on Tue Feb 11 21:39:51 PST 1992 by muller     *)


(* The "Color" interface defines a generic representation for color values, 
   as real triples in some color coordinate system. In particular, colors
   in the RGB, HSV, and CIE models were defined as a "Color.T"s. *)

INTERFACE Color;

IMPORT Intensity;

TYPE 
  T = ARRAY [0 .. 2] OF REAL;
(* A color triple, in some color coordinate system. *)

CONST 
  Undefined = T{
    Intensity.Undefined, 
    Intensity.Undefined, 
    Intensity.Undefined
  };
(* Some applications need a special color value meaning
   ``undefined'', ``vacant'', ``don't care'', and so on.  The
   "Undefined" triple above is meant for this purpose; it lies
   well outside the unit cube, and is unlikely to be generated by
   coincidence. The "Intensity" interface appears in 
   Section~\ref{IntensitySection}. *)

CONST
  Default = T{
    Intensity.Default, 
    Intensity.Default, 
    Intensity.Default
  };
(* A reasonable default value, for the sake of generic modules *)

TYPE 
  DistFn = PROCEDURE (READONLY a, b: T; parm: REAL): REAL;
(* The signature of a generic color distance function.  The
   parameter "parm" can be used to tune the distance for
   particular applications, e.g. width of border between the two
   colors, contrast with background, etc. *)

TYPE 
  Byte = BITS 8 FOR [0..255];
  Bytes = ARRAY [0..BYTESIZE(T)-1] OF Byte;

PROCEDURE ToBytes(x: T): Bytes;
(* A safe "LOOPHOLE" of "x" into "ARRAY OF BYTE". *)

END Color.

