(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Revelation.m3                                         *)
(* Last modified on Wed Jul 22 17:34:12 1992 by kalsow     *)
(*      modified on Sat Aug 25 02:55:44 1990 by muller         *)

MODULE Revelation;

IMPORT String, Value, Type, Error, OpaqueType, Scope, Decl;
IMPORT ObjectType, RefType, Emit, Scanner, Token, Module, ValueRep;
FROM Scanner IMPORT GetToken, Fail, Match, Match1, MatchID, cur;

TYPE
  T = UNTRACED BRANDED "Revelation.T" REF RECORD
        home    : String.T; (* name of the containing interface *)
        env     : Scope.T;
        qid     : String.QID;
        obj     : Value.T; (* value named by 'qid' in scope 'env' *)
        rhs     : Type.T;  (* REVEAL qid (<:|=) rhs *)
        lhs     : Type.T;  (* == type that corresponds to qid *)
        equal   : BOOLEAN; (* TRUE => lhs = rhs, FALSE => lhs <: rhs *)
        checked : BOOLEAN;
        origin  : INTEGER;
      END;

TYPE
  List = UNTRACED BRANDED "Revelation.List" REF RECORD
           next  : List;
	   ident : T;
	   local : BOOLEAN; (* as opposed to inherited *)
           used  : BOOLEAN;
           home  : Value.T; (* External.T that caused the import *)
         END;

TYPE
  Node = UNTRACED BRANDED "Revelation.Node" REF RECORD
           next     : Node;
           key      : Type.T;
           best     : List     := NIL;
           contents : List     := NIL;
           reducing : BOOLEAN  := FALSE;
         END;

TYPE
  HashTable = UNTRACED REF ARRAY OF Node;

REVEAL
  Set = UNTRACED BRANDED "Revelation.Set" REF RECORD
          home   : String.T   := NIL;
          count  : INTEGER    := 0;
          idents : List       := NIL;
          hash   : HashTable  := NIL;
          (* the visible revelations are in the union of indents and hash *)
        END;

VAR top: Set := NIL;

PROCEDURE NewSet (): Set =
  BEGIN
    RETURN NEW (Set);
  END NewSet;

PROCEDURE Push (s: Set): Set =
  VAR old := top;
  BEGIN
    <* ASSERT s # NIL *>
    top := s;
    RETURN old;
  END Push;

PROCEDURE Pop (s: Set) =
  BEGIN
    top := s;
  END Pop;

PROCEDURE SetName (s: Set;  name: String.T) =
  BEGIN
    <* ASSERT name # NIL *>
    s.home := name;
  END SetName;


PROCEDURE Parse (READONLY fail: Token.Set; att: Decl.Attributes) =
  TYPE TK = Token.T;
  VAR fail2: Token.Set;  id, id2: String.T;  loc: INTEGER;
  BEGIN
    IF att.isExternal THEN Error.Msg ("a revelation cannot be external"); END;
    IF att.isInline   THEN Error.Msg ("a revelation cannot be inline"); END;
    IF att.isObsolete THEN Error.Msg ("a revelation cannot be obsolete"); END;
    IF att.isUnused   THEN Error.Msg ("a revelation cannot be unused"); END;

    Match (TK.tREVEAL, fail, Token.Set {TK.tSEMI});
    fail2 := fail + Token.Set {TK.tSEMI};
    WHILE (cur.token = TK.tIDENT) DO
      id2 := NIL;
      id  := MatchID (fail2, Token.TypeStart
                               + Token.Set {TK.tEQUAL, TK.tDOT, TK.tSUBTYPE});
      IF (cur.token = TK.tDOT) THEN
        GetToken (); (* . *)
        id2 := id;
        id := MatchID (fail2, Token.Set {TK.tEQUAL, TK.tSUBTYPE});
      END;
      loc := Scanner.offset;
      CASE cur.token OF
      | TK.tEQUAL =>
          GetToken (); (* = *)
          New (id2, id, Type.Parse (fail2), TRUE, loc);
      | TK.tSUBTYPE =>
          GetToken (); (* <: *)
          New (id2, id, Type.Parse (fail2), FALSE, loc);
      ELSE Fail ("missing \'=\' or \'<:\'", fail);
      END;
      Match1 (TK.tSEMI, fail);
    END;
  END Parse;

PROCEDURE New (module, name: String.T;  rhs: Type.T;
                                                 eq: BOOLEAN;  loc: INTEGER) =
  VAR t: T;
  BEGIN
    <* ASSERT top.home # NIL *>
    t := NEW (T);
    t.home       := top.home;
    t.qid.module := module;
    t.qid.item   := name;
    t.obj        := NIL;
    t.rhs        := rhs;
    t.lhs        := NIL;
    t.equal      := eq;
    t.checked    := FALSE;
    t.origin     := loc;
    t.env        := Scope.Top ();
    AddOne (t, TRUE, NIL);
  END New;

PROCEDURE Inherit (s: Set;  import: Value.T) =
  VAR x: List;  n: Node;
  BEGIN
    x := s.idents;
    WHILE (x # NIL) DO
      IF (x.local) THEN AddOne (x.ident, FALSE, import) END;
      x := x.next;
    END;
    IF (s.hash # NIL) THEN
      FOR i := 0 TO LAST (s.hash^) DO
        n := s.hash[i];
        WHILE (n # NIL) DO
          x := n.contents;
          WHILE (x # NIL) DO
            IF (x.local) THEN AddOne (x.ident, FALSE, import) END;
            x := x.next;
          END;
          n := n.next;
        END;
      END;
    END;
  END Inherit;

PROCEDURE AddOne (t: T;  isLocal: BOOLEAN;  import: Value.T) =
  VAR y := NEW (List, ident := t, local := isLocal,
                         used := FALSE, home := import);
  BEGIN
    y.next := top.idents;
    top.idents := y;
    INC (top.count);
  END AddOne;

PROCEDURE TypeCheck (s: Set) =
  VAR l: List;  save, n_buckets: INTEGER;  n: Node;
  BEGIN
    IF (s.count <= 0) THEN RETURN END;

    save := Scanner.offset;

    (* allocate and initialize the hash table *)
    n_buckets := MIN (2 * s.count,  OpaqueType.Population() + 2);
    s.hash := NEW (HashTable, n_buckets);
    FOR i := 0 TO n_buckets - 1 DO s.hash[i] := NIL END;

    (* bind the lhs qid's to types & map them into the hash table *)
    WHILE (s.idents # NIL) DO
      l := s.idents;
      DoBind (l.ident);
      s.idents := l.next;
      HashInsert (s, l);
    END;

    (* type check the lhs and rhs *)
    FOR i := 0 TO n_buckets-1 DO
      n := s.hash[i];
      WHILE (n # NIL) DO
        l := n.contents;
        WHILE (l # NIL) DO DoCheck0 (l.ident); l := l.next;  END;
        n := n.next;
      END;
    END;

    (* make sure that the lhs and rhs are compatible *)
    FOR i := 0 TO n_buckets-1 DO
      n := s.hash[i];
      WHILE (n # NIL) DO
        l := n.contents;
        WHILE (l # NIL) DO DoCheck (l.ident); l := l.next;  END;
        n := n.next;
      END;
    END;

    (* find the strongest revelation for each type *)
    FOR i := 0 TO n_buckets-1 DO
      n := s.hash [i];
      WHILE (n # NIL) DO
        Reduce (s, n.key);
        n := n.next;
      END;
    END;

    Scanner.offset := save;
  END TypeCheck;

PROCEDURE DoBind (t: T) =
  VAR obj: Value.T;
  BEGIN
    IF (t.checked) THEN RETURN END;
    Scanner.offset := t.origin;
    obj := Scope.LookUpQID (t.env, t.qid);
    t.obj := obj;
    IF (obj = NIL) THEN
      Error.QID (t.qid, "undefined");
      t.lhs := t.rhs;
    ELSIF (Value.ClassOf (obj) # Value.Class.Type) THEN
      Error.QID (t.qid, "is not a type");
      t.lhs := t.rhs;
    ELSE
      t.lhs := Value.ToType (obj);
      IF (t.equal) THEN Type.NoteDeclaration (t.rhs, obj); END;
    END;
    t.lhs := Type.Strip (t.lhs);
  END DoBind;

PROCEDURE HashInsert (s: Set;  l: List) =
  VAR lhs := l.ident.lhs;
  VAR hsh := OpaqueType.UID (lhs) MOD NUMBER (s.hash^);
  VAR n   := s.hash [hsh];
  VAR x: List;
  BEGIN
    (* look for the node that contains l's revelations *)
    LOOP
      IF (n = NIL) THEN
        (* we didn't find a node for this type *)
        n := NEW (Node, next := s.hash[hsh], key := lhs);
        s.hash [hsh] := n;
        EXIT;
      END;
      IF (n.key = lhs) THEN EXIT END;
      n := n.next;
    END;

    (* check for a duplicate revelation (possible because both "IMPORT X"
       and "FROM X IMPORT" are allowed in a single unit.  They cause X's
       revelations to be inherited twice.  sigh. *)
    x := n.contents;
    WHILE (x # NIL) DO
      IF (x.ident = l.ident) THEN (* drop 'l' on the floor *) RETURN END;
      x := x.next;
    END;

    (* add 'l' to the list *)
    l.next := n.contents;
    n.contents := l;
  END HashInsert;

PROCEDURE DoCheck0 (t: T) =
  BEGIN
    Scanner.offset := t.origin;
    Type.Check (t.rhs);
    Type.Check (t.lhs);
  END DoCheck0;

PROCEDURE DoCheck (t: T) =
  VAR  xx: Type.T;
  BEGIN
    IF (t.checked) THEN RETURN END;
    Scanner.offset := t.origin;

    IF (NOT OpaqueType.Is (t.lhs)) THEN
      Error.QID (t.qid, "is not an opaque type");
    ELSIF NOT Type.IsSubtype (t.rhs, OpaqueType.Super (t.lhs)) THEN
      Error.QID (t.qid, "identification is not to a legal subtype");
    END;

    IF (t.equal) THEN
      xx := Type.Strip (t.rhs);
      IF (xx # t.rhs)
        OR NOT (RefType.IsBranded (xx) OR ObjectType.IsBranded (xx)) THEN
        Error.QID (t.qid, "right-hand side must be a branded type expression");
        t.rhs := xx;
      END;
    END;

    t.checked := TRUE;
  END DoCheck;

PROCEDURE Reduce (s: Set;  key: Type.T) =
  VAR x: INTEGER;  n: Node;  best, l: List;
  BEGIN
    IF (key = NIL) THEN RETURN END;
    key := Type.Strip (key);

    x := OpaqueType.UID (key);
    IF (x = 0) THEN
      (* it's not an opaque type *)
      Reduce (s, ObjectType.Super (key));
      RETURN;
    END;
    Reduce (s, OpaqueType.Super (key));

    (* find the hash table node *)
    x := x MOD NUMBER (s.hash^);
    n := s.hash [x];
    LOOP
      IF (n = NIL) THEN RETURN END;
      IF (n.key = key) THEN EXIT END;
      n := n.next;
    END;

    IF (n.reducing) THEN (*recursive call*) RETURN END;
    IF (n.best # NIL) THEN (*done*) RETURN END;
    n.reducing := TRUE;

    (* first, reduce the rhs's *)
    l := n.contents;
    WHILE (l # NIL) DO
      Reduce (s, l.ident.rhs);
      l := l.next;
    END;

    (* finally, search for the best candidate *)
    best := n.contents;
    l := best.next;
    WHILE (l # NIL) DO

      Type.Check (best.ident.rhs);
      Type.Check (l.ident.rhs);
      (* We need these checks since Reduce can be called by
         LookUpAll during an active call to  TypeCheck. *)

      IF Type.IsSubtype (best.ident.rhs, l.ident.rhs) THEN
        (* best is better than l *)
        IF (l.ident.equal) THEN TooStrong (l, best) END;
      ELSIF Type.IsSubtype (l.ident.rhs, best.ident.rhs) THEN
        (* l is better than best *)
        IF (best.ident.equal) THEN TooStrong (best, l) END;
        best := l;
      ELSE (* unrelated revelations! *)
        Scanner.offset := best.ident.origin;
        Error.QID (best.ident.qid, "non-comparable revelation");
        Scanner.offset := l.ident.origin;
        Error.QID (l.ident.qid, "non-comparable revelation");
      END;
      l := l.next;
    END;

    n.best := best;
  END Reduce;

PROCEDURE TooStrong (xa, xb: List) =
  (* a.ident.equal *)
  VAR a := xa.ident;  b := xb.ident;
  BEGIN
    <*ASSERT a.equal *>
    IF (b.equal) THEN
      Scanner.offset := a.origin;
      Error.QID (a.qid, "multiple full revelations");
      Scanner.offset := b.origin;
      Error.QID (b.qid, "multiple full revelations");
    ELSE
      Scanner.offset := b.origin;
      Error.QID (b.qid, "partial revelation is stronger than full revelation");
    END;
  END TooStrong;

PROCEDURE LookUp (key: Type.T): Type.T =
  VAR h: INTEGER;  x: Type.T;
  BEGIN
    key := Type.Strip (key);
    IF (top.hash # NIL) THEN
      h := OpaqueType.UID (key) MOD NUMBER (top.hash^);
      x := SearchEQ (top.hash [h], key);
      IF (x # NIL) THEN RETURN x END;
    END;
    RETURN SearchListEQ (top.idents, key);
  END LookUp;

PROCEDURE SearchEQ (n: Node;  key: Type.T): Type.T =
  VAR l: List;  t: T;
  BEGIN
    (* look for the chain header *)
    LOOP
      IF (n = NIL) THEN RETURN NIL END;
      IF (n.key = key) THEN EXIT END;
      n := n.next;
    END;

    (* has it already been reduced? *)
    l := n.best;
    IF (l # NIL) THEN
      t := l.ident;
      IF NOT t.equal THEN RETURN NIL END;
      IF (NOT t.checked) THEN CheckRHS (t) END;
      NoteUse (l);
      RETURN t.rhs;
    END;

    (* no, then search the full list for a match *)
    RETURN SearchListEQ (n.contents, key);
  END SearchEQ;

PROCEDURE SearchListEQ (l: List;  key: Type.T): Type.T =
  VAR t: T;
  BEGIN
    WHILE (l # NIL) DO
      t := l.ident;
      <* ASSERT t.lhs # NIL OR t.rhs = NIL *> (* => LHS is bound *)
      IF (t.equal) AND (t.lhs = key) THEN
        IF (NOT t.checked) THEN CheckRHS (t) END;
        NoteUse (l);
        RETURN t.rhs;
      END;
      l := l.next;
    END;
    RETURN NIL; (* didn't find a full revelation *)
  END SearchListEQ;

PROCEDURE LookUpAll (key: Type.T): TypeList =
  VAR matches: TypeList := NIL;  h: INTEGER;
  BEGIN
    key := Type.Strip (key);
    IF (top.idents # NIL) THEN
      SearchListAll (top.idents, key, matches);
    END;
    IF (top.hash # NIL) THEN
      Reduce (top, key);
      h := OpaqueType.UID (key) MOD NUMBER (top.hash^);
      SearchAll (top.hash [h], key, matches);
    END;
    RETURN matches;
  END LookUpAll;

PROCEDURE SearchAll (n: Node;  key: Type.T;  VAR matches: TypeList) =
  VAR t: T;  z: TypeList;
  BEGIN
    (* search the list for a matching node *)
    LOOP
      IF (n = NIL) THEN RETURN END;
      IF (n.key = key) THEN EXIT END;
      n := n.next;
    END;

    IF (n.best # NIL) THEN
      NoteUse (n.best);
      t := n.best.ident;
      <* ASSERT t.lhs = key *>
      IF (NOT t.checked) THEN CheckRHS (t) END;
      z := NEW (TypeList);
      z.next := matches;
      z.type := t.rhs;
      matches := z;
    ELSE
      (* we haven't reduced this node yet => return all possible nodes *)
      SearchListAll (n.contents, key, matches);
    END;
  END SearchAll;

PROCEDURE SearchListAll (l: List;  key: Type.T;  VAR matches: TypeList) =
  VAR t: T;  z: TypeList;
  BEGIN
    WHILE (l # NIL) DO
      t := l.ident;
      <* ASSERT t.lhs # NIL OR t.rhs = NIL *> (* => LHS is bound *)
      IF (t.lhs = key) THEN
        NoteUse (l);
        IF (NOT t.checked) THEN CheckRHS (t) END;
        z := NEW (TypeList);
        z.next := matches;
        z.type := t.rhs;
        matches := z;
      END;
      l := l.next;
    END;
  END SearchListAll;

PROCEDURE NoteUse (l: List) =
  BEGIN
    IF (Module.depth = 1) THEN
      l.used := TRUE;
      IF (l.home # NIL) THEN l.home.used := TRUE END;
    END;
  END NoteUse;

PROCEDURE CheckRHS (t: T) =
  (* we're doing a lookup while the revelations are being checked... *)
  VAR save := Scanner.offset;
  BEGIN
    Type.Check (t.rhs);
    Scanner.offset := save;
  END CheckRHS;

PROCEDURE Declare (s: Set) =
  VAR n: Node;  l: List;  save: Emit.Stream;
  BEGIN
    <*ASSERT s.idents = NIL *>

    (* declare the underlying types *)
    IF (s.hash # NIL) THEN
      FOR i := 0 TO LAST (s.hash^) DO
        n := s.hash[i];
        WHILE (n # NIL) DO
          l := n.contents;
          WHILE (l # NIL) DO
            IF (l.used) OR (l.local) THEN DeclareTypes (l.ident) END;
            l := l.next;
          END;
          n := n.next;
        END;
      END;
    END;

    save := Emit.Switch (Emit.Stream.LinkTables);

    (* generate the revelations defined or used in the current module *)
    DeclareList (s, TRUE,  TRUE,  "_exported_full_revelations");
    DeclareList (s, TRUE,  FALSE, "_exported_partial_revelations");
    DeclareList (s, FALSE, TRUE,  "_imported_full_revelations");
    DeclareList (s, FALSE, FALSE, "_imported_partial_revelations");

    EVAL Emit.Switch (Emit.Stream.LinkerTypes);

    (* generate the link info for the revelations defined or used here *)
    IF (s.hash # NIL) THEN
      FOR i := 0 TO LAST (s.hash^) DO
        n := s.hash[i];
        WHILE (n # NIL) DO
          l := n.contents;
          WHILE (l # NIL) DO
            IF (l.used) OR (l.local) THEN
              DeclareRevelation (l.ident, l.local);
            END;
            l := l.next;
          END;
          n := n.next;
        END;
      END;
    END;

    EVAL Emit.Switch (save);
  END Declare;

PROCEDURE DeclareTypes (t: T) =
  BEGIN
    Value.Declare0 (t.obj);
    Type.Compile (t.rhs);
    Type.Compile (t.lhs);
  END DeclareTypes;

PROCEDURE DeclareList (s: Set;  local, full: BOOLEAN;  name: TEXT) =
  VAR cnt := 0;  t: T;  l: List;  n: Node;
  BEGIN
    IF (s.hash # NIL) THEN
      FOR i := 0 TO LAST (s.hash^) DO
        n := s.hash[i];
        WHILE (n # NIL) DO
        l := n.contents;
          WHILE (l # NIL) DO
            t := l.ident;
            IF (l.local= local) AND (t.equal = full) AND (local OR l.used) THEN
              IF (cnt = 0) THEN
                Emit.OpX("_PRIVATE _TYPE_PAIR @[] = {\n",name);
              END;
              Emit.OpFF ("  { &@_TC, &@_TC, ", t.lhs, t.rhs);
              Emit.OpHH ("0x@, 0x@ },\n", Type.Name (t.lhs), Type.Name(t.rhs));
              INC (cnt);
            END;
            l := l.next;
          END;
          n := n.next;
        END;
      END;
    END;
    IF (cnt = 0)
      THEN Emit.OpX ("\003#define @ 0\n", name);
      ELSE Emit.Op (" { 0, 0, 0, 0 }\n};\n");
    END;
  END DeclareList;

PROCEDURE DeclareRevelation (t: T;  exported: BOOLEAN) =
  CONST export_fmt = ARRAY BOOLEAN OF TEXT { "X@ @\n", "R@ @\n" };
  CONST import_fmt = ARRAY BOOLEAN OF TEXT { "x@",     "r@" };
  BEGIN
    IF (exported) THEN
      Emit.OpFF (export_fmt [t.equal], t.lhs, t.rhs);
    ELSE (*imported*)
      <*ASSERT t.home # NIL *>
      Emit.OpS (import_fmt [t.equal], t.home);
      Emit.OpFF (" @ @\n", t.lhs, t.rhs);
    END;
  END DeclareRevelation;

PROCEDURE Initialize () =
  BEGIN
    top := NewSet ();
    SetName (top, String.Add ("M3_BUILTIN"));
  END Initialize;

BEGIN
END Revelation.
