MODULE SIO; (*10.05.94. LB*)
(* Implements the procedures to read and write streams.
   Error handling is somewhat strange: the idea was 
   to protect beginners from being confronted with exceptions. *)

(* Procedure Available() modified by J.Jerney in July 1994. *)

  IMPORT
    Rd, Wr, Stdio, Lex, Fmt, c_funcs;

 <*FATAL Rd.Failure, Rd.EndOfFile, Wr.Failure*>

  TYPE
    Set = SET OF CHAR;
  CONST
    T1 = "\nThis is not ";
    T2 = " try again \n";
    null = '\000';
    Umlaute = Set{'', '', '', '', '', '', ''};
    NonBlanks = Set{'!' .. '~'} + Umlaute;

  PROCEDURE LookAhead(rd: Reader := NIL): CHAR RAISES {Error} =
  VAR ch: CHAR;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    ch:= GetChar(rd);
    TRY
      Rd.UnGetChar(rd);
    EXCEPT ELSE
      IF rd # Stdio.stdin THEN RAISE Error ELSE RETURN ch END;
    END; (*TRY*)
    RETURN ch;
  END LookAhead;

  PROCEDURE GetChar(rd: Reader := NIL): CHAR RAISES {Error} =
  VAR ch: CHAR := null;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      ch:= Rd.GetChar(rd);
      RETURN ch;
    EXCEPT ELSE
      RETURN ch;
    END; (*TRY*)
  END GetChar;

  PROCEDURE PutChar(ch: CHAR; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutChar(wr, ch);
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutChar;  

  PROCEDURE GetText(rd: Reader := NIL): TEXT RAISES {Error} =
  VAR t: TEXT;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      Lex.Skip(rd);
      t:= Lex.Scan(rd, NonBlanks); EVAL GetChar(rd); RETURN t
    EXCEPT Lex.Error => RAISE Error
    END;
  END GetText;

  PROCEDURE PutText(t: TEXT; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutText(wr, t);
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutText;

  PROCEDURE GetInt(rd: Reader := NIL): INTEGER RAISES {Error} =
  CONST T = T1 & "an integer" & T2;
  VAR i: INTEGER;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    LOOP
      TRY
        Lex.Skip(rd);
        IF Rd.EOF(rd) AND (rd # Stdio.stdin) THEN RAISE Error
        ELSE
          i:= Lex.Int(rd); EVAL GetChar(rd); RETURN i
        END; (*IF Rd.EOF(rd)*)
      EXCEPT Lex.Error =>
        IF rd = Stdio.stdin THEN PutText(T) ELSE RAISE Error END;
      END; (*TRY*)
    END; (*LOOP*)
  END GetInt;
  
  PROCEDURE PutInt(i: INTEGER; length:= 3; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutText(wr, Fmt.Pad(Fmt.Int(i), length));
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutInt;

  PROCEDURE GetReal(rd: Reader := NIL): REAL RAISES {Error} =
  CONST T = T1 & "a real" & T2;
  VAR r: REAL;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    LOOP
      TRY
        Lex.Skip(rd);
        IF Rd.EOF(rd) AND (rd # Stdio.stdin) THEN RAISE Error
        ELSE
          r:= Lex.Real(rd); EVAL GetChar (rd); RETURN r
        END; (*IF Rd.EOF(rd)*) 
      EXCEPT Lex.Error =>
        IF rd = Stdio.stdin THEN PutText(T) ELSE RAISE Error END;
      END; (*TRY*)
    END; (*LOOP*)
  END GetReal;

  PROCEDURE PutReal(r: REAL; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutText(wr, Fmt.Real(r));
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutReal;

  PROCEDURE GetLongReal(rd: Reader := NIL): LONGREAL RAISES {Error} =
  CONST T = T1 & "a longreal" & T2;
  VAR r: LONGREAL;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    LOOP
      TRY
        Lex.Skip(rd);
        IF Rd.EOF(rd) AND (rd # Stdio.stdin) THEN RAISE Error
        ELSE
          r:= Lex.LongReal(rd); EVAL GetChar (rd); RETURN r
        END; (*IF Rd.EOF(rd)*) 
      EXCEPT Lex.Error =>
        IF rd = Stdio.stdin THEN PutText(T) ELSE RAISE Error END;
      END; (*TRY*)
    END; (*LOOP*)
  END GetLongReal;

  PROCEDURE PutLongReal(r: LONGREAL; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutText(wr, Fmt.LongReal(r));
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutLongReal;
      
  PROCEDURE GetBool(rd: Reader := NIL): BOOLEAN RAISES {Error} =
  CONST T = T1 & "a Boolean" & T2;
  VAR b: BOOLEAN;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    LOOP
      TRY
        Lex.Skip(rd);
        IF Rd.EOF(rd) AND (rd # Stdio.stdin) THEN RAISE Error
        ELSE
          b:= Lex.Bool(rd); EVAL GetChar (rd); RETURN b
        END; (*IF Rd.EOF(rd)*) 
      EXCEPT Lex.Error =>
        IF rd = Stdio.stdin THEN PutText(T) ELSE RAISE Error END;
      END; (*TRY*)
    END; (*LOOP*)
  END GetBool;
  
  PROCEDURE PutBool(b: BOOLEAN; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    IF b THEN Wr.PutText(wr, "TRUE") ELSE Wr.PutText(wr, "FALSE") END;
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutBool;

  PROCEDURE TermChar(rd: Reader := NIL): CHAR RAISES {Error} =
  VAR ch: CHAR := null;
  BEGIN    
    IF rd = NIL THEN rd:= Stdio.stdin END; 
    IF (rd # Stdio.stdin) AND End(rd) THEN RETURN '\n' END; 
      (*This is a terrible work-around, sorry*)                                                
    TRY
      Rd.UnGetChar(rd);
      ch:= GetChar(rd);
    EXCEPT ELSE
      IF rd # Stdio.stdin THEN RAISE Error ELSE RETURN ch END;
    END; (*TRY*)
    RETURN ch;
  END TermChar;

  PROCEDURE Nl(wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutText(wr, "\n");
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END Nl;
  
  PROCEDURE PutUnsigned(i: INTEGER; length := 6; base: [2..16] := 10;
                        wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.PutText(wr, Fmt.Pad(Fmt.Unsigned(i, base), length));
    IF wr = Stdio.stdout THEN Wr.Flush(wr) END;
  END PutUnsigned;

  PROCEDURE ErrText(text: TEXT) =
  BEGIN
    WITH so = Stdio.stdout DO
      Wr.PutText(so, "Error at TIO." & text & "\n");
      Wr.Flush(so);
    END; (*WITH so*)
  END ErrText;

  PROCEDURE Flush(wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    Wr.Flush(wr);
  END Flush;

  PROCEDURE End(rd: Reader := NIL): BOOLEAN =
  VAR eof: BOOLEAN := TRUE;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      eof:= Rd.EOF(rd)
    EXCEPT ELSE
      ErrText("End");
    END; (*TRY*)
    RETURN eof;
  END End; 

  PROCEDURE Available(rd: Reader := NIL): BOOLEAN =
  VAR av: BOOLEAN := FALSE;
  BEGIN
    IF rd=Stdio.stdin OR rd=NIL THEN
      IF c_funcs.kbhit()#0 THEN
        RETURN(TRUE);
      ELSE
        RETURN(FALSE);
      END; (* IF *)
    ELSE (* Reads from file. *)
      RETURN(TRUE);
    END; (* IF *)
    
    (* IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      av:= Rd.CharsReady(rd) > 0
    EXCEPT ELSE
      ErrText("Available");
    END; 
    RETURN av; *)
  END Available;

  PROCEDURE Length(rd: Reader := NIL): CARDINAL =
  VAR l: CARDINAL := 0;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      l:= Rd.Length(rd)
    EXCEPT ELSE
    ErrText("Length");
    END; (*TRY*)
    RETURN l;
  END Length;

  PROCEDURE Seek(rd: Reader := NIL; position: CARDINAL := 0) =
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    IF Rd.Seekable(rd) THEN 
      TRY
        Rd.Seek(rd, position)
      EXCEPT ELSE
      ErrText("Seek");
      END; (*TRY*)
    ELSE
      ErrText("Seek, file file is not seekable")
    END; (*IF Rd.Seekable(rd)*)
  END Seek;

BEGIN
END SIO.
