INTERFACE FieldList;
(* Breaks text lines into fields that can be processed individually as
   text or numbers. *)

(* 	Copyright Samuel P. Harbison 1991. All rights reserved. 
	
    Permission is hereby granted to use, copy, modify, prepare
    integrated and derivative works of and distribute this software
    for non-commercial purposes, provided that you retain the
    copyright notice and disclaimer in the software.

	This software is provided "as-is" and Samuel P. Harbison disclaims
    all warranties with regard to this softtware, including all
    implied warranties of merchantability and fitness of purpose.

	Written by Samuel P. Harbison, Pine Creek Software; Suite 300; 
    305 South Craig Street; Pittsburgh, PA 15213; USA. 
    Phone&FAX: +1 (412) 681 9811. E-mail: harbison@bert.pinecreek.com.

	Part of the Pine Creek Modula-3 Library.

	Version 1.0.

	Printed documentation can be obtained by contacting the author.
*)
IMPORT Rd, Wr;

EXCEPTION Error;

CONST
	DefaultWS = SET OF CHAR{' ', '\t', '\n', '\f', ','};
	DefaultKeepWS = FALSE;
	DefaultQuotes = TRUE;

TYPE 
	Field = CARDINAL;  
		(* Fields are numbered 0, 1, ... *)
	T <: Public;
	Public = OBJECT
	METHODS
		init(): T;
			(* Currently has no effect, but it's a convention... *)

		getLine(rd: Rd.T := NIL) RAISES {Rd.EndOfFile, Rd.Failure};
			(* Reads line from specified reader; default is Stdio.stdin *)
		getText(t: TEXT);
			(* Loads self with specified text *)

		number(): CARDINAL;
			(* Returns number of fields. *)
		pos(n: Field): CARDINAL;
			(* Return number of characters preceeding field n
			in the original line. *)
		length(n: Field): CARDINAL;
			(* Returns number of characters in field n. *)

		line(): TEXT;
			(* Returns the entire line *)
		lineLength(): CARDINAL;
			(* Returns the length of the entire line *)

		(* Following functions return true if the field is of 
		the indicated type. *)
		isWhitespace(n: Field): BOOLEAN;
			(* Can only occur if "keep_whitespace" attribute is TRUE *)
		isInteger(n: Field): BOOLEAN;
			(* Field syntax: [+|-] digit+ *)
		isReal(n: Field): BOOLEAN;
			(* Syntax: [+|-] digit* [.] digit* [(E|e|D|d|X|x) [+|-] digit+]
				must have either decimal point or exponent. *)
		isNumber(n: Field): BOOLEAN;
			(* isInteger OR isReal *)

		integer(n: Field): INTEGER;
			(* Return n'th field as an integer; or 0 *)
		real(n: Field): LONGREAL;
			(* Return n'th field as a floating-point number; or 0.0 *)
		text(n: Field): TEXT;
			(* Return n'th field, or "". *)
		char(n: Field; i: CARDINAL := 0): CHAR;
			(* Returns character i (0-based) in field n; or '\000' *)

		equal(n: Field; text: TEXT; caseSensitive := TRUE): BOOLEAN;
			(* True if contents of field n equals text; if caseSensitive
			is TRUE, it is the same as Text.Equal(T.field(n), t) *)

		put(n: Field; wr: Wr.T := NIL);
			(* Same as PutText(wr, self.text(n)), with wr defaulting
			to Stdio.stdout. *)

		setAttr(attribute: TEXT; value: REFANY) RAISES {Error};
			(* Permissible attributes and the expected argument are:
				"whitespace" (REF SET OF CHAR),
				"keep_whitespace" (REF BOOLEAN),
				"quoted_fields" (REF BOOLEAN) 
			*)
		getAttr(attribute: TEXT): REFANY RAISES {Error};
			(* See 'set'; returns the named attribute *)

		dump(wr: Wr.T := NIL);
			(* Dump state of field list; for debugging *)
	END;	

	(* Functions to help with setAttr/getAttr; each call returns a 
	new, initialized dynamic variable to do with as the client wishes. *)
	PROCEDURE RefInteger(val: INTEGER): REF INTEGER;
	PROCEDURE RefBoolean(val: BOOLEAN): REF BOOLEAN;
	PROCEDURE RefSetOfChar(READONLY val: SET OF CHAR): REF SET OF CHAR;

END FieldList.

(* Local Variables:  *)
(* tab-width: 4 *)
(* End: *)
