/**********************************************************
 * FILE LINKLIST.C version 1.04  - 06/01/95 -
 *
 * DESCRIPTION:
 *  contains the link list's functions.
 *
 * NOTES:
 *
 * Copyright 1993 by EZ Software 
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided 
 * that the above copyright notice appear in all copies and that both that 
 * copyright notice and this permission notice appear in supporting 
 * documentation, and that the name of EZ Software 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific written prior permission. 
 * EZ Software makes no representations about the 
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * EZ Software DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, 
 * IN NO EVENT SHALL EZ Software BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 *
 *
 *  Original Author: Laurent Chavey, EZ Software 
 *           (chavey@cis.udel.edu)
 *            june, 1993
 *
 * UPDATES:
 *          05/15/95  - added parameter validation.
 *                    - added llload and llsave function.
 *                    - added support for tracking the number of compare function.
 *                      This allows validation of indexes in lladdfind 
 *                      and llfind.
 *                    - change lladdfind to return an int.
 *          04/13/95  - fixed problem with llgetwnext function.
 *                    - added support for querry types in find function.
 *          10/13/94  - added function for circular link list.
 *                      llgetwnext, llgetwprev.
 *          05/10/94  - fixed bug in add function.
 *          01/20/94  - changes search and Search to compare Compare. 
 *                    - fixed problem in delete.
 *                    - added support for multiple compare function.
 *                    - added dump function..
 *                    - added support for adding at current rather
 *                      than tail.
 *                    - added parameter to add function.
 *                    - added return standard error code. 
 *                    - removed item size from add, added it to main linklist
 *                      structure and llnew function.
 *
 **********************************************************/

#define LINKLIST
#include "linklist.h"

/********************* PRIMITIVES ***********************/

/*********************************************************
 * FUNCTION: 
 *  llnew
 *
 * PROTOTYPE:
 *  linklist *llnew(int, int);
 *
 * PARAMETERS:
 *  int  nCompare  number of compare function used for linklist element.
 *  int  nDataSize size of the data stored in linkelement.
 *
 * RETURN:
 *  linklist *  pointer to the new linklist.
 *
 * DESCRIPTION:
 *  crates a link list.
 *
 * NOTES:
 *  compare function should return LLFOUND when found, 
 *  LLNOTFOUND if not.
 *
 * UPDATES: 
 *          
 *          05/15/95  - stores nCompare in link list header.
 *          01/20/94  - changes search and Search.
 *                    - added nDataSize.
 *
 *********************************************************/
linklist *llnew(nCompare, nDataSize)
int nCompare;
int nDataSize;
{
   linklist *lpList;
   
   lpList = (linklist *)malloc(sizeof(linklist));
   if(lpList != NULL)
   {
      memset(lpList,0x00,sizeof(linklist));
      lpList->nDataSize   = nDataSize;
      lpList->lpHead      = NULL;
      lpList->lpTail      = NULL;
      lpList->lpCurrent   = NULL;
      lpList->nCompare    = nCompare;
      lpList->lpfnCompare = calloc(nCompare,sizeof(int (*)()));
   }
   return(lpList);
}

/*********************************************************
 * FUNCTION: 
 *  lladdfind
 *
 * PROTOTYPE:
 *  linklist *lladdfind(int (* )(), int);
 *
 * PARAMETERS:
 *  int (* )(void *, void *)  compare function used for linklist element.
 *  int                       index of compare function.
 *
 * RETURN:
 *  int  LLNOERROR.
 *       LLINVPARA if compare is out of range.
 *
 * DESCRIPTION:
 *  creates a link list.
 *
 * NOTES:
 *  compare function should return LLFOUND when found, 
 *  LLNOTFOUND if not.
 *
 * UPDATES: 
 *  
 *          05/15/95  - added check for compare function index.
 *                    - return an int for error check.
 *          01/20/94  - added support for multiple compare 
 *                      functions.
 *                    - changed search to compare.
 *
 *********************************************************/
int lladdfind(lpListName,lpfnCompareFunc, nCompare)
linklist *lpListName;
int       (* lpfnCompareFunc)(void *, void *, int);
int       nCompare;
{
   int nError = LLNOERROR;
   
   if(lpListName && (nCompare < lpListName->nCompare) && (nCompare >= 0))
    lpListName->lpfnCompare[nCompare] = lpfnCompareFunc;
   else
    nError = LLINVPARA;
   
   return(nError);
}


/*********************************************************
 * FUNCTION:
 *  llfree.
 *
 * PROTOTYPE:
 *  int llfree(linklist *);
 *
 * PARAMETERS:
 *  linklist *lpListName   link list header.
 *
 * RETURN:
 *  LLNOERROR.
 *  LLINVPARA.
 *
 * DESCRIPTION:
 *  free a link list, removing all its elements.
 *
 * NOTES:
 *
 * UPDATES: 
 *          05/15/95 - added parameter validation.
 *          01/20/94 - added LLNOERROR on return.
 *
 *********************************************************/
int llfree(lpListName)
linklist *lpListName;
{
   int nError = LLNOERROR;
   
   if(lpListName)
   {
      while(lpListName->lpHead != NULL)
      {
         lpListName->lpCurrent = (linkelement *)lpListName->lpHead->lpNext;
         free(lpListName->lpHead->lpData);
         free(lpListName->lpHead);
         lpListName->lpHead = lpListName->lpCurrent;
      }
      
      free(lpListName->lpfnCompare);
      free(lpListName);
   }
   else
    nError = LLINVPARA;
   
   return(LLNOERROR);
}

/*********************************************************
 * FUNCTION:
 *  lladd.
 *
 * PROTOTYPE:
 *  int lladd(linklist *, void *, int);
 *
 * PARAMETERS:
 *  linklist *lpListName  Link list definition.
 *  void     *lpValue     Value to be inserted.
 *  int      nLoc         before or after current 
 *                          LLPREV before.
 *                          LLNEXT after.
 *
 * RETURN:
 *  int   LLNOERROR.
 *        LLGENERROR error inserting, out of heap.
 *        LLINVPARA  invalid nLoc or empty lpListName.
 *
 * DESCRIPTION:
 *  insert a new element in link list at current.
 *  can insert before or after current.
 *
 * NOTES:
 *
 * UPDATES:
 *           05/15/95 - added validation of parameters.
 *           05/10/94 - fixed bug in insert.
 *           01/20/94 - added support for adding at current
 *                      rather than tail. can specify 
 *                      direction.
 *                    - added support for using defined error.
 *                    - removed nSize parameter.
 *
 *********************************************************/
int lladd(lpListName, lpValue, nLoc)
linklist *lpListName;
void     *lpValue;
int      nLoc;
{
   linkelement *lpElement;
   void        *lpData;
   int         nError = LLNOERROR; 

   if(lpListName)
   {
      lpElement = (linkelement *)malloc(sizeof(linkelement));
      lpData = malloc(lpListName->nDataSize);
      if((lpElement != NULL) && (lpData != NULL))
      {
	 memcpy(lpData,lpValue,lpListName->nDataSize);
	 lpElement->lpData = lpData;
	 
	 if(lpListName->lpHead == NULL)
	 {
	    lpElement->lpNext = NULL;
	    lpElement->lpPrevious = NULL;
	    lpListName->lpHead = lpElement;
	    lpListName->lpTail = lpElement;
	    lpListName->lpCurrent = lpElement;
	 }
	 else
	 {
	    switch(nLoc)
	    {
	     case LLNEXT:
	       lpElement->lpNext = lpListName->lpCurrent->lpNext;
	       lpElement->lpPrevious = lpListName->lpCurrent;
	       lpListName->lpCurrent->lpNext = lpElement;
	       if(lpElement->lpNext == NULL)
	       	lpListName->lpTail = lpElement;
	       else
	       	lpElement->lpNext->lpPrevious = lpElement;
	       lpListName->lpCurrent = lpElement;
	       break;
	       
	     case LLPREV:
	       lpElement->lpNext = lpListName->lpCurrent;
	       lpElement->lpPrevious = lpListName->lpCurrent->lpPrevious;
	       lpListName->lpCurrent->lpPrevious = lpElement;
	       if(lpElement->lpPrevious == NULL)
	       	lpListName->lpHead = lpElement;
	       else
	       	lpElement->lpPrevious->lpNext = lpElement;
	       lpListName->lpCurrent = lpElement;
	       break;
	       
	     default:
	       nError = LLINVPARA;
	       break;
	    }
	 }
      }
      else
       nError = LLGENERROR;
   }
   else
    nError = LLINVPARA;
   
   return(nError);
}


/*********************************************************
 * FUNCTION:
 *  lldelete.
 *
 * PROTOTYPE:
 *  int lldelete(linklist *, linkelement *);
 *
 * PARAMETERS:
 *  LinkLIst    *lpListName  Link list definition.
 *  linkelement *lpElement   Element to be removed.
 *
 * RETURN:
 *  int  LLNOERROR.
 *       LLGENERROR error deleting.
 *
 * DESCRIPTION:
 *  remove an element from the link list.
 *
 * NOTES:
 *  The current pointer is updated in this fation
 *  if deleting the head, points to the new head.
 *  if deleting the tail, points to the new tail.
 *  if deleting a node anywhere else in link,
 *  points to the one link preceding the deleted one.
 *
 * UPDATES:  
 *           05/15/95 - check for error in llnumber call.
 *           01/20/94 - fixed problem with delete.
 *                    - added standard error.
 *
 *********************************************************/
int lldelete(lpListName, lpElement)
linklist    *lpListName;
linkelement *lpElement;
{
   int nNumber = 0;

   if((nNumber = llnumber(lpListName)) <= 0)
    return(LLGENERROR);

   if(nNumber == 1)
   {
      /* delete last element */
      lpListName->lpHead = NULL;
      lpListName->lpTail = NULL;
      lpListName->lpCurrent = NULL;
   }  
   else if(lpElement == lpListName->lpTail)
   {
      /* move tail back by one */
      lpListName->lpTail = (linkelement *)lpElement->lpPrevious;
      lpListName->lpCurrent = lpListName->lpTail;
      lpListName->lpCurrent->lpNext = NULL;
   }
   else if(lpElement == lpListName->lpHead)
   {
      /* move head up by one */
      lpListName->lpHead = (linkelement *)lpElement->lpNext;
      lpListName->lpCurrent = lpListName->lpHead;
      lpListName->lpCurrent->lpPrevious = NULL;
   }
   else
   {
      /* delete element anywhere else in list */
      lpListName->lpCurrent = (linkelement *)lpElement->lpNext;
      lpListName->lpCurrent->lpPrevious = lpElement->lpPrevious;
      lpListName->lpCurrent = (linkelement *)lpElement->lpPrevious;
      lpListName->lpCurrent->lpNext = lpElement->lpNext;
   }
   
   if(lpElement->lpData)
    free(lpElement->lpData);
   
   free(lpElement);
   return(LLNOERROR);
}

/*********************************************************
 * FUNCTION:
 *  llnumber.
 *
 * PROTOTYPE:
 *  int llnumber(linklist *);
 *
 * PARAMETERS:
 *  linklist    *lpListName   Link list definition.
 *
 * RETURN:
 *  int number of elements in link list.
 *      or LLINVPARA.
 *
 * DESCRIPTION:
 *  return the number of elements in the link list.
 *
 * NOTES:
 *
 * UPDATES:
 *          05/15/95 - added parameter validation.
 *
 *********************************************************/
int llnumber(lpListName)
linklist *lpListName;
{
   int          nNum = 0;
   linkelement *lpList; 
   
   if(lpListName)
   {
      lpList = lpListName->lpHead;
      
      while(lpList != NULL)
      {
         nNum ++;
         lpList = (linkelement *)lpList->lpNext;
      }
   }
   else
    nNum = LLINVPARA;
   
   return(nNum);
}


/*********************************************************
 * FUNCTION:
 *  llget.
 *
 * PROTOTYPE:
 *  void *llget(linkelement *);
 *
 * PARAMETERS:
 *  linkelement *lpElement     Element to retrieve the value from.
 *
 * RETURN:
 *  void * pointer to data.
 *         NULL if error.
 *
 * DESCRIPTION:
 *  extract the data from a link element.
 *
 * NOTES:
 *
 * UPDATES: 10/13/94 - fixed on entry point, one exit point
 *                     by adding lpData.
 *
 *********************************************************/
void *llget(lpElement)
linkelement *lpElement;
{
   void *lpData = NULL;
   
   if(lpElement != NULL)
    lpData = lpElement->lpData;
   
   return(lpData);
}

/********************* END OF PRIMITIVES ****************/  

/*********************************************************
 * FUNCTION:
 *  llgetnext.
 *
 * PROTOTYPE:
 *  void *llgetnext(linklist *);
 *
 * PARAMETERS:
 *  linklist    *lpListName Link list definition.
 *
 * RETURN:
 *  void * pointer to value.
 *         NULL if error.
 *
 * DESCRIPTION:
 *  extract the value fallowing the current link in list.
 *  the current link is moved forward by one.
 *
 * NOTES:
 *
 * UPDATES: 10/13/94 - fixed one entry point, one exit point
 *                     by adding lpData.
 *
 *********************************************************/
void *llgetnext(lpListName)
linklist *lpListName;
{
   void *lpData = NULL;
   
   if(lpListName && (lpListName->lpCurrent != lpListName->lpTail))
   {
      lpListName->lpCurrent = (linkelement *)lpListName->lpCurrent->lpNext;
      lpData = llget(lpListName->lpCurrent);
   }
   
   return(lpData);
}


/*********************************************************
 * FUNCTION:
 *  llgetwnext.
 *
 * PROTOTYPE:
 *  void *llgetwnext(linklist *);
 *
 * PARAMETERS:
 *  linklist    *lpListName Link list definition.
 *
 * RETURN:
 *  void * pointer to value.
 *         NULL if error.
 *
 * DESCRIPTION:
 *  extract the value preceding the current link in list.
 *  The current link is moved backward by one. If it is
 *  at the tail, it wraps back to the head.
 *  This is to allow for circular link list.
 *
 * NOTES:
 *
 * UPDATES: 
 *          05/15/95 - added validation of parameters.
 *          04/13/95 - fixed bug with wrapp arround after
 *                     reaching tail.
 *          10/13/94 - added function. Suggestion from
 *                     Darius Miller.
 *
 *********************************************************/
void *llgetwnext(lpListName)
linklist *lpListName;
{
   void *lpData = NULL;
   
   if(lpListName)
   {
      if(lpListName->lpCurrent != lpListName->lpTail)
       lpListName->lpCurrent = (linkelement *)lpListName->lpCurrent->lpNext;
      else
       llsetbeg(lpListName);
      
      lpData = llget(lpListName->lpCurrent);
   }
   
   return(lpData);
}


/*********************************************************
 * FUNCTION:
 *  llgetprev.
 *
 * PROTOTYPE:
 *  void *llgetprev(linklist *);
 *
 * PARAMETERS:
 *  linklist    *lpListName Link list definition.
 *
 * RETURN:
 *  void * pointer to value.
 *         NULL if error.
 *
 * DESCRIPTION:
 *  extract the value preceding the current link in list.
 *  The current link is moved backward by one.
 *
 * NOTES:
 *
 * UPDATES: 05/15/95 - added parameter check.
 *          10/13/94 - added one entry point one exit point. 
 *                     added lpData .
 *
 *********************************************************/
void *llgetprev(lpListName)
linklist *lpListName;
{
   void *lpData = NULL;
   
   if ((lpListName) && (lpListName->lpCurrent != lpListName->lpHead))
   {
      lpListName->lpCurrent = (linkelement *)lpListName->lpCurrent->lpPrevious;
      lpData = llget(lpListName->lpCurrent);
   }
   
   return(lpData);
}


/*********************************************************
 * FUNCTION:
 *  llgetwprev.
 *
 * PROTOTYPE:
 *  void *llgetwprev(linklist *);
 *
 * PARAMETERS:
 *  linklist    *lpListName Link list definition.
 *
 * RETURN:
 *  void * pointer to value.
 *         NULL if error.
 *
 * DESCRIPTION:
 *  extract the value preceding the current link in list.
 *  The current link is moved backward by one. If the current
 *  link is at the head, then it is wrapped to the tail.
 *  This is to allow for circular link list.
 *
 * NOTES:
 *
 * UPDATES: 05/15/95 - added parameter check.
 *          10/13/94 - added function. Suggestion from
 *                     Darius Miller.
 *
 *********************************************************/
void *llgetwprev(lpListName)
linklist *lpListName;
{
   void *lpData = NULL;
   
   if(lpListName)
   {
      if(lpListName->lpCurrent != lpListName->lpHead)
       lpListName->lpCurrent = (linkelement *)lpListName->lpCurrent->lpPrevious;
      else
       llsetend(lpListName);
   
      lpData = llget(lpListName->lpCurrent);
   }
   else
    return(lpData);
}


/*********************************************************
 * FUNCTION:
 *  llfind.
 *
 * PROTOTYPE:
 *  void *llfind(linklist *, void *, int);
 *
 * PARAMETERS:
 *  linklist *lpListName   list name variable, link list.
 *  void     *lpValue      value to compare for.
 *  int       nCompare     compare function to use.
 *  int       nType        type of comparaison.
 *
 * RETURN:
 *  void *   pointer to data found.
 *           NULL if none found or error.
 *
 * DESCRIPTION:
 *  compare a link list for a specific element containg data.
 *
 * NOTES:
 *  uses the compare function associated with link list.
 *  Comapre function should return LLFOUND when found, 
 *  LLNOTFOUND if not.
 *  The compare starts at the current linklist element
 *  on forward through the list until tail is reached.
 *  To compare from begining use llsetbeg().
 *  These compare function only does a compare = or != or < or >
 *
 * UPDATES: 
 *          05/15/95  - added support for compare index validation.
 *          04/13/95  - added nType for LLGREATER, LLLESS, LLEQUAL.
 *          10/13/94  - added LLFOUND & LLNOTFOUND. 
 *          01/20/94  - added support for multiple compare 
 *                    - changed search to compare.
 *                      functions.
 *********************************************************/
void *llfind(lpListName,lpValue, nCompare, nType)
linklist *lpListName;
void     *lpValue;
int       nCompare;
int       nType;
{
   void *lpData = NULL;
   int   nFound = LLNOTFOUND;
   
   if(lpListName && llnumber(lpListName) && (lpListName->lpfnCompare))
   {
      if((nCompare < lpListName->nCompare) && (nCompare >= 0))
      {
	 while((nFound = (*lpListName->lpfnCompare[nCompare])(lpListName->lpCurrent->lpData, lpValue, nType)) == LLNOTFOUND)
	 {
	    if (lpListName->lpCurrent == lpListName->lpTail)
	     break;
	 
	    lpListName->lpCurrent = (linkelement *)lpListName->lpCurrent->lpNext;
	    
	    
	    if (!lpListName->lpCurrent)
	     break;;
	 }
      }
   }
   if(nFound == LLFOUND)
    lpData = lpListName->lpCurrent->lpData;
   
   return(lpData);
}

/*********************************************************
* FUNCTION:
*  llsave.
*
* PROTOTYPE:
*  int llsave(linklist *, char *);
*
* PARAMETERS:
*  linklist *lpListName   link list header.
*  char     *szFile       file name to save to.
*
* RETURN:
*  LLNOERROR.
*  LLNOFILE.
*
* DESCRIPTION:
*  save a link list to a file.
*
* NOTES:
*
* UPDATES: 05/15/95 - created.
*
*********************************************************/
int llsave (lpListName, lpfnSaveFunc, fFile)
linklist *lpListName;
int      (* lpfnSaveFunc)(void *, FILE *);
FILE     *fFile;
{
   int      nError = LLNOERROR;
   void    *lpData;
   int      nLink;
   linklist LinkHeader;
   
   /* get the number of links to save */
   nLink = llnumber(lpListName);
   /* write the header to file */
   if(fwrite((char *)lpListName,sizeof(linklist),1,fFile) == 1) 
   {
      /* save the number of links */
      if(fwrite((void *)&nLink,sizeof(int),1,fFile) != 1)
       nError = LLFILEERROR;

      /* use circular list, start at tail go next */
      else
       llsetend(lpListName);
      
      /* for every links */
      while(nLink && (nError == LLNOERROR))
      {
	 /* get link from list */
	 if(!(lpData = llgetwnext(lpListName)))
	  break;
	 
	 /* write link to file */
	 else if(fwrite((void *)lpData,lpListName->nDataSize,1,fFile) != 1)
	  break;
	  
	 /* call user save function if one is present*/
	 else if(lpfnSaveFunc && ((*lpfnSaveFunc)(lpData,fFile) != LLNOERROR))
	  break;
	 
	 /* saved link */
	 else
	  nLink --;
      }
     
      /* did not save all the links */
      if(nLink)
       nError = LLFILEERROR;
   }
   else
    nError = LLFILEERROR;

   return(nError);
}

/*********************************************************
 * FUNCTION:
 *  llload.
 *
 * PROTOTYPE:
 *  int llload(linklist *, char *);
 *
 * PARAMETERS:
 *  linklist *lpListName     link list pointer, used to return
 *                           the pointer to the loaded link list
 *                           this is the equivalent of calling the 
 *                           llnew function.
 * int (*) (void *, FILE *) load function used to load each link from list.
 *                          this function is provided by the user.
 *  char     *szFile         File name to load list from.
 *
 * RETURN:
 *  LLNOERROR.
 *  LLGENERROR.
 *  LLFILEERROR.
 *
 * DESCRIPTION:
 *  loads a link list from a file.
 *
 * NOTES:
 *
 * UPDATES: 05/15/95 - created.
 *
 *********************************************************/
int llload(lpListName, lpfnLoadFunc, fFile)
linklist **lpListName;
int      (* lpfnLoadFunc)(void *, FILE *);
FILE     *fFile;
{
   int      nError = LLNOERROR;
   void    *lpData;
   linklist LinkHeader;
   int      nLink;

   /* get the linklist header */
   if(fread(&LinkHeader,sizeof(linklist),1,fFile) == 1)
   {
      /* allocate temporary link buffer */
      lpData = malloc(LinkHeader.nDataSize);
      
      /* initialise the linklist header */
      memset(*lpListName,0x00,sizeof(linklist));
      *lpListName = llnew(LinkHeader.nCompare, LinkHeader.nDataSize);

      /* we have a header and room for a link */
      if(*lpListName && lpData)
      {
	 /* get the number of links */
	 if(fread((void *)&nLink,sizeof(int),1,fFile) != 1)
	  nError = LLFILEERROR;
	 
	 /* load evry links */
	 while(nLink && (nError == LLNOERROR) && (fread(lpData,LinkHeader.nDataSize,1,fFile) == 1))
	 {
	    /* add the value to the link list */
	    if(lladd(*lpListName,lpData,LLNEXT) != LLNOERROR)
	     break;
	    
	    /* let the user update the data list and read from the file */
	    else if((lpfnLoadFunc) && ((*lpfnLoadFunc)(lpData, fFile) != LLNOERROR))
	     break;
	    
	    /* saved the link */
	    else
	     nLink --;
	 }

	 /* did not read all the links links */
	 if(nLink)
	  nError = LLFILEERROR;
      }
      else
       nError = LLGENERROR;
      
      /* free the temporary link buffer */
      if(lpData)
       free(lpData);
   }
   else
    nError = LLFILEERROR;
   
   return(nError);
}


/*********************** DEBUG OPTION *******************/

/*********************************************************
* FUNCTION:
*  lldump.
*
* PROTOTYPE:
*  int lldump(linklist *);
*
* PARAMETERS:
*  linklist *lpListName   link list header.
*
* RETURN:
*  LLNOERROR.
*
* DESCRIPTION:
*  dump a link list.
*
* NOTES:
*
* UPDATES: 01/20/94 - created for debug purpose.
*                   - added LLNOERROR.
*
*********************************************************/
int lldump(lpListName)
linklist *lpListName;
{
   linkelement *lpLink;

   if (lpListName)
   {
      fprintf(stdout,"Head %x Tail %x Current %x\n",lpListName->lpHead,
                                                    lpListName->lpTail,
                                                    lpListName->lpCurrent);
      lpLink = lpListName->lpHead;
      while(lpLink != NULL)
      {
         fprintf(stdout," Link %x Next %x Prev %x Data %x\n",lpLink,
                                                             lpLink->lpNext,
                                                             lpLink->lpPrevious,
                                                             lpLink->lpData);
         lpLink = lpLink->lpNext;
      }
   }

   return (LLNOERROR);
}

