/***********************************************************************
this is a short demo of the use of the link list for a simulated 
scheduler

there is a define for system V ipc. For linux, do not define.

notes:
-----
	- priotity is given to a process finishing a cpu run over a process
	  finishing an I/o run in the same quantum.

	- I/o and cpu are synchronized.
	- one I/o takes two quantums.
			1 to be put in the I/o queue.
			1 to be run by I/o process.
	- one Cpu takes one quantum.
	- process are generated in bulk. Because of the randomness
	  of the generation, but also because the real scheduler does not
	  stop at the same point, processes arrive at different time, for 
	  different runs. (need  to be accounted to prove validity).
	- I/O processes run for one I/o and are put back in the ready queue
	  there after, even if the next process requirement is an I/o.
	- for shortest job first, if 2 jobs are found to be the same length 
	  requirement, the oldest one is put first.

************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>

#include "../linklist/linklist.h"

/*#define SYSTEM_V*/

/* message used for sync, update,new */
#define MSG_TICK	1
#define MSG_END   2
#define MSG_PRINT 3
#define MSG_NONE	4	
#define MSG_PUT	5
#define MSG_GET	6
#define MSG_NEW	7

#define MAX_PROCESS  10  /* maximum number of processes */
#define MAX_RUN		10  /* maximum number of states/ process */

/* possible processes states */
#define P_READY	0
#define P_RUN		1
#define P_BLOCK	2
#define P_DONE		3

/* used by print_status */
#define STATE	0
#define PRINT	1
#define INIT	2


/*  system V only */
#ifdef SYSTEM_V
	#define SCHED_Q	(key_t) 0x1000
	#define IO_Q		(key_t) 0x2000
/* berkley system*/
#else
	#define SCHED_Q	100L
	#define IO_Q		200L
#endif

#define PERMS     0666

#define MODE_RR	1
#define MODE_SJF  2

/* used to record process state */
typedef struct STATUS
{
	int nReq;
	int nState;
	int nServ;
	int nIo;
} STATUS;

/* process control block */
typedef struct PCB
{
	int nPid;
	int nReq;
	int nServ;
	int nIo;
	int nSreq;
} PCB;

/* message send accross processes */
typedef struct MSG
{
	long type;
	PCB  data;
} MSG;

/* prototype */
int sched_process(linklist *, int, int,int);
int find_job(void *, void *);

/*************************
	start the ball rolling 
*************************/
int main()
{
	linklist *lpReadyQ;
	int 		 SchedQ,
	    		 IoQ,
		 		 nStatus;
	MSG 		 Msg;
	int       nMode;

	/* create ready queue */
	lpReadyQ = llnew(1,sizeof(PCB));
	lladdfind(lpReadyQ,find_job,0);

	/* create system queues */
	if (!(SchedQ = msgget(SCHED_Q,IPC_CREAT | PERMS))  || 
	    !(IoQ    = msgget(IO_Q,   IPC_CREAT | PERMS)))
	{
		printf("I/O process error getting queue ids \\n");
		return(-1);
	}

	/* initialize process status to ready */
	print_status(INIT,NULL,P_READY,0);

	/* create user processes */
	if (!fork())
	{
		gen_process();
		exit(0);
	}

	/* create i/o process */
	if (!fork())
	{
		io_process();
		exit(0);
	}

	nMode = MODE_SJF;

	/* start scheduler */
	sched_process(lpReadyQ,SchedQ,IoQ,nMode);
	
	/* wait on the terminated process generator */
	wait(&nStatus);

	/* free ready process queue */
	llfree(lpReadyQ);

	/* terminate i/o process */
	Msg.type = MSG_END;
	msgsnd(IoQ,&Msg,sizeof(PCB),0);
	wait(&nStatus); 

	/* remove system queues */
	msgctl(SchedQ,IPC_RMID,NULL);	
	msgctl(IoQ,IPC_RMID,NULL);	

	exit(0);
}

/*******************************
** scheduler process
**
********************************/
int sched_process(linklist *lpReadyQ, int inQ, int IoQ,int nMode)
{
	int   nProcessInIo = 0;
	int   nProcess = 0;
	MSG   Msg;

	/* keep on going until last process exit system, and 10 have arrived */
	while(llnumber(lpReadyQ) || nProcessInIo || (nProcess < MAX_PROCESS))
	{
		if (nMode == MODE_RR)
			rr_run(lpReadyQ,IoQ,&nProcessInIo);
		else
			sjf_run(lpReadyQ,IoQ,&nProcessInIo);

		/* send a tick to i/o for sync */
		Msg.type = MSG_TICK;
		msgsnd(IoQ,&Msg,sizeof(PCB),0);

		/* wait for response from I/O for sync */
		while (msgrcv(inQ,&Msg,sizeof(PCB),0,0))
		{
			int nExit = 1;

			switch(Msg.type)
			{
				/* new process */
				case MSG_NEW:
					nProcess ++;
					print_status(STATE,&Msg.data,P_READY,0);
					if (nMode == MODE_RR)
						rr_insert(lpReadyQ, &Msg.data);
					else
						sjf_insert(lpReadyQ, &Msg.data);
					nExit = 0;
					break;

				/* put process back in ready queue */
				case MSG_PUT:
					/* if process is not done */
					if ((Msg.data.nIo + Msg.data.nServ) < MAX_RUN)
					{
						print_status(STATE,&Msg.data,P_READY,0);
						if (nMode == MODE_RR)
							rr_insert(lpReadyQ, &Msg.data);
						else
							sjf_insert(lpReadyQ, &Msg.data);
					}
					else
						print_status(STATE,&Msg.data,P_DONE,0);

					nProcessInIo --;
					break;

				case MSG_NONE:
					break;

				default:
					break;
			}

			/* received sync from I/O */
			if (nExit)
				break;
		} 
	}

	/* print last state */
	print_status(PRINT,NULL,0,1);

	return(0);
}

/********************************
** run process round robin.
********************************/
int rr_run(linklist *lpReadyQ,int IoQ, int *lpnOut)
{
	MSG Msg;
	
	if (llnumber(lpReadyQ))
	{
		/* find next process to run */
		llsetbeg(lpReadyQ);
		memcpy(&Msg.data,llgetcur(lpReadyQ),sizeof(PCB));
		lldelcur(lpReadyQ);
		print_status(STATE,&Msg.data,P_RUN,0);

		/* run process for one quantum */
		if (Msg.data.nReq & 0x0001)
		{
			print_status(PRINT,NULL,0,1);
			Msg.data.nReq >>= 1;
			Msg.data.nServ ++;
			if ((Msg.data.nIo + Msg.data.nServ) < MAX_RUN)
			{
				print_status(STATE,&Msg.data,P_READY,0);
				rr_insert(lpReadyQ, &Msg.data);
			}
			else
				print_status(STATE,&Msg.data,P_DONE,0);
		}

		/* process blocks for i/o */
		else
		{
			print_status(STATE,&Msg.data,P_BLOCK,0);
			Msg.type = MSG_NONE;
			(*lpnOut)++;
			msgsnd(IoQ,&Msg,sizeof(PCB),0);
			print_status(PRINT,NULL,0,1);
		}
	}
	else
		print_status(PRINT,NULL,0,1);
		
	return(0);
}


/********************************
** run process shortest job first
********************************/
int sjf_run(linklist *lpReadyQ,int IoQ, int *lpnOut)
{
	static MSG Msg;
	static int nProcess = 0;
	
	if (nProcess || llnumber(lpReadyQ))
	{
		if (!nProcess)
		{
			/* find next process to run */
			llsetbeg(lpReadyQ);
			memcpy(&Msg.data,llgetcur(lpReadyQ),sizeof(PCB));
			lldelcur(lpReadyQ);
			print_status(STATE,&Msg.data,P_RUN,0);
			nProcess = 1;
		}

		/* run process for one quantum */
		if (Msg.data.nReq & 0x0001)
		{
			print_status(PRINT,NULL,0,1);
			Msg.data.nReq >>= 1;
			Msg.data.nServ ++;
			if ((Msg.data.nIo + Msg.data.nServ) >= MAX_RUN)
			{
				print_status(STATE,&Msg.data,P_DONE,0);
				nProcess = 0;
			}
			else
				print_status(STATE,&Msg.data,P_RUN,0);
		}

		/* process blocks for i/o */
		else
		{
			print_status(STATE,&Msg.data,P_BLOCK,0);
			Msg.type = MSG_NONE;
			(*lpnOut)++;
			msgsnd(IoQ,&Msg,sizeof(PCB),0);
			print_status(PRINT,NULL,0,1);
			nProcess = 0;
		}
	}
	else
		print_status(PRINT,NULL,0,1);
		
	return(0);
}

/********************************
** handles the i/o processes
********************************/
int io_process()
{
	int 		 inQ,
	    		 outQ;
	MSG		 Msg;
	linklist  *lpQueue;

	static    nAdded = 0;

	/* get message queue id's */
	if (get_queueid(&outQ, &inQ))
	{
		printf("I/O process error getting queue ids \n");
		return(-1);
	}

	lpQueue = llnew(0,sizeof(PCB));

	/* process the received message */
	while (msgrcv(inQ,&Msg,sizeof(PCB),0,0))
	{
		switch(Msg.type)
		{
			/*  terminate */
			case MSG_END:
				llfree(lpQueue);
				exit(0);

			/* get tick from scheduler for sync */
			case MSG_TICK:
				/* get I/O process if some are present, and not last added */
				if (llnumber(lpQueue)  > nAdded)
				{ 
					memcpy(&(Msg.data),llgetcur(lpQueue),sizeof(PCB));
					lldelcur(lpQueue);
			   	Msg.data.nReq >>= 1;
			   	Msg.data.nIo ++;
					Msg.type = MSG_PUT;
				} 
				else
				{
					Msg.type = MSG_NONE;
				}

				msgsnd(outQ,&Msg,sizeof(PCB),0);
				nAdded = 0;
				break;

			/* new i/o process */
			case MSG_NONE:
				llsetend(lpQueue);
				lladd(lpQueue,&Msg.data,LLNEXT);
				llsetbeg(lpQueue);
				/* flags that this process has to wait one quantum */
				nAdded  = 1;
				break;
		}
	}
	return(0);
}

/************************************
** print process state
** at every desired quantum.
** also used to update the current record
** of process states.
**
************************************/
print_status(int nSelect, PCB *pcb, int nState, int nQuantum)
{
	static STATUS  Process_status[MAX_PROCESS];
	static int  CurQuantum = 0;

	int x;
	char szTemp[200];

	switch(nSelect)
	{
		/* set process state */
		case STATE:
			Process_status[pcb->nPid].nState = nState;
			Process_status[pcb->nPid].nServ = pcb->nServ;
			Process_status[pcb->nPid].nIo   = pcb->nIo;
			Process_status[pcb->nPid].nReq  = pcb->nReq;
			CurQuantum += nQuantum;
			break;

		/* print all processes state */
		case PRINT:	
			szTemp[0] = 0x00;

			for (x=0;x < MAX_PROCESS; x++)
			{
				char szSt[10];

				sprintf(szSt,"%3x,%1d,%1d|", Process_status[x].nReq,
													Process_status[x].nServ,
													Process_status[x].nIo);
				strcat(szTemp,szSt);

			}
			for (x=0;x < MAX_PROCESS;x++)
			{
				switch(Process_status[x].nState)
				{

					case P_READY:
						strcat(szTemp, " R");
						break;

					case P_RUN:
						strcat(szTemp, " X");
						break;

					case P_BLOCK:
						strcat(szTemp, " B");
						break;

					case P_DONE:
						strcat(szTemp, "  ");
						break;
				}
			}
			CurQuantum += nQuantum;
			printf("%s - %d\n", szTemp,CurQuantum);
			break;

		/* initialize process state table */
		case INIT:
			for (x = 0; x < MAX_PROCESS; x++)
			{
				Process_status[x].nState = P_DONE;
				Process_status[x].nServ = 0;
				Process_status[x].nIo   = 0;
			}

			printf("\n\n |000,00,00| .... \n");
			printf("   A   B  C \n");
			printf(" A : random serie of cpu/io bit field in Hex.\n");
			printf("     if even next run is I/O else run is CPU\n");
			printf(" B : amount of cpu time received\n");
			printf(" C : amount of I/o time received\n");
			printf("\nThe state for each process at every quantum is shown next\n");
			printf("[space] : process not arrived or departed.\n");
			printf("      R : process ready\n");
			printf("      B : process blocked\n");
			printf("      X : process executing\n\n");
			for (x=0;x < MAX_PROCESS; x++)
				printf("   P%1d  |",x);
			printf(" States P0,...,P10    Qua\n");
			for(x=0;x < MAX_PROCESS;x++)
				printf("-------|");
			printf("-------------------------\n");
			break;
	}
}

/*********************************
** generate user processes.
**  pid = [0 ... 9]
**  random execution req xx xxxx xxxx
**    if x = 0 i/o
**    else cpu req.
**********************************/
int gen_process()
{
	int x,
		 y;

	MSG Msg;
	int outQ,
		 inQ;

	PCB pcb;

	/* get message queue id's */
	if (get_queueid(&outQ, &inQ))
	{
		printf("I/O process error getting queue ids \n");
		return(-1);
	}

	for (x=0; x < MAX_PROCESS; x++)
	{
		Msg.data.nPid = x;
		Msg.data.nReq = (rand() & 0x03FF);
		Msg.data.nServ = 0;
		Msg.data.nIo = 0;
		Msg.data.nSreq = get_servreq(Msg.data.nReq);
		Msg.type = MSG_NEW;
		msgsnd(outQ,&Msg,sizeof(PCB),0);
		/* wait a litle  to get randomness*/
		for (y=0;y< 30000;y++); 
	}
	return(0);
}

/*********************************
** get the system queues.
*********************************/
int get_queueid(int *SchedQ, int *IoQ)
{
	/* get message queue id's */
	if (!(*SchedQ = msgget(SCHED_Q, PERMS))  || 
	    !(*IoQ    = msgget(IO_Q,    PERMS)))
		return(-1);

	return(0);
}

/*********************************
** insert process in ready queue
** for round robin scheduler
*********************************/
int rr_insert(linklist *lpReadyQ, PCB *lpPcb)
{
	llsetend(lpReadyQ);
	return(lladd(lpReadyQ,lpPcb,LLNEXT));
}

/*********************************
** insert process in ready queue
** for shortest job first scheduler
*********************************/
int sjf_insert(linklist *lpReadyQ, PCB *lpPcb)
{
	llsetbeg(lpReadyQ);
	/* if found greater job, insert in front */
	if (llfind(lpReadyQ,lpPcb,0,LLEQUAL))
		return(lladd(lpReadyQ,lpPcb,LLPREV));
	/* else insert at tail */
	return(lladd(lpReadyQ,lpPcb,LLNEXT));
}

/*********************************
** get the cpu max requirement.
** used for shortest job first.
*********************************/
int get_servreq(int nValue)
{
	int x;
	int nRetval = 0;

	for (x=0; x < MAX_RUN; x++)
	{
		nRetval += nValue & 0x0001;
		nValue >>= 1;	
	}

	return (nRetval);
}

/*********************************
** search function used for
** ready queue in shortest job first
** mode.
*********************************/
int find_job(void *lpData, void *lpValue)
{
	/* order by remaining service to provide 
			lpData = job in ready queue
			lpValue - new job */
	if ((((PCB *)lpData)->nSreq  - ((PCB *)lpData)->nServ)   > 
		 (((PCB *)lpValue)->nSreq - ((PCB *)lpValue)->nServ))
		return(LLFOUND);

	return(LLNOTFOUND);
}
