/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		highpass.c - Apply the Ideal Highpass Filter or the
*				     Butterworth Highpass Filter to a complex
*			      	     Fourier image.  The resultant effect is
*				     to sharpen the image in the spatial domain.
* 				     The input complex image should be produced
*				     by the FFT routine with the origin shifted
*				     to the centre of the image.
* REVISION:             3.1
* AUTHOR:               DH
* CREATION DATE:        19 Nov 1991
* REVISION DATE:	7/10/92        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.1
* REVISION DATE:	11 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:		
* REVISION DATE: 	21 Oct 1988
* COMMENT:		modified from an older version of V.I.P.
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)highpass.c	3.1 7/10/92";
#endif



#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "vip.h"
#include "misc.h"

void            Usage();
void Cal_Dist();
void Filter();


double *dist;
int     ideal = 0, butterw = 0, n;
double  radius;
static int nr, nc, half_nr, half_nc;
IMAGE  *cim;


/*- Usage -----------------------------------------------------------

--------------------------------------------------------------------*/

void Usage()
{
    (void) fprintf(stderr, "usage: highpass [-f filter_type] [-r D n] [-i in_file] [-o out_file]\n");
    (void) fprintf(stderr, "  filter_type: I - Ideal highpass filter  (default)\n");
    (void) fprintf(stderr, "               B - Butterworth highpass filter\n");
    (void) fprintf(stderr, "  D (float)  : radius of the ring  (default=half image size)\n");
    (void) fprintf(stderr, "  n (integer): used only for the Butterworth filter (default=1)\n");
    exit(1);
}


/*- Cal_Dist --------------------------------------------------------

Calculate H(u,v) = 1  if  D(u,v) >= radius
                   0  if  D(U,v) <  radius
for Ideal Highpass Filter
or        H(u,v) = 1/(1 + D(u,v)/radius)^2*n
for Butterworth Highpass Filter.

--------------------------------------------------------------------*/

void Cal_Dist()
{
    static int rr, cc;
    double  diffr, diffc, d_uv;
    double *dp;

    dist = (double *) malloc(sizeof(double) * half_nr * half_nc);

    dp = dist;
    for (rr = 0; rr < half_nr; rr++)
	for (cc = 0; cc < half_nr; cc++, dp++) {
	    diffr = (double) (rr - half_nr);
	    diffc = (double) (cc - half_nc);
	    d_uv = sqrt(diffr * diffr + diffc * diffc);
	    if (ideal) {
		if (d_uv <= radius)
		    *dp = 0.0;
		else
		    *dp = 1.0;
	    }
	    else
		*dp = 1.0 / (1.0 + pow(radius / d_uv, (double) n));
	}
    free((char * ) dist);
}


/*- Filter ----------------------------------------------------------

Filter the image here.  Quadrant index: 1 3
                                        2 4

--------------------------------------------------------------------*/

void Filter()
{
    static int rr, cc, r, c;
    double *dp;
    COMPLEX *cp;


    for (dp = dist, rr = 0; rr < half_nr; rr++) {	/* first quadrant */
	cp = cim->i.cx[rr];
	for (cc = 0; cc < half_nc; cc++, dp++, cp++) {
	    cp->r *= *dp;
	    cp->i *= *dp;
	}
    }

    for (dp = dist, rr = nr - 1; rr >= half_nr; rr--) {	/* second quadrant */
	cp = cim->i.cx[rr];
	for (cc = 0; cc < half_nc; cc++, dp++, cp++) {
	    cp->r *= *dp;
	    cp->i *= *dp;
	}
    }

    for (dp = dist, rr = 0; rr < half_nr; rr++) {	/* third quadrant */
	cp = &(cim->i.cx[rr][nc - 1]);
	for (cc = nc - 1; cc >= half_nc; cc--, dp++, cp--) {
	    cp->r *= *dp;
	    cp->i *= *dp;
	}
    }

    for (dp = dist, rr = nr - 1; rr >= half_nr; rr--) {	/* fourth quadrant */
	cp = &(cim->i.cx[rr][nc - 1]);
	for (cc = nc - 1; cc >= half_nc; cc--, dp++, cp--) {
	    cp->r *= *dp;
	    cp->i *= *dp;
	}
    }
}


/*- Main ------------------------------------------------------------

Main body of the program.

--------------------------------------------------------------------*/

main(argc, argv)
int     argc;
char   *argv[];
{
    int     argn;
    char   *in_arg = NULL, *out_arg = NULL;

    for (argn = 1; argn < argc; argn++)
	if (argv[argn][0] == '-')
	    switch (argv[argn][1]) {
	    case 'o':
		out_arg = argv[++argn];
		break;
	    case 'f':		/* filter method: Ideal or Butterworth */
		argn++;
		if (argv[argn][0] == 'I')
		    ideal = 1;
		else if (argv[argn][0] == 'B')
		    butterw = 1;
		else
		    Usage();
		break;
	    case 'r':
		radius = atof(argv[++argn]);
	
                if (argv[argn+1][0] == '-') {
		    break;
	        }
		else {
                    n = atoi(argv[++argn]);
		    break;
		}
	    case 'i':
		in_arg = argv[++argn];
		break;
	    default:
		Usage();
	    }
       else
       {
            Usage();
       }

    if (ideal + butterw > 1)
	Usage();
    else if (ideal + butterw == 0)
	ideal = 1;		/* default filter method */
    if (radius < 0.0)
	Usage();

    if ((cim = ( IMAGE * ) Read_Image(in_arg)) == NULL)
	exit(1);

    (void) fprintf(stderr, "\n");
    if (ideal)
	(void) fprintf(stderr, "Ideal");
    else
	(void) fprintf(stderr, "Butterworth");
    (void) fprintf(stderr, " Highpass filtering -- program running.  Be patient...\n");

    n = n << 1;
    nr = cim->rows;
    nc = cim->cols;
    half_nr = nr >> 1;
    half_nc = nc >> 1;

    Cal_Dist();
    Filter();

    Write_Image(cim, out_arg);

    (void) fprintf(stderr, "Program completed...\n");

    exit(0);
}
