/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) Australian Wool Corporation/University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            LIB
* MODULE:		vipcalib.c		
* REVISION:             3.5
* AUTHOR:               PK
* CREATION DATE:        June 1991
* REVISION DATE:	4/22/94
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.5
* REVISION DATE:	22 April 1994
* COMMENT:		fixed to compile cleanly
* BY:			PK
*
* REVISION:		3.4
* REVISION DATE:	21 March 1994
* COMMENT:		Removed vector.h
* BY:			CFF
*
* REVISION:		3.3
* REVISION DATE:	07 Oct 1993
* COMMENT:		Fixed up for GENERIC build
* BY:			CFF
*
* REVISION:		3.2
* REVISION DATE:	16 August 1993
* COMMENT:		Fixed up for DEC build
* BY:			CFF
*
* REVISION:		3.1
* REVISION DATE:	9 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:
* REVISION DATE:	June 1991
* COMMENT:		Version 2.0  (UNIX)
* BY:			PK
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)vipcalib.c	3.5 4/22/94";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#ifdef SUN
#include <floatingpoint.h>
#endif
#include <string.h>
#include <memory.h>



#include "vip.h"
#include "vipiofn.h"
#include "vipcalibfn.h"
#include "misc.h"
#include "miscfn.h"
#include "vectorfn.h"

#ifdef XVIEW
#include <X11/Xlib.h>
#include <xview/xview.h>
#include <xview/notice.h>
#include <xview/frame.h>
#endif

long	PLANEBYTE;
int	Plane_no_array[40];
#ifdef XVIEW
extern  Frame frame;
extern  Frame toolsframe;
#endif
char	spl_str[250];


/*- xyz2uv -----------------------------------------------------------

Routine to convert xyz machine coordinates to integer uv image coordinates

--------------------------------------------------------------------*/

void    xyz2uv(camera, uv, xyz)
CAMERA  *camera;
int     *uv;
double  *xyz;
{
    double  hxyz[4], hu, hv, s;

    Move_d(xyz, 1, hxyz, 1, 3);
	
    hxyz[3] = 1;

    hu = Dot_d(&camera->calib_matrix[0][0], 1, hxyz, 1, 4);
    hv = Dot_d(&camera->calib_matrix[1][0], 1, hxyz, 1, 4);
    s =  Dot_d(&camera->calib_matrix[2][0], 1, hxyz, 1, 4);

    uv[0] = hu / s + camera->uvoffset;
    uv[1] = hv / s + camera->uvoffset;

    if (uv[0] > 511)
	uv[0] = 511;
    if (uv[1] > 511)
	uv[1] = 511;
    if (uv[0] < 0)
	uv[0] = 0;
    if (uv[1] < 0)
	uv[1] = 0;
    return;
}

/*- xyz2uvNC  ------------------------------------------------------------

Routine to convert xyz machine coordinates to uv image coordinates
This routine differs from xyzTOuv in that there is no clamping of
uv values to 0-500.  This is needed for converting direction vectors
in xyz space to vectors in uv space as used by contourpeak.

--------------------------------------------------------------------------*/


void    xyz2uvNC(camera, uv, xyz)
CAMERA  camera;
int     *uv;
double  *xyz;
{
    double  hxyz[4], hu, hv, s;

    Move_d(xyz, 1, hxyz, 1, 3);
    hxyz[3] = 1;

    hu = Dot_d(&camera.calib_matrix[0][0], 1, hxyz, 1, 4);
    hv = Dot_d(&camera.calib_matrix[1][0], 1, hxyz, 1, 4);
    s = Dot_d(&camera.calib_matrix[2][0], 1, hxyz, 1, 4);

    uv[0] = hu / s + camera.uvoffset;
    uv[1] = hv / s + camera.uvoffset;

    return;
}

/*- uv2xyz --------------------------------------------------------

Routine to convert image uv coords to machine xyz coords
using a sensor matrix.

------------------------------------------------------------------*/

void    uv2xyz(uv, xyz, sensor)
int     uv[2];
double  xyz[3];
SENSOR  sensor;
{
    double  huv[3], s;

    /* lens distortion correction will go here */


    huv[0] = (double) (uv[0] - sensor.camera.uvoffset);
    huv[1] = (double) (uv[1] - sensor.camera.uvoffset);
    huv[2] = 1.0;

    xyz[0] = Dot_d(&sensor.matrix[0][0], 1, huv, 1, 3);
    xyz[1] = Dot_d(&sensor.matrix[1][0], 1, huv, 1, 3);
    xyz[2] = Dot_d(&sensor.matrix[2][0], 1, huv, 1, 3);
    s = Dot_d(&sensor.matrix[3][0], 1, huv, 1, 3);

    Scal_d(1.0 / s, xyz, 1, xyz, 1, 3);

    return;
}

#ifdef CRAP
/*--------------------------------------------------------------------

c  Test correctness of system

      void test(xyz,u,v,n)
      implicit none
      int n
      double xyz[3], u, v
      save /sensordata/
      double sensor
      common /sensordata/sensor(14,10)

      save /peqn/
      double pdat
      common /peqn/pdat(4,10)

      save /calibdata/
      double calib
      common /calibdata/calib(12,0:3)

      double A[3][3],Ainv[3][3],b[3], bt[3]
      int i
      i = sensor[13][n]     !  camera No

      A[1][1] = pdat[1][n]
      A[1][2] = pdat[2][n]
      A[1][3] = pdat[3][n]
      A[2][1] = Calib[1][i] - u*Calib[9][i]
      A[2][2] = Calib[2][i] - u*Calib[10][i]
      A[2][3] = Calib[3][i] - u*Calib[11][i]
      A[3][1] = Calib[5][i] - v*Calib[9][i]
      A[3][2] = Calib[6][i] - v*Calib[10][i]
      A[3][3] = Calib[7][i] - v*Calib[11][i]

      b[1] = -pdat[4][n]
      b[2] = u*Calib[12][i] - Calib[4][i]
      b[3] = v*Calib[12][i] - Calib[8][i]

        inverse3(A,Ainv);

bt[1] =        Dot_d(Ainv[1][1],3,b,1,3);
bt[2] =        Dot_d(Ainv[2][1],3,b,1,3);
bt[3] =        Dot_d(Ainv[3][1],3,b,1,3);

      VIP_Error_Msg('("xyz (old)", 3f14.7)')xyz[1],xyz[2],xyz[3]
      VIP_Error_Msg('("xyz (new)", 3f14.7)')bt[1],bt[2],bt[3]

      return
      }

*/
#endif

/*-  Read_Plane_List  -------------------------------------------------

Function to read the plane list file and place the data in global array
PLANE

Expected format:

* Coments begin with '*' . Blank lines can be anywhere

plane n

 x1  y1  z1
 x2  y2  z2
 x3  y3  z3

plane n+1
etc.
etc.

Plane numbering does not have to be in order.  If a plane number is
repeated the plane data is overwritten with the new plane. (This should
be fixed to give you a warning if this happens).

---------------------------------------------------------------------*/

int     Read_Plane_List(filename, pln_no)
char   *filename;
int	pln_no;
{
    FILE   *datafile;
    char    file[80], message[80], line[80], *identifier,
	    *identifier2, *parameters, *token;
    int     planeNo, lineNo, PtNo, i, j, n;
    int	    command = 0;
    int	    PLANEEXISTS = 0;  /* ******************************************/
    long    PLANEMASK = 0;    /* BITWISE OPERATIONS ARE USED TO STORE AND */
    extern long	PLANEBYTE;    /* CHECK WHETHER A NUMBERED PLANE HAS BEEN  */
			      /* REQUESTED PREVIOUSLY BY USER etc.        */
			      /********************************************/

    for (j=0; j<40; j++) {
	Plane_no_array[j] = 0;
    }

    if (strcasecmp(filename, "default") == 0)	/* use standard file */
	(void) strcpy(file, PLANEDATAFILE);
    else
	(void) strcpy(file, filename);

    datafile = fopen(file, "r");
    if (!datafile) {
	(void) sprintf(message, "Read_Plane_List: Could not open file %s", file);
	VIP_Error_Msg(message);
	return (9);
    }
    PtNo = 0;
    lineNo = 0;



    while (fgets(line, 80, datafile) != NULL) {

	lineNo++;
	identifier = (char *) leftjust(line);
	parameters = strchr(line, ' ');	/* stuff after the first blank */

	if (strlen(identifier) == 0 ||
	    strncmp(identifier, "*", 1) == 0 ||
	    strncmp(identifier, "\n", 1) == 0) {
	    ;			/* Blank line or comment */

	}
	else if (strncasecmp(identifier, "plane", 5) == 0) {
	    (void) sscanf(parameters, "%d", &planeNo);
	    if (planeNo < FILEPLANEMIN || planeNo > FILEPLANEMAX) {
		(void) sprintf(message, "Read_Plane_List: plane No of %d out of legal range %d to %d",
		    planeNo, FILEPLANEMIN, FILEPLANEMAX);
		VIP_Error_Msg(message);
		(void) fclose(datafile);
		return (5);
	    }

	    if (planeNo==pln_no) {
		PLANEEXISTS = 1;
		if (Plane_no_array[planeNo] == 1) { 
		  (void) strcpy (spl_str, "Plane no already specified in this file\ny to overwite, n to ignore");

#ifdef XVIEW
		  command = overwite_opt (spl_str);
#endif
                  if (command == 0) {
                      fgets(line, 80, datafile);
                      fgets(line, 80, datafile);
                      fgets(line, 80, datafile);
                  }
	          else { /* ovewrite if plane array  no == 1 */
                      fgets(line, 80, datafile);
		      identifier2 = (char *) leftjust(line);
	              while ((strlen(identifier2) == 0) ||
	                     strncmp(identifier2, "\n", 1) == 0) {
                          fgets(line, 80, datafile);
		          identifier2 = (char *) leftjust(line);
	              }

	              for (n=0; n<3; n++) {
                          token = strtok(line, " ,");
	    	          i = 0;

                          while (token != NULL) {
	                      PLANE[planeNo][PtNo][i] = atof(token);
		              token = strtok((char *) NULL, " ,");
		              i++;
	                  }
			  i=0;
	                  PtNo++;
		          fgets(line, 80, datafile);
		       }
		       PtNo = 0;
		  }      /* end of overwiting if plane no array == 1*/
                }	 /* end of if plane no array == 1 */
	        else {  /*  overwiting if plane no array == 0*/
                   fgets(line, 80, datafile);
		   identifier2 = (char *) leftjust(line);
	           while ((strlen(identifier2) == 0) ||
	                strncmp(identifier2, "\n", 1) == 0) {
                        fgets(line, 80, datafile);
		        identifier2 = (char *) leftjust(line);
	           }

	           Plane_no_array[planeNo] = 1;
	           for (n=0; n<3; n++) {
                       token = strtok(line, " ,");
	    	       i = 0;

                          while (token != NULL) {
	                      PLANE[planeNo][PtNo][i] = atof(token);
		              token = strtok((char *) NULL, " ,");
		              i++;
	                  }
		       i=0;
	               PtNo++;
		       fgets(line, 80, datafile);
		   }
		   PtNo=0; 
	           Plane_no_array[planeNo] = 1;
                } 	/* end overwiting if plane no array == 0 */	
            }
	}
   }				/* while not EOF */
	        
    if (!PLANEEXISTS) {
        (void) sprintf(message, "Read_Plane_List: plane No %d is non-existant", planeNo);
        VIP_Error_Msg(message);
        (void) fclose(datafile);
        return (8);
    }

    return (OK);
}

/*--  Calc_Sensor  --------------------------------------------------

Routine to set up a sensor matrix.  The sensor matrix is the
transformation between image uv coords and machine xyz coords.
It is generated from the camera calibration matrix and the
equation of the plane in which the measurements are assumed to lie.
The No of the camera that is currently being used is also set.
See Bolles, Kremers and Cain "A Simple Sensor to Gather Three-Dimensional
Data", SRI Technical note 249 July 1981, for the derivation of the
equations.

--------------------------------------------------------------------*/

int     Calc_Sensor(camera, planepts, sensor)
CAMERA  camera;			/* input */
double  planepts[3][3];		/* input: 3 points defining plane */
SENSOR *sensor;			/* output */
{
    double  a[3][3], b[3], c[3];
    double  a11, a12, a13, a14, a21, a22, a23, a24, a31, a32, a33;
    double  b1, b2, b3, b4;
    int     i;

    /** set up matrix equation to solve plane coefficients

    First try to fit equation of the form  b[1]x + b[2]y + z + b[3] = 0.
    Note this equation cannot represent planes parallel to the z axis. If this
    fails we try an alternative representation below.  */


    for (i = 0; i < 3; i++) {
	a[i][0] = planepts[i][0];
	a[i][1] = planepts[i][1];
	a[i][2] = 1.0;
	c[i] = -planepts[i][2];
    }
    i = Gauss(( double * )a, b, c, 3);
    if (i) {			/* solution was ok */
	b1 = b[0];
	b2 = b[1];
	b3 = 1;
	b4 = b[2];
    }
    else {			/* That didn't work - try equation of the
				 * form  b[1]x + y + b[3]z + b[2] = 0 */
	for (i = 0; i < 3; i++) {
	    a[i][0] = planepts[i][0];
	    a[i][1] = 1.0;
	    a[i][2] = planepts[i][2];
	    c[i] = -planepts[i][1];
	}
	i = Gauss(( double * )a, b, c, 3);
	if (i) {		/* solution was ok */
	    b1 = b[0];
	    b2 = 1.0;
	    b3 = b[2];
	    b4 = b[1];
	}
	else {			/* That didn't work - try equation of the
				 * form  x + b[1]y + b[3]z + b[2] = 0 */
	    for (i = 0; i < 3; i++) {
		a[i][0] = 1.0;
		a[i][1] = planepts[i][1];
		a[i][2] = planepts[i][2];
		c[i] = -planepts[i][0];
	    }
	    i = Gauss(( double * )a, b, c, 3);
	    if (i) {		/* solution was ok */
		b1 = 1.0;
		b2 = b[1];
		b3 = b[2];
		b4 = b[0];
	    }
	    else {		/* failed */
		VIP_Error_Msg("Calc_Sensor: Unable to generate plane from data points");
		Print_Matrix_d((double *)a, 3, 3);
		return (ERROR);
	    }
	}
    }

    /* Set up some convenient variable names */

    a11 = camera.calib_matrix[0][0];
    a12 = camera.calib_matrix[0][1];
    a13 = camera.calib_matrix[0][2];
    a14 = camera.calib_matrix[0][3];

    a21 = camera.calib_matrix[1][0];
    a22 = camera.calib_matrix[1][1];
    a23 = camera.calib_matrix[1][2];
    a24 = camera.calib_matrix[1][3];

    a31 = camera.calib_matrix[2][0];
    a32 = camera.calib_matrix[2][1];
    a33 = camera.calib_matrix[2][2];	/* a34 assumed to be 1 */

    /* Now evaluate elements of the sensor matrix  */

    sensor->matrix[0][0] = (b4 * a22 - b2 * a24) * a33 + (b3 * a24 - b4 * a23) * a32 +
	(b2 * a23 - b3 * a22);
    sensor->matrix[0][1] = (b2 * a14 - b4 * a12) * a33 + (b4 * a13 - b3 * a14) * a32 +
	(b3 * a12 - b2 * a13);
    sensor->matrix[0][2] = (b2 * a13 - b3 * a12) * a24 + (b4 * a12 - b2 * a14) * a23 +
	(b3 * a14 - b4 * a13) * a22;
    sensor->matrix[1][0] = (b1 * a24 - b4 * a21) * a33 + (b4 * a23 - b3 * a24) * a31 +
	(b3 * a21 - b1 * a23);
    sensor->matrix[1][1] = (b4 * a11 - b1 * a14) * a33 + (b3 * a14 - b4 * a13) * a31 +
	(b1 * a13 - b3 * a11);
    sensor->matrix[1][2] = (b3 * a11 - b1 * a13) * a24 + (b1 * a14 - b4 * a11) * a23 +
	(b4 * a13 - b3 * a14) * a21;
    sensor->matrix[2][0] = (b4 * a21 - b1 * a24) * a32 + (b2 * a24 - b4 * a22) * a31 +
	(b1 * a22 - b2 * a21);
    sensor->matrix[2][1] = (b1 * a14 - b4 * a11) * a32 + (b4 * a12 - b2 * a14) * a31 +
	(b2 * a11 - b1 * a12);
    sensor->matrix[2][2] = (b1 * a12 - b2 * a11) * a24 + (b4 * a11 - b1 * a14) * a22 +
	(b2 * a14 - b4 * a12) * a21;
    sensor->matrix[3][0] = (b2 * a21 - b1 * a22) * a33 + (b1 * a23 - b3 * a21) * a32 +
	(b3 * a22 - b2 * a23) * a31;
    sensor->matrix[3][1] = (b1 * a12 - b2 * a11) * a33 + (b3 * a11 - b1 * a13) * a32 +
	(b2 * a13 - b3 * a12) * a31;
    sensor->matrix[3][2] = (b2 * a11 - b1 * a12) * a23 + (b1 * a13 - b3 * a11) * a22 +
	(b3 * a12 - b2 * a13) * a21;


    memcpy(&sensor->camera, &camera, sizeof(CAMERA));	/* copy camera data over
							 * to get lens
							 * distortion
							 * compensation data for
							 * use when converting
							 * uv coords to xyz */

    return (OK);
}

/*-- Set_Plane  -----------------------------------------------------------------

Function to allow planes to be specified by user or from other vision measurements
rather than from the plane list file.

-------------------------------------------------------------------------------*/

int     Set_Plane(planeNo, data)
int     planeNo;
double *data;
{
    char    message[80];

    if (planeNo < USERPLANEMIN || planeNo > USERPLANEMAX) {
	(void) sprintf(message, "Set_Plane: Illegal plane No of %d not in range %d to %d",
	    planeNo, USERPLANEMIN, USERPLANEMAX);
	VIP_Error_Msg(message);
	return (ERROR);
    }
    Move_d(data, 1, *PLANE[planeNo], 1, 9);
    return (OK);
}

/*--  Stereo_Point  ----------------------------------------------------------

Stereo point calculation

Function calculates world xyz position of a point given the
uv coordinates from two cameras.

-----------------------------------------------------------------------------*/

int     Stereo_Point(uv1, uv2, cam1, cam2, P)
double  uv1[2], uv2[2];		/* input image uv coords */
CAMERA  cam1, cam2;		/* cameras associated with uv coords */
double  P[3];			/* output world xyz coord */
{
    double  M[4][3], Mt[3][4], A[3][3], b[4], c[3];
    int     i;

    /* set up over constrained matrix equation (4 equations, 3 unknowns) */

    Piv_d(-uv1[0], &cam1.calib_matrix[2][0], 1, &cam1.calib_matrix[0][0], 1, &M[0][0], 1, 3);

    Piv_d(-uv1[1], &cam1.calib_matrix[2][0], 1, &cam1.calib_matrix[1][0], 1, &M[1][0], 1, 3);
    Piv_d(-uv2[0], &cam2.calib_matrix[2][0], 1, &cam2.calib_matrix[0][0], 1, &M[2][0], 1, 3);
    Piv_d(-uv2[1], &cam2.calib_matrix[2][0], 1, &cam2.calib_matrix[1][0], 1, &M[3][0], 1, 3);

    b[0] = uv1[0] * cam1.calib_matrix[2][3] - cam1.calib_matrix[0][3];
    b[1] = uv1[1] * cam1.calib_matrix[2][3] - cam1.calib_matrix[1][3];
    b[2] = uv2[0] * cam2.calib_matrix[2][3] - cam2.calib_matrix[0][3];
    b[3] = uv2[1] * cam2.calib_matrix[2][3] - cam2.calib_matrix[1][3];

    /* solve using the Pseudo-inverse */

    Transpose_d(4, 3, &M[0][0], &Mt[0][0]);
    Mul_d(3, 4, 3, &Mt[0][0], &M[0][0], &A[0][0]);
    Mul_d(3, 4, 1, &Mt[0][0], b, c);
    i = Gauss(&A[0][0], P, c, 3);
    if (i == 0) {		/* singular */
	VIP_Error_Msg("Stereo_Point: Stereo solution failure");
	return (ERROR);
    }
    return (OK);
}

/*-- Lens_Comp_4  --------------------------------------------------------------

4th order Function to compensate for lens distortion

--------------------------------------------------------------------------------*/

int     Lens_Comp_4(iuv, duv, camera, npts)
int     iuv[];			/* input: integer pixel coords */
double  duv[];			/* output: real valued compensated
				 * coordinates */
CAMERA  camera;
int     npts;
{
    int     i;
    double  u, v, u2, v2, u3, v3, u4, v4, A[15];

    for (i = 0; i < npts; i++) {
	u = iuv[2 * i] - camera.uvoffset;
	v = iuv[2 * i + 1] - camera.uvoffset;
	u2 = u * u;
	v2 = v * v;
	u3 = u * u2;
	v3 = v * v2;
	u4 = u2 * u2;
	v4 = v2 * v2;
	A[0] = 1.0;
	A[1] = u;
	A[2] = v;
	A[3] = u2;
	A[4] = u * v;
	A[5] = v2;
	A[6] = u3;
	A[7] = u2 * v;
	A[8] = u * v2;
	A[9] = v3;
	A[10] = u4;
	A[11] = u3 * v;
	A[12] = u2 * v2;
	A[13] = u * v3;
	A[14] = v4;

	duv[2 * i] = camera.uvoffset + Dot_d(camera.Ucomp, 1, A, 1, 15);
	duv[2 * i + 1] = camera.uvoffset + Dot_d(camera.Vcomp, 1, A, 1, 15);
    }
    return (OK);
}

/*-- Lens_Comp_3  --------------------------------------------------------------

3rd order Function to compensate for lens distortion

--------------------------------------------------------------------------------*/

int     Lens_Comp_3(iuv, duv, camera, npts)
int     iuv[];			/* input: integer pixel coords */
double  duv[];			/* output: real valued compensated
				 * coordinates */
CAMERA  *camera;
int     npts;
{
    int     i;
    double  u, v, u2, v2, u3, v3, A[10];

    for (i = 0; i < npts; i++) {
	u = iuv[2 * i] - camera->uvoffset;
	v = iuv[2 * i + 1] - camera->uvoffset;
	u2 = u * u;
	v2 = v * v;
	u3 = u * u2;
	v3 = v * v2;
	A[0] = 1.0;
	A[1] = u;
	A[2] = v;
	A[3] = u2;
	A[4] = u * v;
	A[5] = v2;
	A[6] = u3;
	A[7] = u2 * v;
	A[8] = u * v2;
	A[9] = v3;

	duv[2 * i] = camera->uvoffset + Dot_d(camera->Ucomp, 1, A, 1, 10);
	duv[2 * i + 1] = camera->uvoffset + Dot_d(camera->Vcomp, 1, A, 1, 10);
    }
    return (OK);
}



/*- Calibrate --------------------------------------------------------

Camera calibration function.

Inputs are:
an array of image coordinates (actually frame grabber coordinates).
an array of corresponding world xyz coordinates

Output is a CAMERA structure containing a calibration matrix
and lens distortion factors.

Peter Kovesi                      December 1987
    Updated for lens distortion   October  1988
    Version 2 system on the Sun   October  1991

----------------------------------------------------------------------*/

int     Calibrate(uv, xyz, npts, camera, filename, lenscomp)
double  uv[][2];		/* input: observed image positions */
double  xyz[][3];		/* input: 3D positions of calibration points */
int     npts;			/* input: No of points */
CAMERA *camera;			/* output: Camera data structure */
char   *filename;		/* input: Filename to place calibration data */
int     lenscomp;		/* input: 1/0 calculate lens distortion
				 * compensation */
{
    int     i;
    int    *iuv;
    double  zero = 0.0;
    double *E, *Etrans, *P, *Ptrans;
    double  Ac[121], b[11];
    double  hxyz[4], huv[3];
    double  cu, cv, *cuv;
    double *uv_i, *uv_comp;
    double *buf;
    double  Condition, ouerr, overr, uerr, verr;
    char    message[80], lfilename[80];
    FILE   *file;


    if (npts < 6) {
	VIP_Error_Msg("Calibrate: Camera calibration cannot be done with less than six points ");
	return (ERROR);
    }
    E = (double *) malloc(11 * 2 * npts * sizeof(double));
    Etrans = (double *) malloc(11 * 2 * npts * sizeof(double));
    P = (double *) malloc(npts * 10 * sizeof(double));
    Ptrans = (double *) malloc(npts * 10 * sizeof(double));
    uv_i = (double *) malloc(2 * npts * sizeof(double));
    uv_comp = (double *) malloc(2 * npts * sizeof(double));
    cuv = (double *) malloc(2 * npts * sizeof(double));
    iuv = (int *) malloc(2 * npts * sizeof(int));
    buf = (double *) malloc(npts * sizeof(double));


    for (i = 0; i < npts; i++) {/* For each calibration point ... */
	iuv[2 * i + 0] = uv[i][0];
	iuv[2 * i + 1] = uv[i][1];

	cuv[2 * i + 0] = (uv[i][0] - UVOFFSET);
	cuv[2 * i + 1] = (uv[i][1] - UVOFFSET);

	/* fill E matrix */

	Move_d(&xyz[i][0], 1, &E[11 * (i * 2) + 0], 1, 3);
	E[11 * (i * 2) + 3] = 1.0;
	Movexd(&zero, 0, &E[11 * (i * 2) + 4], 1, 4);
	Scal_d(-cuv[2 * i + 0], &xyz[i][0], 1, &E[11 * (i * 2) + 8], 1, 3);

	Movexd(&zero, 0, &E[11 * (i * 2 + 1) + 0], 1, 4);
	Movexd(&xyz[i][0], 1, &E[11 * (i * 2 + 1) + 4], 1, 3);
	E[11 * (i * 2 + 1) + 7] = 1.0;
	Scal_d(-cuv[2 * i + 1], &xyz[i][0], 1, &E[11 * (i * 2 + 1) + 8], 1, 3);

    }				/* for each calibration point */


    /*
     * Now have to solve equation   [E][a] = [uv] using the pseudo-inverse
     * where [E] is a 2*Npts x 11 matrix [a] is an 11 vector [cuv] is a
     * 2*Npts vector
     */

    Transpose_d(2 * npts, 11, E, Etrans);	/* Take transpose of E */

    Mul_d(11, 2 * npts, 11, Etrans, E, Ac);	/* Multiply Etrans and E */
    Mul_d(11, 2 * npts, 1, Etrans, cuv, b);	/* Multiply Etrans and cuv */

    /* Solve by Gaussion elimination */

    Condition = Gauss_Cond(Ac, (double *)&camera->calib_matrix, b, 11, 1);
    (void) printf("\nCondition No of matrix used to solve camera calibration matrix was %f \n", Condition);

    if (Condition == ERROR)
	goto END;

    /* Fill in remaining parameters */

    camera->idcode = CAMERA_ID;
    camera->calib_matrix[2][3] = 1.0;
    camera->uvoffset = UVOFFSET;
    if (lenscomp)
	camera->calib_status = MATRIX_PLUS_3POLY;
    else
	camera->calib_status = MATRIX_ONLY;

    (void) printf("\n Camera matrix \n");
    Print_Matrix_d((double *)camera->calib_matrix, 3, 4);

    /*
     * Apply the calibration points to  calib_matrix to check that they map
     * back onto the right spots in the image. Also they provide 'ideal'
     * image points to be compared to observed image points.
     */

    for (i = 0; i < npts; i++) {
	Move_d(&xyz[i][0], 1, hxyz, 1, 3);
	hxyz[3] = 1.0;
	Mul_d(3, 4, 1, (double *)&camera->calib_matrix, hxyz, huv);
	uv_i[2 * i + 0] = huv[0] / huv[2] + UVOFFSET;
	uv_i[2 * i + 1] = huv[1] / huv[2] + UVOFFSET;
    }


    /*
     * Now calculate coefficients of 3nd order polynomial correction function
     * which (hopes) to correct for lens non-linearity. See Bowman and
     * Forrest "Transformation calibration of a camera mounted on a robot",
     * Image and Vision Computing Vol 5, No 4 November 87. pp 261-266.
     */

    Move_d(&zero, 0, camera->Ucomp, 1, 18);	/* first clear the arrays */
    Move_d(&zero, 0, camera->Vcomp, 1, 18);

    if (lenscomp) {		/* proceed with the lens distortion
				 * compensation calculation */

	/* fill P matrix */

	for (i = 0; i < npts; i++) {
	    cu = (uv[i][0] - UVOFFSET);
	    cv = (uv[i][1] - UVOFFSET);
	    P[10 * i + 0] = 1.0;
	    P[10 * i + 1] = cu;
	    P[10 * i + 2] = cv;
	    P[10 * i + 3] = cu * cu;
	    P[10 * i + 4] = cu * cv;
	    P[10 * i + 5] = cv * cv;
	    P[10 * i + 6] = cu * cu * cu;
	    P[10 * i + 7] = cu * cu * cv;
	    P[10 * i + 8] = cu * cv * cv;
	    P[10 * i + 9] = cv * cv * cv;
	}

	Transpose_d(npts, 10, P, Ptrans);	/* Take transpose of P */
	Mul_d(10, npts, 10, Ptrans, P, Ac);	/* Multiply Ptrans and P */

	/* solve for u coordinate compensation parameters */

	Move_d(uv_i, 2, buf, 1, npts);	/* fill buf with the 'ideal' u
					 * coordinates */
	for (i = 0; i < npts; i++)
	    buf[i] -= UVOFFSET;
	Mul_d(10, npts, 1, Ptrans, buf, b);	/* Multiply Ptrans and buf */

	Condition = Gauss_Cond(Ac, camera->Ucomp, b, 10, 1);	/* Solve by Gaussian
								 * Elimination */

	if (Condition == ERROR) {
	    VIP_Error_Msg("Calibrate: Solution error in calculating lens distortion");
	    goto END;
	}
	/* solve for v coordinate compensation parameters */

	Move_d(&uv_i[2 * 0 + 1], 2, buf, 1, npts);	/* fill buf with the
							 * 'ideal' v coordinates */
	for (i = 0; i < npts; i++)
	    buf[i] -= UVOFFSET;
	Mul_d(10, npts, 1, Ptrans, buf, b);	/* Multiply Ptrans and buf */

	Condition = Gauss_Cond(Ac, camera->Vcomp, b, 10, 1);	/* Solve by Gaussian
								 * Elimination */
	(void) printf("\nCondition No of matrix used to solve lens distortion parameters was %f \n", Condition);
	if (Condition == ERROR) {
	    VIP_Error_Msg("Calibrate: Solution error in calculating lens distortion");
	    goto END;
	}
    }				/* if lenscomp */
    /* write camera structure to a binary file */
    file = fopen(filename, "w");
    if (!file) {
	(void) sprintf(message, "Calibrate: Could not open file %s", filename);
	VIP_Error_Msg(message);
	goto END;
    }
    (void) fwrite(camera, sizeof(struct struct_vip_CAMERA), 1, file);
    (void) fclose(file);

    /* also write out a text file with error data */

    (void) strcpy(lfilename, filename);
    (void) strcat(lfilename, ".txt");

    (void) Write_Camera_Txt(lfilename, camera);

    /*
     * Now test effectiveness of calibration and compensation and append this
     * data to the camera calibration file
     */

    (void) fopen(lfilename, "a");
    (void) fprintf(file, "\n\n Calibration Accuracy Data - all units are in 'pixels'\n");
    (void) fprintf(file, "1st Col is observed image position\n");
    (void) fprintf(file, "2nd Col is corrected image position\n");
    (void) fprintf(file, "3rd Col is ideal image position\n");
    (void) fprintf(file, "4th Col is difference between observed and ideal image positions\n");
    (void) fprintf(file, "5th Col is difference between compensated and ideal image positions\n");

    if (lenscomp)
	(void) Lens_Comp_3(iuv, uv_comp, camera, npts);
    else
	Move_d(uv_i, 1, uv_comp, 1, 2 * npts);	/* make compensated data =
						 * ideal positions */

    ouerr = overr = uerr = verr = 0.0;

    for (i = 0; i < npts; i++) {
	(void) fprintf(file, "%3d   %8.3f    %8.3f    %8.3f    %8.5f    %8.5f    \n",
	    i + 1, uv[i][0], uv_comp[2 * i + 0], uv_i[2 * i + 0],
	    uv[i][0] - uv_i[2 * i + 0], uv_comp[2 * i + 0] - uv_i[2 * i + 0]);

	(void) fprintf(file, "      %8.3f    %8.3f    %8.3f    %8.5f    %8.5f    \n",
	    uv[i][1], uv_comp[2 * i + 1], uv_i[2 * i + 1],
	    uv[i][1] - uv_i[2 * i + 1], uv_comp[2 * i + 1] - uv_i[2 * i + 1]);

	uerr += SQR(uv_comp[2 * i + 0] - uv_i[2 * i + 0]);
	verr += SQR(uv_comp[2 * i + 1] - uv_i[2 * i + 1]);
	ouerr += SQR(uv[i][0] - uv_i[2 * i + 0]);
	overr += SQR(uv[i][1] - uv_i[2 * i + 1]);
    }
    uerr = uerr / npts;
    verr = verr / npts;
    ouerr /= npts;
    overr /= npts;

    (void) fprintf(file, "\nUncompensated Data\n");
    (void) fprintf(file, "\nMean Squared U error %f\n", ouerr);
    (void) fprintf(file, "Mean Squared V error %f\n", overr);
    (void) printf("\nUncompensated Data\n");
    (void) printf("\nMean Squared U error %f\n", ouerr);
    (void) printf("Mean Squared V error %f\n", overr);

    if (lenscomp) {
	(void) fprintf(file, "\nCompensated Data\n");
	(void) fprintf(file, "\nMean Squared U error %f\n", uerr);
	(void) fprintf(file, "Mean Squared V error %f\n", verr);
	(void) printf("\nCompensated Data\n");
	(void) printf("\nMean Squared U error %f\n", uerr);
	(void) printf("Mean Squared V error %f\n", verr);
    }
    (void) fclose(file);
    return (1);
END:
    return (0);
}


/** Read_Calib_Pts ---------------------------------------------------

Function to read a calibration data file containing 3D coordinates
of calibration points.

 Expected file format:

 File description, comments etc

 point 1 x y z
 point 2 x y z
   .   . . . .
 point n x y z

 All lines in the file are ignored unless they start with the word 'POINT'.
 Hence, text description can be used liberally.  Point numbers are ignored
 and are simply there for convenient reference.

----------------------------------------------------------------------*/

int     Read_Calib_Pts(filename, CalibPts, Npts)
char   *filename;
double  CalibPts[][3];
int    *Npts;
{
    char    buffer[80], text[80], *line;
    int     n, PtNo, Pt;
    FILE   *datafile;
    double  buf[3];
    char    message[80];

    datafile = fopen(filename, "r");

    if (datafile == NULL) {
	(void) sprintf("Read_Calib_Pts: Could not open data file %s", filename);
	VIP_Error_Msg(message);
	return (ERROR);
    }
    PtNo = 0;

    while (fgets(buffer, 81, datafile) != NULL) {
	line = (char *) leftjust(buffer);

	if (strncasecmp(line, "POINT", 5) == 0) {
	    PtNo++;

	    n = sscanf(line, " %s %d %lf %lf %lf ", text, &Pt,
		buf, buf + 1, buf + 2);

	    if (n != 5) {
		(void) sprintf(message, "Read_Calib_Pts: Error in reading data file %s", filename);
		VIP_Error_Msg(message);
		return (ERROR);
	    }
	    else {
		Move_d(buf, 1, &CalibPts[PtNo - 1][0], 1, 3);
	    }
	}
    }



    *Npts = PtNo;

    (void) fclose(datafile);
    return (OK);
}



/** Read_Image_Pts ---------------------------------------------------

Function to read a calibration data file containing image data points

 Expected file format:

 File description, comments etc

 imagepoint 1 u v
 imagepoint 2 u v
   .   . . . .
 imagepoint n u v

 All lines in the file are ignored unless they start with the word 'IMAGEPOINT'.
 Hence, text description can be used liberally.  Point numbers are ignored
 and are simply there for convenient reference.

----------------------------------------------------------------------*/

int     Read_Image_Pts(filename, ImagePts, Npts)
char   *filename;
double  ImagePts[][2];
int    *Npts;
{
    char    buffer[80], text[80], *line;
    int     n, PtNo, Pt;
    FILE   *datafile;
    double  buf[2];
    char    message[80];

    datafile = fopen(filename, "r");

    if (datafile == NULL) {
	(void) sprintf(message, "Read_Image_Pts: Could not open data file %s", filename);
	VIP_Error_Msg(message);
	return (ERROR);
    }
    PtNo = 0;

    while (fgets(buffer, 81, datafile) != NULL) {
	line = (char *) leftjust(buffer);

	if (strncasecmp(line, "IMAGEPOINT", 10) == 0) {
	    PtNo++;

	    n = sscanf(line, " %s %d %lf %lf ", text, &Pt,
		buf, buf + 1);

	    if (n != 4) {
		(void) sprintf(message, "Read_Image_Pts: Error in reading data file %s", filename);
		VIP_Error_Msg(message);
		return (ERROR);
	    }
	    else {
		Move_d(buf, 1, &ImagePts[PtNo - 1][0], 1, 2);
	    }
	}
    }



    *Npts = PtNo;

    (void) fclose(datafile);
    return (OK);
}


/** Write_Camera_Txt --------------------------------------------------

Function to write a CAMERA data structure into a text file

----------------------------------------------------------------------*/

int     Write_Camera_Txt(filename, camera)
char   *filename;
CAMERA *camera;
{
    FILE   *datafile;
    int     i;

    datafile = fopen(filename, "w");

    /*
     * _strdate(date); _strtime(time);
     */
    (void) fprintf(datafile, "* Calibration Data:   %s  \n\n", filename);

    (void) fprintf(datafile, "channel      %d\n", camera->channel);
    (void) fprintf(datafile, "gain         %d\n", camera->gain);
    (void) fprintf(datafile, "offset       %d\n", camera->offset);
    (void) fprintf(datafile, "uvoffset     %d\n", camera->uvoffset);
    (void) fprintf(datafile, "calib_status %d\n", camera->calib_status);

    /* Write in calibration data */
    (void) fprintf(datafile, "\nCalibration Data\n");
    for (i = 0; i < 3; i++) {
	(void) fprintf(datafile, "%16.9f %16.9f %16.9f %16.9f \n",
	    camera->calib_matrix[i][0], camera->calib_matrix[i][1],
	    camera->calib_matrix[i][2], camera->calib_matrix[i][3]);
    }

    /* Now write in compensation factors */

    (void) fprintf(datafile, "\n\nLens Distortion Parameters\n");
    (void) fprintf(datafile, "\n           Ucomp                       Vcomp\n\n");
    for (i = 0; i < 18; i++)
	(void) fprintf(datafile, "%24.14e  %24.14e\n", camera->Ucomp[i], camera->Vcomp[i]);

    (void) fclose(datafile);
    return (OK);
}



/** Read_Image_Pts_From_Calib_File  -----------------------------------

Function to read uv coords from a text camera calibration data file.

-----------------------------------------------------------------------*/

int     Read_Image_Pts_From_Calib_File(filename, uv, Npts)
char   *filename;
double  uv[][2];
int    *Npts;
{
    char    buffer[80], *line, *identifier;
    FILE   *datafile;
    int     uData = FALSE, vData = FALSE;
    int     i, row = 0;
    char    message[80];

    datafile = fopen(filename, "r");

    if (datafile == NULL) {
	(void) sprintf(message, "Read_Image_Pts: Could not open data file %s", filename);
	VIP_Error_Msg(message);
	return (ERROR);
    }
    while (fgets(buffer, 80, datafile) != NULL) {

	line = strdup(buffer);
	identifier = (char *) leftjust(line);

	if (strncasecmp(identifier, "5TH", 3) == 0) {
	    uData = TRUE;
	    row = 0;		/* initialise counter */
	}
	else if (uData == TRUE) {
	    if (sscanf(line, "%d %lf",
		    &i, &uv[row][0]) != 2)
		goto END;
	    vData = TRUE;
	    uData = FALSE;
	}
	else if (vData == TRUE) {
	    if (sscanf(line, "%lf",
		    &uv[row][1]) != 1)
		goto END;
	    row++;
	    uData = TRUE;
	    vData = FALSE;
	}
    }				/* while not EOF */
END:
    *Npts = row;

    (void) fclose(datafile);
    return (OK);
}


/** Read_Camera ------------------------------------------------------

Function to read a binary camera data file.

--------------------------------------------------------------------*/

CAMERA *Read_Camera(filename)
char   *filename;
{
    FILE   *file;
    CAMERA *camera;
    char    message[80];

    file = fopen(filename, "r");
    if (!file) {
	(void) sprintf(message, "Read_Camera: Could not open file %s", filename);
	VIP_Error_Msg(message);
	return (NULL);
    }
    camera = (struct struct_vip_CAMERA *) malloc(sizeof(CAMERA));
    (void) fread(camera, sizeof(struct struct_vip_CAMERA), 1, file);
    if (camera->idcode != CAMERA_ID) {
	(void) sprintf(message, "Read_Camera: %s is not a binary camera file", filename);
	VIP_Error_Msg(message);
	return (NULL);
    }
    return (camera);
}



#ifdef XVIEW
int	overwite_opt(err_str)
 char   *err_str;
{
    Xv_notice   badfile_notice;
    int         res_stat;

        badfile_notice = xv_create (toolsframe,
                            NOTICE,
                            NOTICE_FOCUS_XY, 30, 30,
                            NOTICE_MESSAGE_STRING, err_str,
                            NOTICE_BUTTON_YES, "Overwrite",
                            NOTICE_BUTTON_NO, "Ignore",
                            NOTICE_STATUS, &res_stat,
                            XV_SHOW, TRUE,
                            XV_NULL);

        switch (res_stat) {
            case NOTICE_YES:
                xv_destroy_safe(badfile_notice);
                res_stat = 0;
		return (1);
            case NOTICE_NO:
                xv_destroy_safe(badfile_notice);
                res_stat = 0;
		(void) strcpy (err_str, "\0");
		return (0);
        }
}
#endif


