/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            LIB
* MODULE:		vector.c - Library of vector manipulation functions
* REVISION:             3.5
* AUTHOR:		PK               
* CREATION DATE:        October  1986
* REVISION DATE:	4/26/94        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.5
* REVISION DATE:	26 April 1994
* COMMENT:		Fixed to compile more cleanly
* BY:			PK
*
* REVISION:		3.4
* REVISION DATE:	21 March 1994
* COMMENT:		Added misc.h
* BY:			CFF
*
* REVISION:		3.3
* REVISION DATE:	08 Nov 1993
* COMMENT:		Fixed up for GENERIC build
* BY:			CFF
*
* REVISION:		3.2
* REVISION DATE:	16 August 1993
* COMMENT:		Fixed up for DEC build
* BY:			CFF
*
* REVISION:		3.1
* REVISION DATE:	7 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:	
* REVISION DATE:	December 1991
* COMMENT:		
* BY:			DH
*
* REVISION:	
* REVISION DATE:	June 1991
* COMMENT:	        Version 2.0  (UNIX)
* BY:			
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)vector.c	3.5 4/26/94";
#endif


/*-----------------------------------------------------------------------------

Note that several versions of each function exist depending on the data
type of the arguments.  The convention is: if the function expects integers
it will have postscript i, long integers - postscript l, floats - f and
doubles - d.  Note that C does all its floating point operations in double
precision.  Hence only integer and double versions of functions will typically
be supported.

The general form of list of arguments for many of the vector manipulation
functions are:

pointer to vector 1, size of incremental step along vector 1,
pointer to vector 2, size of incremental step along vector 2, etc, number
of elements to which these operations must be performend.

For example,  Add_i( v1, i1, v2, i2, v3, i3, n)
- add the contents of v2 to v1 and place the result in v3 stepping
along each vector at the specified increments for n elements.

Usually the increment size is one, however for matrix operations other
increments may be useful.  For example if one wanted to rescale the
first column of a 3x3 matrix of integers

       | a1  a2  a3 |
       | a4  a5  a6 |  one would use Scale_i(scalar, a, 3, a, 3, 3)
       | a7  a8  a9 |

Note that v1, v2 etc can be the same vector, also increments may be
zero or negative.

Eg. to initialize an integer vector v1 of n elements to zero one could use:
  zero = 0;
  Move_i( &zero, 0, v1, 1, n);

To reverse the order of an array one can use
  Move_i( &v1[n-1], -1, v1, 1, n);

--------------------------------------------------------------------------*/

/*
#include <stdlib.h>
#include <stdarg.h>
*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <limits.h>
#include <values.h>

#include "misc.h"
#include "vectorfn.h"

/** Function to find largest element in an array of integers *******/

int     Max_i(array, n, element)
int     array[], n, *element;
{
    int     max = -MAXINT;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] > max) {
	    max = array[i];
	    *element = i;
	}
    }
    return (max);
}


/** Function to find largest element in an array of long integers *****/

long    Max_l(array, n, element)
long    array[];
int     n, *element;
{
    long    max = -MAXLONG;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] > max) {
	    max = array[i];
	    *element = i;
	}
    }
    return (max);
}

/** Function to find largest element in an array of floats ************/

float   Max_f(array, n, element)
float   array[];
int     n, *element;
{
    float   max = MINFLOAT;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] > max) {
	    max = array[i];
	    *element = i;
	}
    }
    return (max);
}

/** Function to find largest element in an array of doubles ************/

double  Max_d(array, n, element)
double  array[];
int     n, *element;
{
    double  max = MINDOUBLE;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] > max) {
	    max = array[i];
	    *element = i;
	}
    }
    return (max);
}

/** Function to find smallest element in an array of integers *******/

int     Min_i(array, n, element)
int     array[], n, *element;
{
    int     min = MAXINT;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] < min) {
	    min = array[i];
	    *element = i;
	}
    }
    return (min);
}

/** Function to find smallest element in an array of long integers *****/

long    Min_l(array, n, element)
long    array[];
int     n, *element;
{
    long    min = MAXLONG;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] < min) {
	    min = array[i];
	    *element = i;
	}
    }
    return (min);
}

/** Function to find smallest element in an array of floats ************/

float   Min_f(array, n, element)
float   array[];
int     n, *element;
{
    float   min = MAXFLOAT;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] < min) {
	    min = array[i];
	    *element = i;
	}
    }
    return (min);
}

/** Function to find smallest element in an array of doubles ************/

double  Min_d(array, n, element)
double  array[];
int     n, *element;
{
    double  min = MAXDOUBLE;
    int     i;

    for (i = 0; i < n; i++) {
	if (array[i] < min) {
	    min = array[i];
	    *element = i;
	}
    }
    return (min);
}



/** Function to clamp an integer to limits *************************/
/* Function returns 1 if variable had to be clamped, 0 if not.    */

int     Limit_Value_i(x, xmin, xmax, error_message)
int    *x, xmin, xmax;
char   *error_message;
{

    if (*x < xmin) {
	*x = xmin;
	(void) printf("%s", error_message);
	return (1);
    }
    else if (*x > xmax) {
	*x = xmax;
	(void) printf("%s", error_message);
	return (1);
    }
    else
	return (0);

}


/** Function to clamp a long integer to limits *************************/
/* Function returns 1 if variable had to be clamped, 0 if not.    */

int     Limit_Value_l(x, xmin, xmax, error_message)
long   *x, xmin, xmax;
char   *error_message;
{

    if (*x < xmin) {
	*x = xmin;
	(void) printf("%s", error_message);
	return (1);
    }
    else if (*x > xmax) {
	*x = xmax;
	(void) printf("%s", error_message);
	return (1);
    }
    else
	return (0);

}


/** Function to clamp a float to limits *************************/
/* Function returns 1 if variable had to be clamped, 0 if not.    */

int     Limit_Value_f(x, xmin, xmax, error_message)
float  *x, xmin, xmax;
char   *error_message;
{

    if (*x < xmin) {
	*x = xmin;
	(void) printf("%s", error_message);
	return (1);
    }
    else if (*x > xmax) {
	*x = xmax;
	(void) printf("%s", error_message);
	return (1);
    }
    else
	return (0);

}


/** Function to clamp a double to limits *************************/
/* Function returns 1 if variable had to be clamped, 0 if not.    */

int     Limit_Value_d(x, xmin, xmax, error_message)
double *x, xmin, xmax;
char   *error_message;
{

    if (*x < xmin) {
	*x = xmin;
	(void) printf("%s", error_message);
	return (1);
    }
    else if (*x > xmax) {
	*x = xmax;
	(void) printf("%s", error_message);
	return (1);
    }
    else
	return (0);

}



/** Function to test whether an integer is out of limits **************/
/* Function returns 1 if out of range, 0 if not.    */

int     Out_Of_Range_i(x, xmin, xmax, message)
int     x, xmin, xmax;
char   *message;
{

    if (x < xmin || x > xmax) {
	(void) printf("value is %d, limits are: %d %d, %s \n", x, xmin, xmax, message);
	return (1);
    }
    else
	return (0);

}



/** Function to test whether a long integer is out of limits **************/
/* Function returns 1 if out of range, 0 if not.    */

int     Out_Of_Range_l(x, xmin, xmax, message)
long    x, xmin, xmax;
char   *message;
{

    if (x < xmin || x > xmax) {
	(void) printf("value is %ld, limits are: %ld %ld, %s \n", x, xmin, xmax, message);
	return (1);
    }
    else
	return (0);

}


/** Function to test whether a float is out of limits **************/
/* Function returns 1 if out of range, 0 if not.    */

int     Out_Of_Range_f(x, xmin, xmax, message)
float   x, xmin, xmax;
char   *message;
{

    if (x < xmin || x > xmax) {
	(void) printf("value is %g, limits are: %g %g, %s \n", x, xmin, xmax, message);
	return (1);
    }
    else
	return (0);

}

/** Function to test whether a double is out of limits **************/
/* Function returns 1 if out of range, 0 if not.    */

int     Out_Of_Range_d(x, xmin, xmax, message)
double  x, xmin, xmax;
char   *message;
{

    if (x < xmin || x > xmax) {
	(void) printf("value is %g, limits are: %g %g, %s \n", x, xmin, xmax, message);
	return (1);
    }
    else
	return (0);

}



/** Function to move integers from one array to another. Arrays may overlap */

int     Move_i(v1, i1, v2, i2, n)
int    *v1;
int     i1;
int    *v2;
int     i2, n;
{
    int    *pv1;
    int    *v3, *pv3;
    int     i;

    v3 = (int *) malloc(n * i2 * sizeof(int));	/* allocate buffer */
    if (v3 == NULL) {
	(void) printf(" Move failed to allocate buffer space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv3 = v3; i < n; i++) {
	*pv3 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv3 += i2;
    }

    (void) Movexi(v3, i2, v2, i2, n);	/* Copy contents of buffer into v2 */
    free((char *) v3);		/* ... and free the space          */

    return (OK);
}


/** Function to move floats from one array to another. Arrays may overlap */

int     Move_f(v1, i1, v2, i2, n)
float *v1;
int     i1;
float *v2;
int     i2, n;
{
    float *pv1;
    float *v3, *pv3;
    int     i;

    v3 = (float *) malloc(n * i2 * sizeof(float));	/* allocate buffer */
    if (v3 == NULL) {
	(void) printf(" Move failed to allocate buffer space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv3 = v3; i < n; i++) {
	*pv3 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv3 += i2;
    }

    (void) Movexf(v3, i2, v2, i2, n);	/* Copy contents of buffer into v2 */
    free((char *) v3);		/* ... and free the space          */

    return (OK);
}



/** Function to move doubles from one array to another. Arrays may overlap */

int     Move_d(v1, i1, v2, i2, n)
 double *v1;
 int     i1;
 double *v2;
 int	i2, n;
{
    double *pv1;
    double *v3, *pv3;
    int     i;

    v3 = (double *) malloc(n * i2 * sizeof(double));	/* allocate buffer */
    if (v3 == NULL) {
	(void) printf(" Move failed to allocate buffer space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv3 = v3; i < n; i++) {
	*pv3 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv3 += i2;
    }

    (void) Movexd(v3, i2, v2, i2, n);	/* Copy contents of buffer into v2 */
    free((char *) v3);		/* ... and free the space          */

    return (OK);
}


/** Function to move integers from one array to another ************/
/** Arrays v1 and v2 must not overlap                             */

int     Movexi(v1, i1, v2, i2, n)
int    *v1;
int     i1;
int    *v2;
int     i2, n;
{
    int    *pv1, *pv2;
    int     i;

    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv2 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv2 += i2;
    }

    return (OK);
}


/** Function to move floats from one array to another ************/
/** Arrays v1 and v2 must not overlap                             */

int     Movexf(v1, i1, v2, i2, n)
float *v1;
int     i1;
float *v2;
int     i2, n;
{
    float *pv1, *pv2;
    int     i;

    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv2 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv2 += i2;
    }

    return (OK);
}


/** Function to move doubles from one array to another ************/
/** Arrays v1 and v2 must not overlap                             */

int     Movexd(v1, i1, v2, i2, n)
double *v1;
int     i1;
double *v2;
int     i2, n;
{
    double *pv1, *pv2;
    int     i;

    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv2 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv2 += i2;
    }

    return (OK);
}


/** Vector Add Function  (for integers) ****************************/

int     Add_i(v1, i1, v2, i2, v3, i3, n)
int    *v1;
int     i1;
int    *v2;
int     i2;
int    *v3;
int    i3, n;
{
    int    *pv1, *pv2;
    int    *v4, *pv4;
    int     i;

    pv4 = v4 = (int *) malloc(n * i3 * sizeof(int));
    if (v4 == NULL) {
	(void) printf("Add failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = *pv1 + *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexi(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}

/** Vector Add Function  (for floats) ****************************/

int     Add_f(v1, i1, v2, i2, v3, i3, n)
float *v1;
int     i1;
float *v2;
int     i2;
float *v3;
int   i3, n;
{
    float *pv1, *pv2;
    float *v4, *pv4;
    int     i;

    pv4 = v4 = (float *) malloc(n * i3 * sizeof(float));
    if (v4 == NULL) {
	(void) printf("Add failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = *pv1 + *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexf(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}


/** Vector Add Function  (for doubles) ****************************/

int     Add_d(v1, i1, v2, i2, v3, i3, n)
double *v1;
int     i1;
double *v2;
int     i2;
double *v3;
int     i3, n;
{
    double *pv1, *pv2;
    double *v4, *pv4;
    int     i;

    pv4 = v4 = (double *) malloc(n * i3 * sizeof(double));
    if (v4 == NULL) {
	(void) printf("Add failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = *pv1 + *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexd(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}


/** Vector Subtract Function  (for integers) ****************************/

int     Sub_i(v1, i1, v2, i2, v3, i3, n)
int    *v1, i1, *v2, i2, *v3, i3, n;
{
    int    *pv1, *pv2;
    int    *v4, *pv4;
    int     i;

    pv4 = v4 = (int *) malloc(n * i3 * sizeof(int));
    if (v4 == NULL) {
	(void) printf("Sub failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = *pv1 - *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexi(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}


/** Vector Subtract Function  (for floats) ****************************/

int     Sub_f(v1, i1, v2, i2, v3, i3, n)
float *v1;
int     i1;
float *v2;
int     i2;
float *v3;
int     i3, n;
{
    float *pv1, *pv2;
    float *v4, *pv4;
    int     i;

    pv4 = v4 = (float *) malloc(n * i3 * sizeof(float));
    if (v4 == NULL) {
	(void) printf("Sub failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = *pv1 - *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexf(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}

/** Vector Subtract Function  (for doubles) ****************************/

int     Sub_d(v1, i1, v2, i2, v3, i3, n)
double *v1;
int     i1;
double *v2;
int     i2;
double *v3;
int     i3, n;
{
    double *pv1, *pv2;
    double *v4, *pv4;
    int     i;

    pv4 = v4 = (double *) malloc(n * i3 * sizeof(double));
    if (v4 == NULL) {
	(void) printf("Sub failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = *pv1 - *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexd(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}


/** Vector Pivot Function  (for integers) ****************************/

int     Piv_i(scal, v1, i1, v2, i2, v3, i3, n)
int     scal, *v1, i1, *v2, i2, *v3, i3, n;
{
    int    *pv1, *pv2;
    int    *v4, *pv4;
    int     i;

    pv4 = v4 = (int *) malloc(n * i3 * sizeof(int));
    if (v4 == NULL) {
	(void) printf("Piv failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = scal * *pv1 + *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexi(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}


/** Vector Pivot Function  (for floats) ****************************/

int     Piv_f(scal, v1, i1, v2, i2, v3, i3, n)
float   scal, *v1;
int     i1;
float   *v2;
int     i2;
float   *v3;
int     i3, n;
{
    float *pv1, *pv2;
    float *v4, *pv4;
    int     i;

    pv4 = v4 = (float *) malloc(n * i3 * sizeof(float));
    if (v4 == NULL) {
	(void) printf("Piv failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = scal * *pv1 + *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexf(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}


/** Vector Pivot Function  (for doubles) ****************************/

int     Piv_d(scal, v1, i1, v2, i2, v3, i3, n)
double   scal, *v1;
int     i1;
double *v2;
int     i2;
double *v3;
int     i3, n;
{
    double *pv1, *pv2;
    double *v4, *pv4;
    int     i;

    pv4 = v4 = (double *) malloc(n * i3 * sizeof(double));
    if (v4 == NULL) {
	(void) printf("Piv failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv4 = scal * *pv1 + *pv2;
	pv1 += i1;
	pv2 += i2;
	pv4 += i3;
    }
    (void) Movexd(v4, i3, v3, i3, n);
    free((char *) v4);

    return (OK);
}



/** Vector Scale Function  (for integers) ***************************/

int     Scal_i(scal, v1, i1, v2, i2, n)
int     scal, *v1, i1, *v2, i2, n;
{
    int    *pv1;
    int    *v3, *pv3;
    int     i;

    pv3 = v3 = (int *) malloc(n * i2 * sizeof(int));
    if (v3 == NULL) {
	(void) printf("Scal failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1; i < n; i++) {
	*pv3 = scal * *pv1;
	pv1 += i1;
	pv3 += i2;
    }
    (void) Movexi(v3, i2, v2, i2, n);
    free((char *) v3);

    return (OK);
}

/** Vector Scale Function  (for floats) ****************************/

int     Scal_f(scal, v1, i1, v2, i2, n)
float  scal, *v1;
int     i1;
float  *v2;
int     i2, n;
{
    float *pv1;
    float *v3, *pv3;
    int     i;

    pv3 = v3 = (float *) malloc(n * i2 * sizeof(float));
    if (v3 == NULL) {
	(void) printf("Scal failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1; i < n; i++) {
	*pv3 = scal * *pv1;
	pv1 += i1;
	pv3 += i2;
    }
    (void) Movexf(v3, i2, v2, i2, n);
    free((char *) v3);

    return (OK);
}


/** Vector Scale Function  (for doubles) ****************************/

int     Scal_d(scal, v1, i1, v2, i2, n)
double scal, *v1; 
int     i1;
double *v2;
int     i2, n;
{
    double *pv1;
    double *v3, *pv3;
    int     i;

    pv3 = v3 = (double *) malloc(n * i2 * sizeof(double));
    if (v3 == NULL) {
	(void) printf("Scal failed to allocate space \n");
	return (ERROR);
    }
    for (i = 0, pv1 = v1; i < n; i++) {
	*pv3 = scal * *pv1;
	pv1 += i1;
	pv3 += i2;
    }
    (void) Movexd(v3, i2, v2, i2, n);
    free((char *) v3);

    return (OK);
}


/**  Dot Product Function  (for integers)  **************************/

int     Dot_i(v1, i1, v2, i2, n)
int    *v1, i1, *v2, i2, n;
{
    int    *pv1, *pv2, dotp;
    int     i;

    dotp = 0;
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	dotp += *pv1 * *pv2;
	pv1 += i1;
	pv2 += i2;
    }

    return (dotp);
}


/**  Dot Product Function  (for floats)  **************************/

float  Dot_f(v1, i1, v2, i2, n)
float *v1;
int     i1;
float *v2;
int     i2, n;
{
    float *pv1, *pv2, dotp;
    int     i;

    dotp = 0;
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	dotp += *pv1 * *pv2;
	pv1 += i1;
	pv2 += i2;
    }

    return (dotp);
}

/**  Dot Product Function  (for doubles)  **************************/

double  Dot_d(v1, i1, v2, i2, n)
double *v1;
int     i1;
double *v2;
int     i2, n;
{
    double *pv1, *pv2, dotp;
    int     i;

    dotp = 0;
    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	dotp += *pv1 * *pv2;
	pv1 += i1;
	pv2 += i2;
    }

    return (dotp);
}


/**  Cross Product Function  (for integers  **************************/

int     Crosp_i(v1, v2, v3)
int     v1[], v2[], v3[];
{
    int     v4[3];

    v4[0] = v1[1] * v2[2] - v1[2] * v2[1];
    v4[1] = -v1[0] * v2[2] + v1[2] * v2[0];
    v4[2] = v1[0] * v2[1] - v1[1] * v2[0];
    (void) Movexi(v4, 1, v3, 1, 3);
    return (OK);
}


/**  Cross Product Function  (for floats)  **************************/

int     Crosp_f(v1, v2, v3)
float  v1[], v2[], v3[];
{
    float  v4[3];

    v4[0] = v1[1] * v2[2] - v1[2] * v2[1];
    v4[1] = -v1[0] * v2[2] + v1[2] * v2[0];
    v4[2] = v1[0] * v2[1] - v1[1] * v2[0];
    (void) Movexf(v4, 1, v3, 1, 3);
    return (OK);
}


/**  Cross Product Function  (for doubles)  **************************/

int     Crosp_d(v1, v2, v3)
double  v1[], v2[], v3[];
{
    double  v4[3];

    v4[0] = v1[1] * v2[2] - v1[2] * v2[1];
    v4[1] = -v1[0] * v2[2] + v1[2] * v2[0];
    v4[2] = v1[0] * v2[1] - v1[1] * v2[0];
    (void) Movexd(v4, 1, v3, 1, 3);
    return (OK);
}


/** Function to normalize an array of floats, function returns magnitude *****/

float   Norm_f(array, n)
float   array[];
int     n;
{
    int     i;
    double  magnitude = 0.0;

    for (i = 0; i < n; i++)
	magnitude += array[i] * array[i];

    if (magnitude == 0) {
	(void) printf("Magnitude of vector is 0 \n");
    }
    else {
	magnitude = sqrt(magnitude);
	for (i = 0; i < n; i++)
	    array[i] = array[i] / magnitude;
    }

    return (magnitude);
}


/** Function to normalize an array of doubles, function returns magnitude *****/

double  Norm_d(array, n)
double  array[];
int     n;
{
    int     i;
    double  magnitude = 0.0;

    for (i = 0; i < n; i++)
	magnitude += array[i] * array[i];

    if (magnitude == 0) {
	(void) printf("Magnitude of vector is 0 \n");
    }
    else {
	magnitude = sqrt(magnitude);
	for (i = 0; i < n; i++)
	    array[i] = array[i] / magnitude;
    }

    return (magnitude);
}

/** Function to find the magnitude of an array of floats ***************/

float   Mag_f(array, n)
float   array[];
int     n;
{
    int     i;
    double  magnitude = 0.0;

    for (i = 0; i < n; i++)
	magnitude += array[i] * array[i];

    return ((float) sqrt(magnitude));
}


/** Function to find the magnitude of an array of doubles ***************/

double  Mag_d(array, n)
double  array[];
int     n;
{
    int     i;
    double  magnitude = 0.0;

    for (i = 0; i < n; i++)
	magnitude += array[i] * array[i];

    return (sqrt(magnitude));
}

#ifdef ANSI
/** Function to set elements in an array of integers */

void    Set_Int_Array(array, n, value, ...)
int    *array, n, value;
{
    int     i;
    va_list arg_marker;

    va_start(arg_marker, value);

    for (i = 1, array[0] = value; i < n; i++) {
	array[i] = va_arg(arg_marker, int);
    }
    va_end(arg_marker);
    return;
}

/** Function to set elements in an array of doubles */

void    Set_Double_Array(array, n, value, ...)
double  value;
double *array;
int     n;
{
    int     i;
    va_list arg_marker;

    va_start(arg_marker, value);

    for (i = 1, array[0] = value; i < n; array[i] = va_arg(arg_marker, double), i++);

    return;
}
#endif

/** Copy array of ints to doubles */

void    Copy_Int_To_Double(v1, i1, v2, i2, n)
int    *v1, i1;
double *v2;
int    i2, n;
{
    int    *pv1;
    double *pv2;
    int     i;

    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {
	*pv2 = *pv1;
	pv1 += i1;		/* increment pointers */
	pv2 += i2;
    }
    return;
}

/** Copy array of doubles to ints */

void    Copy_Double_To_Int(v1, i1, v2, i2, n)
double *v1;
int     i1, *v2, i2, n;
{
    double *pv1;
    int    *pv2;
    int     i;

    for (i = 0, pv1 = v1, pv2 = v2; i < n; i++) {

	if (*pv1 > INT_MAX)
	    *pv2 = INT_MAX;
	else if (*pv1 < INT_MIN)
	    *pv2 = INT_MIN;
	else
	    *pv2 = *pv1;

	pv1 += i1;		/* increment pointers */
	pv2 += i2;
    }
    return;
}




/*******************************************************************

Matrix operation functions

Peter Kovesi             November 1986

********************************************************************/


/** Matrix multiplication  (for integers) **************************/
/* C = A * B                                                       */

int     Mul_i(l, m, n, A, B, C)
int     l;			/* rows in A    */
int     m;			/* columns in A */
int     n;			/* columns in B */
int    *A, *B, *C;
{
    int    *Aptr, *Bptr, *D, *Dptr;
    int     row, col;

    D = (int *) malloc(l * n * sizeof(int));	/* buffer */

    for (row = 0; row < l; row++) {
	Aptr = A + row * m;
	Bptr = B;
	Dptr = D + row * n;

	for (col = 0; col < n; col++) {
	    *Dptr = Dot_i(Aptr, 1, Bptr, n, m);
	    Dptr++;
	    Bptr++;
	}
    }

    (void) Movexi(D, 1, C, 1, l * n);	/* copy buffer over to C */
    free((char *) D);

    return (OK);
}


/** Matrix multiplication  (for doubles) **************************/
/* C = A * B                                                      */

int     Mul_d(l, m, n, A, B, C)
int     l;			/* rows in A    */
int     m;			/* columns in A */
int     n;			/* columns in B */
double *A, *B, *C;
{
    double *Aptr, *Bptr, *D, *Dptr;
    int     row, col;

    D = (double *) malloc(l * n * sizeof(double));	/* buffer */

    for (row = 0; row < l; row++) {
	Aptr = A + row * m;
	Bptr = B;
	Dptr = D + row * n;

	for (col = 0; col < n; col++) {
	    *Dptr = Dot_d(Aptr, 1, Bptr, n, m);
	    Dptr++;
	    Bptr++;
	}
    }

    (void) Movexd(D, 1, C, 1, l * n);	/* copy buffer over to C */
    free((char *) D);

    return (OK);
}


/** Matrix Transpose  (for integers) *******************************/

int     Transpose_i(l, m, A, AT)
int     l;			/* rows in A    */
int     m;			/* columns in A */
int    *A, *AT;
{
    int     row, col;
    int    *Aptr, *B, *Bptr;

    B = (int *) malloc(l * m * sizeof(int));	/* buffer */

    for (row = 0; row < l; row++) {
	Aptr = A + row * m;
	Bptr = B + row;

	for (col = 0; col < m; col++) {
	    *Bptr = *Aptr;
	    Aptr++;
	    Bptr += l;
	}
    }

    (void) Movexi(B, 1, AT, 1, l * m);	/* copy buffer to AT */
    free((char *) B);

    return (OK);
}


/** Matrix Transpose  (for doubles) *******************************/

int     Transpose_d(l, m, A, AT)
int     l;			/* rows in A    */
int     m;			/* columns in A */
double *A, *AT;
{
    int     row, col;
    double *Aptr, *B, *Bptr;

    B = (double *) malloc(l * m * sizeof(double));	/* buffer */

    for (row = 0; row < l; row++) {
	Aptr = A + row * m;
	Bptr = B + row;

	for (col = 0; col < m; col++) {
	    *Bptr = *Aptr;
	    Aptr++;
	    Bptr += l;
	}
    }

    (void) Movexd(B, 1, AT, 1, l * m);	/* copy buffer to AT */
    free((char *) B);

    return (OK);
}


/**  Homogeneous transformation function  **************************/

int     Homog_Trans(A, b, c)
double  A[4][4], b[3], c[3];
{
    double  bh[4], ch[4];
    int     i, j;

    (void) Movexd(b, 1, bh, 1, 3);	/* create 4-vector */
    bh[3] = 1;

    for (i = 0; i < 4; i++) {
	for (ch[i] = 0, j = 0; j < 4; j++) {
	    ch[i] += A[i][j] * bh[j];
	}
    }

    (void) Scal_d(1.0 / ch[3], ch, 1, c, 1, 3);	/* normalise vector */

    return (OK);
}



/** Gaussian elimination routine (for doubles) *********************/

/*
 * Solves x  for  Ax = b Partial pivotal condensation is used
 *
 * Code from Dorn and McCracken "Numerical Methods in Fortran IV" Translated
 * into C by PK
 */

int     Gauss(A, x, b, n)
double  A[], x[], b[];
int     n;
{
    int     row, col, i, j, L;
    int     result;
    double *B;			/* buffer */
    double  m, sum;

    B = (double *) malloc(n * (n + 1) * sizeof(double));
    (void) Movexd(A, 1, B, 1, n * n);	/* copy A to B (A will get destroyed  */
    /* by the elimination process)        */

    for (row = 0; row < n - 1; row++) {

	/*
	 * find term in column = row and below main diagonal that is largest
	 * in absolute value. L is the row No having this largest element
	 */
	for (i = row + 1, L = row; i < n; i++) {
	    if (fabs(A[n * i + row]) > fabs(A[n * L + row]))
		L = i;
	}

	if (L != row) {		/* we need to swap rows */
	    m = b[row];
	    b[row] = b[L];
	    b[L] = m;

	    for (j = row; j < n; j++) {
		m = A[n * row + j];
		A[n * row + j] = A[n * L + j];
		A[n * L + j] = m;
	    }
	}
	if (A[n * row + row] == 0) {
	    (void) printf("Determinant of matrix is 0 \n");
	    result = ERROR;
	    goto END;
	}

	/*
	 * Eliminate elements in column = row+1 below main diagonal. In fact
	 * elements are not elimated but ignored in future
	 */

	for (i = row + 1; i < n; i++) {
	    m = A[n * i + row] / A[n * row + row];
	    b[i] = b[i] - m * b[row];

	    for (j = row + 1; j < n; j++)
		A[n * i + j] = A[n * i + j] - m * A[n * row + j];

	}

    }				/* for each row */

    /* Now back substitute */

    x[n - 1] = b[n - 1] / A[n * (n - 1) + (n - 1)];

    for (row = n - 2; row >= 0; row--) {
	for (col = row + 1, sum = 0; col < n; col++)
	    sum = sum + A[n * row + col] * x[col];

	x[row] = (b[row] - sum) / A[n * row + row];
    }

    (void) Movexd(B, 1, A, 1, n * n);	/* copy back saved version of A */
    result = OK;

END:
    free((char *) B);
    return (result);
}



/** Gaussian elimination routine (for doubles) *********************/

/*
 * With Condition Number Calculation
 *
 * Solves x  for  Ax = b Where x and b are m x n matricies Partial pivotal
 * condensation is used
 *
 * Code from Dorn and McCracken "Numerical Methods in Fortran IV" Translated
 * into C by PK
 */

int     GaussX(Abuf, x, b, m, n)
int     m, n;
double  Abuf[], x[], b[];
{
    int     row, col, i, j, L, mn;
    double *A;			/* buffer */
    double *mul, temp, sum;

    mn = m + n;
    A = (double *) malloc(m * mn * sizeof(double));
    mul = (double *) malloc(m * sizeof(double));

    for (i = 0; i < m; i++) {
	(void) Movexd(&Abuf[i * m], 1, &A[i * mn], 1, m);	/* copy Abuf abd b to A   */
	(void) Movexd(&b[i * n], 1, &A[i * mn + m], 1, n);
    }

    for (row = 0; row < m - 1; row++) {	/* for every row */

	/*
	 * find term in column = row and below main diagonal that is largest
	 * in absolute value. L is the row No having this largest element
	 */
	for (i = row + 1, L = row; i < m; i++) {
	    if (fabs(A[mn * i + row]) > fabs(A[mn * L + row]))
		L = i;
	}

	if (L != row) {		/* we need to swap rows. Only elements */
	    for (j = row; j < mn; j++) {	/* from column = row onwards
						 * need be swapped */
		temp = A[mn * row + j];	/* as other elements eliminated  */
		A[mn * row + j] = A[mn * L + j];
		A[mn * L + j] = temp;
	    }
	}
	if (A[mn * row + row] == 0) {
	    (void) printf("Determinant of matrix is 0 \n");
	    return (ERROR);
	}

	/*
	 * Eliminate elements in column = row+1 below main diagonal. In fact
	 * elements are not elimated but ignored in future
	 */

	for (i = row + 1; i < m; i++) {
	    mul[i] = A[mn * i + row] / A[mn * row + row];
	    for (j = row + 1; j < mn; j++)
		A[mn * i + j] = A[mn * i + j] - mul[i] * A[mn * row + j];
	}

    }				/* for each row */

    /* Now back substitute */

    for (j = 0; j < n; j++) {

	x[(m - 1) * n + j] = A[(m - 1) * mn + m + j] / A[mn * (m - 1) + (m - 1)];

	for (row = m - 2; row >= 0; row--) {
	    for (col = row + 1, sum = 0; col < m; col++)
		sum = sum + A[mn * row + col] * x[n * col + j];

	    x[n * row + j] = (A[mn * row + m + j] - sum) / A[mn * row + row];
	}
    }


    free((char *) A);
    free((char *) mul);
    return (OK);

}

/** Gaussian elimination routine (for doubles) *********************/

/*
 * With Condition Number Calculation
 *
 * Solves x  for  Ax = b Where x and b are m x n matricies Partial pivotal
 * condensation is used
 *
 * Code from Dorn and McCracken "Numerical Methods in Fortran IV" Translated
 * into C by PK
 */

double  Gauss_Cond(Abuf, x, b, m, n)
int     m, n;
double  Abuf[], x[], b[];
{
    int     row, col, i;
    double *Bb, *Ainv;
    double  Nsum, NIsum;
    double  Condition, ANorm, AinvNorm;


    (void) GaussX(Abuf, x, b, m, n);	/* Solve equations */

    Bb = (double *) malloc(m * m * sizeof(double));
    Ainv = (double *) malloc(m * m * sizeof(double));

    /* set up identity matrix */
    for (i = 0; i < m * m; i++)
	Bb[i] = 0;

    for (i = 0; i < m * m; i = i + m + 1)
	Bb[i] = 1.0;

    (void) GaussX(Abuf, Ainv, Bb, m, m);	/* get inverse */

    /*
     * printf("A matrix \n"); Print_Matrix_d(Abuf,m,m); printf("\n Ainv \n");
     * Print_Matrix_d(Ainv,m,m);
     */


    /*
     * Now evaluate condition No., the 1-Norm is used - column with greatest
     * sum of absolute values
     */
    ANorm = 0.;
    AinvNorm = 0.;

    for (col = 0; col < m; col++) {	/* for every column... */
	for (row = 0, Nsum = 0.0, NIsum = 0.0; row < m; row++) {
	    Nsum += fabs(Abuf[m * row + col]);
	    NIsum += fabs(Ainv[m * row + col]);
	}
	if (Nsum > ANorm)
	    ANorm = Nsum;	/* ANorm is largest column sum */
	if (NIsum > AinvNorm)
	    AinvNorm = NIsum;

    }				/* For every column */


    (void) printf(" |A| = %f   |Ainv| = %f \n", ANorm, AinvNorm);
    Condition = ANorm * AinvNorm;

    /*
     * Mul_d(m,m,m,Abuf, Ainv, Ainv); Print_Matrix_d(Ainv,m,m);
     */

    free((char *) Bb);
    free((char *) Ainv);
    return (Condition);
}



/** Matrix Printing function (for doubles)  ********************************/

void    Print_Matrix_d(M, rows, cols)
double *M;
int     rows, cols;
{
    int     i, j;
    double *Mptr;

    if (M == NULL) {
	(void) printf(" Print_Matrix error - NULL matrix address \n");
	return;
    }
    Mptr = M;
    (void) printf("\n");

    for (i = 0; i < rows; i++) {
	for (j = 0; j < cols; j++)
	    (void) printf(" %7.3f ", *Mptr++);

	(void) printf("\n");
    }

    return;
}


/** Integer array allocation function  *************************************/

int    *Allocate_IArray(n)
int     n;
{
    int    *a;

    if (!(a = (int *) malloc(sizeof(int) * n)))
	return (NULL);
    return (a);
}



/** Long array allocation function  ****************************************/

long   *Allocate_LArray(n)
int     n;
{
    long   *a;

    if (!(a = (long *) malloc(sizeof(long) * n)))
	return (NULL);
    return (a);
}



/** Float array allocation function  ***************************************/

float  *Allocate_FArray(n)
int     n;
{
    float  *a;

    if (!(a = (float *) malloc(sizeof(float) * n)))
	return (NULL);
    return (a);
}



/** Double array allocation function  **************************************/

double *Allocate_DArray(n)
int     n;
{
    double *a;

    if (!(a = (double *) malloc(sizeof(double) * n)))
	return (NULL);
    return (a);
}


/*-  Allocate_Matrix  -------------------------------------------------

Function to allocate memory for a matrix.

This function dynamically allocates space for a 2D matrix in such a
way that one can address individual elements in the usual way.  It
does this by allocating a contiguous section of memory for the
complete 2D array and then allocating an array of pointers of size
rows, each element being assigned the address of the element in the
main array corresponding to the start of each row.  This means that
you should NOT pass the name of the allocated matrix as an address of
the matrix to functions as you will only be passing the address of the
array of pointers to the rows.  You should use matrix[0] or
&matrix[0][0] as in the example below.

Example of use:

double **matrix;
double one = 1.0, three = 3.0;

matrix = (double **)Allocate_Matrix(3, 6, sizeof(double));

Move_d(&one,0,&matrix[0][0],6,3)   - set 1st column to 1's 
Move_d(&three,0,&matrix[0][2],6,3)  - set 3rd column to 3's 
matrix[0][3] = 1.0;   -  etc ...

--------------------------------------------------------------------*/

char   **Allocate_Matrix(rows, cols, size)
int      rows, cols, size;
{
    char   **pptr, *ptr;
    int      i;

    pptr = (char **) malloc(rows * sizeof(char *));
    ptr = (char *) calloc(rows * cols, size);

    if ((!pptr) OR (!ptr)) {
		(void) fprintf(stderr,"Allocate_Matrix: Could not allocate matrix\n");
		return (NULL);
    }

    for (i = 0; i < rows; i++)
		pptr[i] = (char *) &ptr[i * cols * size];

    return (pptr);
}


/*- Free_Matrix  -----------------------------------------------------

Function to free memory allocated for a matrix.

--------------------------------------------------------------------*/

void      Free_Matrix(matrix)
char   **matrix;
{
    if (matrix == NULL)
		return;		/* it was already free */

    free((char *) matrix[0]);
    free((char *) matrix);

    return;
}

