/**********************************************************************/
/* struct.c                                                           */
/*                                                                    */
/* Routines to print out structures, and writing final scene.         */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <string.h>
#include "geo.h"
#include "misc.h"
#include "io.h"
#include "struct.h"
#include "rad.h"
#include "adj.h"

extern Vertex_tree *vtree;
extern int vtree_size;
extern RadParams ReadLog;
extern void BoundsPrint();
extern void Poly_VertexB();
extern OptionType Option;

#define ON 1
#define OFF 0
FILE *logf;                        /* File for logging */
char *logfilename = "out.scene";
FILE *oscenef;                     /* File for scene output */
char *oscenefilename = "out.scene";

/**********************************************************************/
/* Print polygon and it's children                                    */
/**********************************************************************/
void print_Polygon(fp,label,p)
     Polygon *p;
     FILE *fp;
     char *label;
{
  int i;

  fprintf(fp,"\tPolygon %s%d {\n", p->name, p->id);
  fprintf(fp,"\ttype: %s; area: %g", 
	  (p->class == TRIANGLE ? "tri" : "quad"),
	  p->area);
  fprintf(fp," level %d; chgnrm %d; number vertices %d, plane d %g\n",
	  p->level, p->changingNormal, p->numVert, p->d);
  if (p->uvw != 0) {
    fprintf(fp,"\tU(N=%g,%g,%g d=%g); V(N=%g,%g,%g d=%g);\n",
	    p->uvw->Nu.x, p->uvw->Nu.y,  p->uvw->Nu.z, p->uvw->du, 
	    p->uvw->Nv.x, p->uvw->Nv.y,  p->uvw->Nv.z, p->uvw->dv);
    if (p->class == TRIANGLE) 
      fprintf(fp,"\tW(N=%g,%g,%g d=%g)\n",
	      p->uvw->Nw.x, p->uvw->Nw.y,  p->uvw->Nw.z, p->uvw->dw);
  }
  fprintf(fp,"\tunshot B: ");
  for(i=0;i<MAX_SPECTRA_SAMPLES;i++) 
    fprintf(fp,"%g ", p->unshot_B.samples[i]);
  fprintf(fp," total B: ");
  for(i=0;i<MAX_SPECTRA_SAMPLES;i++) 
    fprintf(fp,"%g ", p->B.samples[i]);
  fprintf(fp,"\n");

  if (p->Mfather != 0)
    fprintf(fp,"\tParent mesh: %s%d\n", p->Mfather->name, p->Mfather->id);
  if (p->Pfather != 0)
    fprintf(fp,"\tParent patch: %s\n", p->Pfather->name);
  if (p->Links != 0) {
    print_PolyList(fp, "of FFlinks", p->Links, 
		   p->polyhead.num_polys, 0);
  }

  for (i=0;i<(p->numVert);i++) {
    print_vector(fp, p->vert[i]->name, &(p->vert[i]->pos));
    print_vector(fp, "Normal", &(p->normal[i]));
  }
  
  for (i=0;i<MAX_PATCH_CHILDREN;i++)
    if(p->child[i] != 0) {
      fprintf(fp, "\tChild #%d\n", i);
      print_Polygon(fp, "", p->child[i]);
    }
  fprintf(fp,"\t}\n");
}

/*********************************************************************/
/* Print list of polygons                                            */
/*********************************************************************/
void print_PolyList(fp, label, pl,plsize,full)
     FILE *fp;
     char *label;
     PolyList *pl;
     int plsize;
     int full;
{
  PolyList *plptr;
  int i;
  
  fprintf(fp, "\tPolylist %s {\n", label);
  for(i=0, plptr= pl;i<plsize;i++, plptr=plptr->next) {
    if(full) {
      fprintf(fp,"\n");
      print_Polygon(fp, "", plptr->patch);
    } else 
      fprintf(fp,"\tPolygon %s%d\n", plptr->patch->name, plptr->patch->id);
  }
  fprintf(fp,"\t}\n");
}

/**********************************************************************/
/* Print mesh                                                         */
/**********************************************************************/
void print_Mesh(fp, label,m)
     Mesh *m;
     FILE *fp;
     char *label;
{
  Polygon *pptr;
  int i;

  fprintf(fp,"\n\tMesh %s%d; %d polygon(s); Object: %s%d\n", 
	  m->name, m->id, m->num_polys, m->Ofather->name,
	  m->Ofather->id);
  if (m->box != 0)
    BoundsPrint(*(m->box), fp, m->name);
  for (i=0, pptr = m->polys; i<m->num_polys; i++, pptr = pptr->next) {
    print_Polygon(fp,"",pptr);
  }
}

/**********************************************************************/
/* Print vertex                                                       */
/**********************************************************************/
void print_Vertex(fp,id,v)
     FILE *fp;
     int id;
     Vertex *v;
{
  fprintf(fp,"\tVertex%d {\n", id);
  print_vector(fp, "Position", &(v->pos));
  
  fprintf(fp,"\t%d adjacent polygons\n", v->polyhead.num_polys);
  print_PolyList(fp, "of adjacent polyons", 
		 v->polylist, v->polyhead.num_polys,0);
  fprintf(fp,"\t}\n");
}

int vtx_count = 0;                    /* For labelling vertices */
/**********************************************************************/
/* Print tree of vertices                                             */
/**********************************************************************/
void print_Vertex_Tree(fp, label, vtptr)
     FILE *fp;
     char *label;
     Vertex_tree *vtptr;
{
  int octant;

  if (vtptr != 0) { 
    for (octant=0;octant<num_octants;octant++) 
      if (vtptr->child[octant] != 0) 
	print_Vertex_Tree(fp, label, vtptr->child[octant]);
    print_Vertex(fp,vtx_count,vtptr->vtx);
    vtx_count++;
  }
}
    
/**********************************************************************/
/* Print a colour                                                     */
/**********************************************************************/
void print_Colour(fp, label, c)
     FILE *fp;
     char *label;
     Colour *c;
{
  fprintf(fp,"\tColour %s: %g %g %g\n", label, c->r, c->g, c->b);
}

/**********************************************************************/
/* Print spectrum                                                     */
/**********************************************************************/
void print_Spectra(fp, label, s)
     FILE *fp;
     char *label;
     Spectra s;
{
  int i;
  fprintf(fp,"\t%s { ", label);
  for (i=0;i<MAX_SPECTRA_SAMPLES;i++) fprintf(fp, "%g ",s.samples[i]);
  fprintf(fp,"}\n");
}

/**********************************************************************/
/* Print shade properties */
/**********************************************************************/
void print_ShadeType(fp, label, st)
     FILE *fp;
     char *label;
     ShadeType st;
{
  fprintf(fp,"\tShadetype %s\n", label);
  print_Spectra(fp,"Ka",st.Ka);
  print_Spectra(fp,"Kd",st.Kd);
  print_Spectra(fp,"Ks",st.Ks);
  print_Spectra(fp,"p",st.p);
  print_Spectra(fp,"t",st.t);
  print_Spectra(fp,"n",st.n);
}

/**********************************************************************/
/* Print energy                                                       */
/**********************************************************************/
void print_Radiance(fp, label, r)
     FILE *fp;
     char *label;
     Radiance r;
{
  fprintf(fp,"\tRadiance %s\n", label);
  print_Spectra(fp,"E",r.E);
  print_Spectra(fp,"B",r.B);
}

/**********************************************************************/
/* Print surface properties                                           */
/**********************************************************************/
void print_SurfaceProp(fp, label, s)
     FILE *fp;
     char *label;
     SurfaceProp *s;
{
  fprintf(fp,"\tSurfaceProp%d %s\n", s->id, s->name);
  print_ShadeType(fp,label,s->shade);
  /* print_Radiance(fp,label,s->rad); */
}

/**********************************************************************/
/* Print a texture map                                                */
/**********************************************************************/
void print_TextureMap(fp, label, t)
     FILE *fp;
     char *label;
     TextureProp *t;
{
  int i,j;

  fprintf(fp,"\tTextureMap%d %s\n", t->id, t->name);
  fprintf(fp,"\tclass %d; resolution %d\n", t->class, t->resolution);
  for(i=0;i<t->resolution;i++) {
    fprintf(fp,"\t");
    for(j=0;j<t->resolution;j++) {
      fprintf(fp,"%u ", *t->map);
      t->map++;
    }
    fprintf(fp,"\n");
  }
}

/**********************************************************************/
/* Print an object                                                    */
/**********************************************************************/
void print_Object(fp, label, n)
     FILE *fp;
     char *label;
     Objectt *n;
{
  fprintf(fp,"\tObject%d %s\n", n->id, n->name);
  fprintf(fp,"\trayID %d\n", n->rayID);
  fprintf(fp,"\tprimtype[%d] =  %s\n", n->primid, n->primtype);
  BoundsPrint(*(n->box), fp, n->name);
  /* print_SurfaceProp(fp, "", n->surface); */
}

/**********************************************************************/
/* Switch log on / off                                                */
/**********************************************************************/
void Log(logflag,what)
     int logflag;
     char *what;
{
  if (logflag) {
    if (!(logf = fopen(logfilename, "w"))) {
      fprintf(stderr,"%s: cannot open log file %s\n", ProgName, logfilename);
      exit(1);
    } 
    printf("\n\t*** Print log of %s to %s ***\n", what, logfilename);
  } else 
    fclose(logf);
}


/**********************************************************************/
/* Print viewing position */
/**********************************************************************/
void print_View(fp,c)
     FILE *fp;
     Camera *c;
{
  fprintf(fp,"\tCamera {\n");

  fprintf(fp,"\t\tlookfrom %g %g %g\n",
	  c->lookfrom.x, c->lookfrom.y, c->lookfrom.z);
  fprintf(fp,"\t\tlookat %g %g %g\n", 
	  c->lookat.x,c->lookat.y,c->lookat.z);
  fprintf(fp,"\t\tlookup %g %g %g\n",
	  c->lookup.x,c->lookup.y, c->lookup.z);
  fprintf(fp,"\t\tfovx %d fovy %d near %g far %g\n",
	  c->fovx, c->fovy,c->near, c->far);
  fprintf(fp,"\t\txRes %d yRes %d\n", c->xRes, c->yRes);
  fprintf(fp,"\t}");
}

/**********************************************************************/
/* Print the scene out to log file */
/**********************************************************************/
void print_Scene(s, fname)
     Scene *s;
     char *fname;
{
  int i,j;
  Objectt *optr;
  Mesh *mptr;

  /* Print all polygons, and vertices */
  logfilename = fname;

  Log(ON,"scene");
  optr = s->objects;
  for(i=0;i<s->num_objects;i++,optr++) {
    mptr = optr->meshes;
    print_Object(logf,"",optr);
    for (j=0;j<optr->num_meshes;j++, mptr++) 
      print_Mesh(logf,"blah",mptr);
  }

  vtx_count = 0;
  fprintf(logf,"\n\tVertexTree size: %d {\n", vtree_size);
  print_Vertex_Tree(logf, "", vtree);
  Log(OFF,"");

  /* Print display view */
  /* logfilename = "log.view";
     Log(ON,"view");
     print_View(logf,&(ReadLog.displayView));
     Log(OFF,""); */
}

/**********************************************************************/
/* Output results after running radiosity on scene                    */
/**********************************************************************/
void Write_Rad(rparams, filename, num)
     RadParams rparams;
     char *filename;
     int num;
{
  int i,j,k;
  Elist *elptr;
  Polygon *eptr;
  FILE *fp;
  char ffilename[80];
  Colour c[MAX_PATCH_VTX];

  if (num > 0) 
    sprintf(ffilename,"%s%d", filename, num);
  else
    sprintf(ffilename,"%s", filename);
  if (!(fp = fopen(ffilename, "w"))) {
    fprintf(stderr,"%s: cannot open results file %s\n", ProgName, ffilename);
    exit(1);
  } 
  printf("\n\t*** Printing %d polygons to %s ***\n", 
	 rparams.num_elements, ffilename);
  
  /* Print elements to file */
  fprintf(fp,"Number polygons %d\n", rparams.num_elements);
  elptr = rparams.elements;
  for(i=0; i<rparams.num_elements;i++, elptr = elptr->next) {
    eptr = elptr->element;

    /* Print patch label */
    fprintf(fp,"Poly %s%d %d {\n", "poly", i, 
	    eptr->numVert);

    /* Print colour of patch (flat shade it for now) */
    fprintf(fp,"  B { ");
    if (Option.rad_interp_type == INTERP_VTX_FROM_PATCH) {
      for (k=0;k<MAX_SPECTRA_SAMPLES;k++) 
	fprintf(fp,"%g ", 
		(eptr->B.samples[k] > 1.0 ? 1.0 : eptr->B.samples[k]));
    } else {
      for (k=0;k<MAX_SPECTRA_SAMPLES;k++) fprintf(fp,"-1.0 ");
    }      
    fprintf(fp,"}\n");

    /* Find and print colour of vertices */
    Poly_VertexB(eptr, c);
    fprintf(fp,"  vtx_B { ");
    for (j=0;j<eptr->numVert;j++)
      fprintf(fp,"{ %g %g %g } ", c[j].r, c[j].g, c[j].b);
    fprintf(fp,"}\n");
    
    /* Print normals */
    fprintf(fp,"  norm {",i);
    for (j=0;j<eptr->numVert;j++) 
      fprintf(fp," { %g %g %g } ",
	     eptr->normal[j].x, eptr->normal[j].y, eptr->normal[j].z);
    fprintf(fp,"}\n");
    
    /* Print vertices */
    fprintf(fp,"  vert {",i);
    for (j=0;j<eptr->numVert;j++) 
      fprintf(fp," { %g %g %g } ", eptr->vert[j]->pos.x, 
	      eptr->vert[j]->pos.y, eptr->vert[j]->pos.z);
    fprintf(fp,"}\n");

    fprintf(fp,"}\n");
  }

  fclose(fp);
}
