/* FILE: fileio.c

   Routines for loading files in pnm formats and saving files in RAWBITS pbm
   format. These routines are pretty 'klugde' code. No guarantee that they
   will work (but they seem to work on our system :-)).

   Written by Petri Kuittinen, last modifications 19th August 1993.
*/ 

#include <stdlib.h>
#include <stdio.h>
#include "pixel.h"
#include "block.h"
#include "misc.h"
#include "fileio.h"

/* Now, close your eyes! Don't look at this ugly code. */

#define IS_NUMBER(C) ((C)>='0' && (C)<='9')
#define SKIP_NUMBERS while (IS_NUMBER(*s)) s++; /* Skip ASCII numbers */
#define MAX_LINE_LENGTH 80 /* PPM format guarantees that no line should be
			      longer than 70 characters. 80 is defined here
			      just for safety */
#define READ_LINE do {s = fgets(line, MAX_LINE_LENGTH-1, fp); if (s==NULL) \
			return NULL;} while (*s=='#'); /* Read new line,
							  skip lines which
							  start with '#' */
#define IS_WHITE_SPACE(C) ((C)==' ' || (C)=='\t' || (C)=='\v')
#define IS_LF(C) ((C)=='\n' || (C)=='\r')
#define IS_COMMENT(C) ((C)=='#')


/* FUNCTION:     struct picture *load(char *name, FILE *fp)
   DESCRIPTION:  Load .pnm picture to internal data structure and allocate
   memory for it. File needs to be opened before calling this.
   PARAMETERS:
   char *name    name of the picture (complete with path)
   FILE *fp      pointer to FILE structure 
   RETURNS:      pointer to picture struct, returns NULL if something fails.
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
struct picture *load(char *name, FILE *fp)
{
  char line[MAX_LINE_LENGTH], *s;
  int type, width, height, maxval;
  struct picture *pic_p;
  register int wc, data, i, val = 0;
  register pixel *buf;

  READ_LINE;

  /* Match file 'magic number' */
  if (*s++!='P') return NULL;
  type = *s++-'0';
  if (type<0 || type >6) return NULL;
  
  if (IS_LF(*s)) {READ_LINE;}
  else if (IS_WHITE_SPACE(*s)) s++; else return NULL;
  if (IS_COMMENT(*s)) {READ_LINE;}

  if (sscanf (s, "%d", &width)<0) return NULL;
  if (width<=0) return NULL;
  SKIP_NUMBERS;

  if (IS_LF(*s)) {READ_LINE;}
  else if (IS_WHITE_SPACE(*s)) s++; else return NULL;
  if (IS_COMMENT(*s)) {READ_LINE;}

  if (sscanf (s, "%d", &height)<0) return NULL;
  if (height<=0) return NULL; 
  SKIP_NUMBERS;

  if (type!=1 && type !=4) /* No maxval for pbm pictures */
    {
      if (IS_LF(*s)) {READ_LINE;}
      else if (IS_WHITE_SPACE(*s)) s++; else return NULL;
      if (IS_COMMENT(*s)) {READ_LINE;}

      if (sscanf (s, "%d", &maxval)<0) return NULL;
      if (maxval>255) return NULL; 
      SKIP_NUMBERS;
    }
  
  pic_p = create_block(name, width, height);
  if (pic_p==NULL) return NULL;
  buf = pic_p->buf;
  i = height*width;

  /* Load the pixel data, EOF causes break in loop */
  switch (type)
    {
    case 4: /* rawbits pbm */
      data = 0;
      wc = width;
      while (i--)
	{
	  if (wc--==0)
	    {
	      wc = width-1;
	      if (data!=7) data = 0;
	    }
	  if (data--==0)
	    {
	      data = 7;
	      val = getc(fp); if (val==EOF) break;
	    }
	  *buf++ = (((val>>7)^1)*CMASK);
	  val = (val<<1)&BMASK;
	}
      break;
    case 5: /* rawbits pgm */
      while (i--)
	{
	  val = getc(fp); if (val==EOF) break;
	  *buf++ = PUT_RED(val)|PUT_GREEN(val)|PUT_BLUE(val);
	}
      break;
    case 6: /* rawbits ppm */
      while (i--)
	{
	  val = getc(fp); if (val==EOF) break; data = PUT_RED(val);
	  val = getc(fp); if (val==EOF) break; data |= PUT_GREEN(val);
	  val = getc(fp); if (val==EOF) break; data |= PUT_BLUE(val);
	  *buf++ = data;
	}
      break;
    default:
      return NULL; /* support for these formats not implemented yet */
      break;
    }
  
  return pic_p;
}


/* FUNCTION:     int save(struct picture *pic_p, FILE *fp);
   DESCRIPTION:  Save picture to rawbits ppm format, file needs to be opened
   before calling this.
   PARAMETERS:
   struct picture *pic_p pointer to picture struct
   FILE *fp      pointer to FILE structure 
   RETURNS:      Returns -1 if something fails, otherwise 0.
   
   Written by Petri Kuittinen, last modifications 20th July 1993
*/
int save(struct picture *pic_p, FILE *fp)
{
  register int i;
  register pixel *buf, data;

  /* Write magic, picture dimensions and maxval */
  if (fprintf (fp, "P6\n%d %d\n255\n", pic_p->width, pic_p->height)<0)
    return -1;

  buf = pic_p->buf;
  i = pic_p->width*pic_p->height;

  while (i--)
    {
      data = *buf++;
      if (putc(GET_RED(data), fp)==EOF) break;
      if (putc(GET_GREEN(data), fp)==EOF) break;
      if (putc(GET_BLUE(data), fp)==EOF) break;
    }

  if (i>0) return -1; else return 0;
}
