/*
$Header: file_access.c,v 1.6 93/08/30 11:01:28 hmallat Exp $
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include "file_access.h"
#include "fileio.h"
#include "list.h"
#include "misc.h"
#include "X_misc.h"
#include "error.h"
#include "tools1.h"

struct io_filter *filter_head_p = NULL;

/* FUNCTION:       find_entries()
   DESCRIPTION:    Feed the names of the two possible .phoenixrc files
                   to scan_entries, first the one that is in the home
		   directory.
   PARAMETERS:     
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 23rd July 1993.
*/
void find_entries()
{
  char name[MAXSTRLEN];

  strcpy(name, getenv("HOME"));
  strcat(name, "/.phoenixrc");
  scan_entries(name);
#ifdef LIBDIR
  strcpy(name, LIBDIR);
  strcat(name, "/.phoenixrc");
  scan_entries(name);
#endif
}

/* FUNCTION:       scan_entries(char *filename)
   DESCRIPTION:    Read a .phoenixrc file, parse it, add valid filters to 
                   the list
   PARAMETERS:
   *filename       string containing the filename
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 23rd July 1993.
*/   
void scan_entries(char *filename)
{
  FILE *f;
  char e[MAXSTRLEN], pin[MAXSTRLEN], pout[MAXSTRLEN];
  char line[MAXSTRLEN], s[MAXSTRLEN], t[MAXSTRLEN];
  char *n;

  f = fopen(filename, "r");
  if(f != NULL)
    {
      /* read in a line, take the first whitespace-separated string as 
	 filename extension, locate aposthropes and take whatever is 
	 inside them as in-filter and out-filter. do nothing if the
	 line begins with a '#', which means a comment */
      while(fgets(line, MAXSTRLEN-1, f) != NULL)
	{
	  strcpy(e, "");
	  strcpy(pin, "");
	  strcpy(pout, "");
	  sscanf(line, "%s", e);
	  if(e[0] != '#')
	    {
	      strcpy(t, "");
	      strcpy(t, line+strlen(e));
	      if((n = strstr(t, "'")) != NULL)
		{
		  strcpy(s, "");
		  strcpy(s, n+1);
		  if((n = strstr(s, "'")) != NULL)
		    {
		      strncpy(pin, s, n-s);
		      pin[n-s] = '\0';
		      strcpy(t, "");
		      strcpy(t, n+1);
		      if((n = strstr(t, "'")) != NULL)
			{
			  strcpy(s, "");
			  strcpy(s, n+1);
			  if((n = strstr(s, "'")) != NULL)
			    {
			      strncpy(pout, s, n-s);
			      pout[n-s] = '\0';
			    }
			}
		    }
		}
	      if(strlen(e) && strlen(pin) && strlen(pout))
		add_filter(e, pin, pout);
	    }
	}
      fclose(f);
    }
}

/* FUNCTION:       add_filter(char *e, char *pin, char *pout);
   DESCRIPTION:    Add a filter as the last item to the list
   PARAMETERS:
   e               filename extension string
   pin             "in-program"
   pout            "out-program"
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 20th July 1993.
*/   
void add_filter(char *e, char *pin, char *pout)
{
  extern struct io_filter *filter_head_p;
  struct io_filter *f_p = NULL, *g_p = filter_head_p;

  if(filter_head_p == NULL)
    {
      f_p = (struct io_filter *)malloc(sizeof(struct io_filter));
    }
  else
    {
      while(g_p->next != NULL)
	g_p = g_p->next;
      f_p = (struct io_filter *)malloc(sizeof(struct io_filter));
      g_p->next = f_p;
    }
  if(f_p != NULL)
    {
      f_p->ext = (char *)malloc((strlen(e)+1)*sizeof(char));
      f_p->in  = (char *)malloc((strlen(pin)+1)*sizeof(char));
      f_p->out = (char *)malloc((strlen(pout)+1)*sizeof(char));
      if(f_p->ext != NULL && f_p->in != NULL && f_p->out != NULL)
	{
	  strcpy(f_p->ext, e);
	  strcpy(f_p->in, pin);
	  strcpy(f_p->out, pout);
	  f_p->next = NULL;
	  if(filter_head_p == NULL)
	    filter_head_p = f_p;
	}
      else
	{
	  if(f_p->ext != NULL) free(f_p->ext);
	  if(f_p->in  != NULL) free(f_p->in );
	  if(f_p->out != NULL) free(f_p->out);
	  panic("Cannot allocate memory (add_filter)", TRUE);
	}
    }
  else
    panic("Cannot allocate memory (add_filter)", TRUE);
}

/* FUNCTION:       del_filters(struct io_filter *filter_head_p);
   DESCRIPTION:    Delete filter list
   PARAMETERS:
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 12th July 1993.
*/   
void del_filters()
{
  extern struct io_filter *filter_head_p;
  struct io_filter *f_p = filter_head_p;

  while(filter_head_p != NULL)
    {
      f_p=filter_head_p->next;
      free(filter_head_p->ext);
      free(filter_head_p->in);
      free(filter_head_p->out);
      free(filter_head_p);
      filter_head_p = f_p;
    }
}

/* FUNCTION:       open_file(char *name)
   DESCRIPTION:    Open a file and call the low-level read routine
   PARAMETERS:
   name            File name
   RETURNS:        Pointer to the view or NULL if failed

   Written by Hannu Mallat, last modifications 13th July 1993.
*/   
struct view *open_file(char *name)
{
  extern struct io_filter *filter_head_p;
  struct io_filter *f_p = filter_head_p;
  struct view *v_p = NULL;
  struct picture *pic_p = NULL;
  char e[MAXSTRLEN], com[MAXSTRLEN];
  FILE *f;
  int found = 0;

  /* if a matching filename extension is found in the filter list,
     open a pipe for converting, otherwise open a file and assume 
     that the file is in pbmplus format */
  f = fopen(name, "rb");
  if(f == NULL)
    {
      all_busy_nomore();
      show_errormsg("Cannot read file (permission denied)!", "Bummer!", 
		    NULL);
    }
  else
    {
      fclose(f);
      while(f_p != NULL)
	{
	  strcpy(e, "");
	  strcpy(e, name+(strlen(name)-strlen(f_p->ext)));
	  if(!strcmp(e, f_p->ext) && strlen(f_p->in))
	    {
	      found = 1;
	      strcpy(com, "");
	      strcat(com, f_p->in);
	      strcat(com, " ");
	      strcat(com, name);
	      f = popen(com, "rb");
	      if(f != NULL)
		{
		  pic_p = load(name, f);
		  pclose(f);
		  all_busy_nomore();
		}
	      else
		{
		  all_busy_nomore();
		  show_errormsg("Unable to covert picture for loading!", "OK", 
				NULL);
		  return NULL;
		}
	      break;
	    }
	  f_p = f_p->next;
	}
      if(!found)
	{
	  f = fopen(name, "rb");
	  if(f != NULL)
	    {
	      pic_p = load(name, f);
	      fclose(f);
	      all_busy_nomore();
	    }
	  else
	    {
	      all_busy_nomore();
	      show_errormsg("Unable to open file for loading!", "OK", NULL);
	      return NULL;
	    }
	}
      if(pic_p != NULL)
	{
	  pic_p->min_x = 0;
	  pic_p->min_y = 0;
	  pic_p->max_x = 0;
	  pic_p->max_y = 0;
	  v_p = add_view(pic_p, 0, 0, pic_p->width-1, pic_p->height-1, 1);
	}
      else
	{
	  all_busy_nomore();
	  show_errormsg("Cannot load picture!", "OK", NULL);
	}
    }
  return v_p;
}

/* FUNCTION:       save_file(struct picture *pic_p, int flag)
   DESCRIPTION:    Save a file and call the low-level write routine
   PARAMETERS:
   pic_p           Pointer to the picture
   flag            Show error messages in X windows if TRUE.
                   FALSE option meant for panic saves, when the system
		   might be so out of memory that showing an alert might
		   crash it.
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 14th July 1993.
*/   
void save_file(struct picture *pic_p, int flag)
{
  extern struct io_filter *filter_head_p;
  extern int show_edges;
  extern PLOT xor_data;
  struct io_filter *f_p = filter_head_p;
  char e[MAXSTRLEN], com[MAXSTRLEN];
  char *name = pic_p->name;
  FILE *f;
  int n, found = 0;

  if(flag == TRUE)
    all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  /* if a matching filename extension is found in the filter list,
     open a pipe for converting, otherwise open a file and save 
     the picture in rawbits ppm format */
  f = fopen(name, "wb");
  if(f == NULL)
    {
      if(flag == TRUE)
	{
	  all_busy_nomore();
	  show_errormsg("Cannot write to file (permission denied)!", 
			"Bummer!", NULL);
	}
      else
	{
	  fprintf(stderr, "Cannot write to file %s (permission denied)!\n", 
		  name);
	}
    }
  else
    {
      fclose(f);
      while(f_p != NULL)
	{
	  strcpy(e, "");
	  strcpy(e, name+(strlen(name)-strlen(f_p->ext)));
	  if(!strcmp(e, f_p->ext) && strlen(f_p->out))
	    {
	      found = 1;
	      strcpy(com, "");
	      strcat(com, f_p->out);
	      strcat(com, " >");
	      strcat(com, name);
	      f = popen(com, "wb");
	      if(f != NULL)
		{
		  n = save(pic_p, f);
		  pclose(f);
		  if(n != 0)
		    {
		      if(flag == TRUE)
			{
			  all_busy_nomore();
			  show_errormsg("Unable to save file!", "OK", NULL);
			}
		      else
			{
			  fprintf(stderr, "Unable to save file %s\n", name);
			}
		      return;
		    }
		  else
		    untouch(pic_p);
		}
	      else
		{
		  if(flag == TRUE)
		    {
		      all_busy_nomore();
		      show_errormsg("Unable to convert picture for saving!", 
				    "OK", NULL);
		    }
		  else
		    {
		      fprintf(stderr, 
			      "Unable to convert picture %s for saving!\n", 
			      name);
		    }
		  return;
		}
	      break;
	    }
	  f_p = f_p->next;
	}
      if(found == 0)
	{
	  f = fopen(name, "wb");
	  if(f != NULL)
	    {
	      n = save(pic_p, f);
	      fclose(f);
	      if(n != 0)
		{
		  if(flag == TRUE)
		    {
		      all_busy_nomore();
		      show_errormsg("Cannot save file!", "OK", NULL);
		    }
		  else
		    {
		      fprintf(stderr, "Cannot save file %s!\n", name);
		    }
		  return;
		}
	      else
		untouch(pic_p);
	    }
	  else
	    {
	      if(flag == TRUE)
		{
		  all_busy_nomore();
		  show_errormsg("Unable to open file!", "OK", NULL);
		}
	      else
		{
		  fprintf(stderr, "Unable to open file %s!\n", name);
		}
	      return;
	    }
	}
    }
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  if(flag == TRUE)
    all_busy_nomore();
}
