/* event.c: View window event handlers and callbacks.

 $Header: event.c,v 2.16 93/09/03 14:46:17 wet Exp $
 */
#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/keysym.h>
#include <Xm/Xm.h>
#include <stdio.h>
#include <stdlib.h>
#include "list.h"
#include "event.h"
#include "window.h"
#include "tools1.h"
#include "error.h"
#include "undo.h"
#include "misc.h"
#include "callb.h"
#include "X_misc.h"
#include "file_access.h"

/* FUNCTION:       canvas_motion_handler(Widget widget_id, struct view *view_p,
                                         XEvent *call_data)
   DESCRIPTION:    Handler for PointerMotion events in the view_window, writes
                   the pointer coordinates to a label widget.
   PARAMETERS:
   Widget widget_id     Widget id of the widget that caused the callback.
   struct view *view_p  Pointer to view structure.
   XEvent *call_data    Pointer to the XEvent structure.
   RETURNS:        Nothing

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void canvas_motion_handler(Widget widget_id, struct view *view_p,
			   XEvent *call_data)
{
  extern struct picture *cur_pic;
  extern struct view *cur_view;
  char coords[12], *name = NULL, *oldname = NULL;
  XmString xs;
  Arg wargs[2];
  int need_free = 0;

  sprintf(coords, "(%04d,%04d)", call_data->xmotion.x/view_p->scale+view_p->x1,
	  call_data->xmotion.y/view_p->scale+view_p->y1);
  xs = XmStringLtoRCreate(coords, XmSTRING_DEFAULT_CHARSET);

  XtSetArg(wargs[0], XmNlabelString, xs);
  XtSetValues(view_p->coord_ind, wargs, 1);
  XmStringFree(xs);  
  if (call_data->xmotion.state==Button1Mask)
    {
      if (cur_pic!=NULL&&cur_view!=NULL&&cur_view!=view_p)
	{
	  oldname = malloc(1+(strlen(cur_pic->name))*sizeof(char));
	  strcpy(oldname, cur_pic->name);
	  XStoreName(call_data->xany.display, XtWindow(cur_view->main),
		     oldname);
	  need_free = 1;
	}
      cur_view = view_p;
      cur_pic = view_p->pic_p;
      update_menu();
      
      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      sprintf(name, "ACTIVE: %s", view_p->pic_p->name);
      XStoreName(call_data->xany.display, XtWindow(view_p->main), name);
      
      toolbranch(view_p, call_data->xmotion.x+view_p->x1*view_p->scale,
		 call_data->xmotion.y+view_p->y1*view_p->scale, LEFT, MOVE_W);

      free(name);
    }
  else if (call_data->xmotion.state==Button2Mask)
    {
      if (cur_pic!=NULL&&cur_view!=NULL&&cur_view!=view_p)
	{
	  oldname = malloc(1+(strlen(cur_pic->name))*sizeof(char));
	  strcpy(oldname, cur_pic->name);
	  XStoreName(call_data->xany.display, XtWindow(cur_view->main),
		     oldname);
	  need_free = 1;
	}
      cur_view = view_p;
      cur_pic = view_p->pic_p;
      update_menu();
      
      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      sprintf(name, "ACTIVE: %s", view_p->pic_p->name);
      XStoreName(call_data->xany.display, XtWindow(view_p->main), name);
      
      toolbranch(view_p, call_data->xmotion.x+view_p->x1*view_p->scale,
		 call_data->xmotion.y+view_p->y1*view_p->scale, CENTER,
		 MOVE_W);
      
      free(name);
    }
  else
    toolbranch(view_p, call_data->xmotion.x+view_p->x1*view_p->scale,
	       call_data->xmotion.y+view_p->y1*view_p->scale, NONE, MOVE_WO);
  if (need_free)
    free(oldname);
}


/* FUNCTION:       canvas_press_handler(Widget widget_id, caddr_t client_data,
                                        XEvent *call_data)
   DESCRIPTION:    Handler for ButtonPress events in the view_window, calls
                   toolbranch.
   PARAMETERS:
   Widget widget_id     Widget id of the widget that caused the callback.
   struct view *view_p  Pointer to view structure.
   XEvent *call_data    Pointer to the XEvent structure.
   RETURNS:        

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void canvas_press_handler(Widget widget_id, struct view *view_p,
			  XEvent *call_data)
{
  extern struct picture *cur_pic;
  extern struct view *cur_view;
  Arg wargs[10];
  Cardinal n;
  char *name = NULL, *oldname = NULL;
  long grab_mask;
  int need_free = 0;

  grab_mask = ButtonPress|PointerMotionMask;

  if (call_data->xbutton.button==1)
    {
      if (cur_pic!=NULL&&cur_view!=NULL&&cur_view!=view_p)
	{
	  oldname = malloc(1+(strlen(cur_pic->name))*sizeof(char));
	  strcpy(oldname, cur_pic->name);
	  XStoreName(call_data->xany.display, XtWindow(cur_view->main),
		     oldname);
	  need_free = 1;
	}
      cur_view = view_p;
      cur_pic = view_p->pic_p;
      update_menu();
      
      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      sprintf(name, "ACTIVE: %s", view_p->pic_p->name);
      XStoreName(call_data->xany.display, XtWindow(view_p->main), name);

      switch (need_grab())
	{
	case NEED_GRAB:
	  XGrabPointer(call_data->xany.display, call_data->xany.window,
		       False, grab_mask, GrabModeAsync, GrabModeAsync, None,
		       None, call_data->xbutton.time);
	  XGrabKeyboard(call_data->xany.display, call_data->xany.window, False,
			GrabModeAsync, GrabModeAsync, call_data->xbutton.time);
	  grab_flag = True;
	  break;
	case END_GRAB:
	  XUngrabPointer(call_data->xany.display, call_data->xbutton.time);
	  XUngrabKeyboard(call_data->xany.display, call_data->xbutton.time);
	  grab_flag = False;
	  break;
	case DONT_NEED_GRAB:
	default:
	  break;
	}
      toolbranch(view_p, call_data->xbutton.x+view_p->x1*view_p->scale,
		 call_data->xbutton.y+view_p->y1*view_p->scale, LEFT, PRESS);
    }
  else if (call_data->xbutton.button==2)
    {
      if (cur_pic!=NULL&&cur_view!=NULL&&cur_view!=view_p)
	{
	  oldname = malloc(1+(strlen(cur_pic->name))*sizeof(char));
	  strcpy(oldname, cur_pic->name);
	  XStoreName(call_data->xany.display, XtWindow(cur_view->main),
		     oldname);
	  need_free = 1;
	}
      cur_view = view_p;
      cur_pic = view_p->pic_p;
      update_menu();

      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      sprintf(name, "ACTIVE: %s", view_p->pic_p->name);
      XStoreName(call_data->xany.display, XtWindow(view_p->main), name);

      switch (need_grab())
	{
	case NEED_GRAB:
	  XGrabPointer(call_data->xany.display, call_data->xany.window,
		       False, grab_mask, GrabModeAsync, GrabModeAsync, None, 
		       None, call_data->xbutton.time);
	  XGrabKeyboard(call_data->xany.display, call_data->xany.window, False,
		        GrabModeAsync, GrabModeAsync, call_data->xbutton.time);
	  grab_flag = True;
	  break;
	case END_GRAB:
	  XUngrabPointer(call_data->xany.display, call_data->xbutton.time);
	  XUngrabKeyboard(call_data->xany.display, call_data->xbutton.time);
	  grab_flag = False;
	  break;
	case DONT_NEED_GRAB:
	default:
	  break;
	}
      toolbranch(view_p, call_data->xbutton.x+view_p->x1*view_p->scale,
		 call_data->xbutton.y+view_p->y1*view_p->scale, CENTER, PRESS);
    }
  else if (call_data->xbutton.button==3)
    {
      if (grab_flag==True)
	{
	  XUngrabPointer(call_data->xany.display, call_data->xbutton.time);
	  XUngrabKeyboard(call_data->xany.display, call_data->xbutton.time);
	}
      toolbranch(view_p, 0, 0, NONE, CANCEL);
      
      n = 0;
      XtSetArg(wargs[n], XmNdefaultPosition, FALSE); n++;
      XtSetArg(wargs[n], XmNx, call_data->xbutton.x_root-TOP_WIDTH/2); n++; 
      XtSetArg(wargs[n], XmNy, call_data->xbutton.y_root-TOP_HEIGHT/2); n++;
      XtSetValues(toplevel, wargs, n);
      XRaiseWindow(XtDisplay(toplevel), XtWindow(toplevel));
    }
  if (need_free)
    free(oldname);
  free(name);
}


/* FUNCTION:       canvas_release_handler(Widget widget_id,
                                          caddr_t client_data,
                                          XEvent *call_data)
   DESCRIPTION:    Handler for ButtonRelease events in the view_window, calls
                   toolbranch.
   PARAMETERS:
   Widget widget_id     Widget id of the widget that caused the callback.
   struct view *view_p  Pointer to view structure.
   XEvent *call_data    Pointer to the XEvent structure.
   RETURNS:        

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void canvas_release_handler(Widget widget_id, struct view *view_p,
			    XEvent *call_data)
{
  if (call_data->xbutton.button==1)
    toolbranch(view_p, call_data->xbutton.x+view_p->x1*view_p->scale,
	       call_data->xbutton.y+view_p->y1*view_p->scale, LEFT, RELEASE);
  if (call_data->xbutton.button==2)
    toolbranch(view_p, call_data->xbutton.x+view_p->x1*view_p->scale,
	       call_data->xbutton.y+view_p->y1*view_p->scale, CENTER, RELEASE);
}



/* FUNCTION:       canvas_keypress_handler(Widget widget_id,
                                           struct view *view_p,
                                           XEvent *call_data)
   DESCRIPTION:    Handler for KeyPress events in the view_window, first
                   version only chaeck for ESC and sends a CANCEL to the
		   main module.
   PARAMETERS:
   Widget widget_id     Widget id of the widget that caused the callback.
   struct view *view_p  Pointer to view structure.
   XEvent *call_data    Pointer to the XEvent structure.
   RETURNS:        Nothing

   Written by Vesa T|rm{nen, last modifications 16th July 1993.
*/   
void canvas_keypress_handler(Widget widget_id, struct view *view_p,
			     XEvent *call_data)
{
  extern struct picture *cur_pic;
  extern struct view *cur_view;
  char *name = NULL, *oldname = NULL;
  int need_free = 0;
  Display *disp = call_data->xany.display;

  if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_Escape))
    {
      if (grab_flag==True)
	{
	  XUngrabPointer(call_data->xany.display, call_data->xkey.time);
	  XUngrabKeyboard(call_data->xany.display, call_data->xkey.time);
	}
      toolbranch(view_p, 0, 0, NONE, CANCEL);
    }
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_Return) ||
	   call_data->xkey.keycode==XKeysymToKeycode(disp, XK_space))
    {
      if (cur_pic!=NULL&&cur_view!=NULL&&cur_view!=view_p)
	{
	  oldname = malloc(1+(strlen(cur_pic->name))*sizeof(char));
	  strcpy(oldname, cur_pic->name);
	  XStoreName(call_data->xany.display, XtWindow(cur_view->main),
		     oldname);
	  need_free = 1;
	}
      cur_view = view_p;
      cur_pic = view_p->pic_p;
      update_menu();
      
      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      sprintf(name, "ACTIVE: %s", view_p->pic_p->name);
      XStoreName(call_data->xany.display, XtWindow(view_p->main), name);
      if (need_free)
	free(oldname);
      free(name);
    }
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_n) &&
	   call_data->xkey.state==ControlMask)
    file_new_callback((Widget)NULL, (char *)NULL, (caddr_t)NULL);
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_o) &&
	   call_data->xkey.state==ControlMask)
    file_open_callback((Widget)NULL, (char *)NULL, (caddr_t)NULL);
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_s) &&
	   call_data->xkey.state==ControlMask)
    save_file(cur_pic, 1);
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_s) &&
	   call_data->xkey.state==Mod1Mask)
    file_saveas_callback((Widget)NULL, (char *)NULL, (caddr_t)NULL);
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_a) &&
	   call_data->xkey.state==ControlMask)
    abandon_q();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_q) &&
	   call_data->xkey.state==ControlMask)
    exit_init();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_u) &&
	   call_data->xkey.state==ControlMask)
    undo();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_c) &&
	   call_data->xkey.state==Mod1Mask)
    cut();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_d) &&
	   call_data->xkey.state==ControlMask)
    delete();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_c) &&
	   call_data->xkey.state==ControlMask)
    copy();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_p) &&
	   call_data->xkey.state==ControlMask)
    paste();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_p) &&
	   call_data->xkey.state==Mod1Mask)
    paste_to_toolbox();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_a) &&
	   call_data->xkey.state==Mod1Mask)
    select_all();
  else if (call_data->xkey.keycode==XKeysymToKeycode(disp, XK_d) &&
	   call_data->xkey.state==Mod1Mask)
    deselect_all();
}



/* FUNCTION:       canvas_leave_handler(Widget widget_id, struct view *view_p,
                                          XEvent *call_data)
   DESCRIPTION:    Handler for LeaveNotify events in the view_window, clears
                   the label when the pointer leaves the window.
   PARAMETERS:
   Widget widget_id     Widget id of the widget that caused the callback.
   struct view *view_p  Pointer to view structure.
   XEvent *call_data    Pointer to the XEvent structure.
   RETURNS:        Nothing

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void canvas_leave_handler(Widget widget_id, struct view *view_p,
			  XEvent *call_data)
{
  char coords[12];
  XmString xs;
  Arg wargs[2];

  if (call_data->xcrossing.state!=Button1Mask &&
      call_data->xcrossing.state!=Button2Mask)
     {
      sprintf(coords, "(    ,    )");
      xs = XmStringLtoRCreate(coords, XmSTRING_DEFAULT_CHARSET);
      
      XtSetArg(wargs[0], XmNlabelString, xs);
      XtSetValues(view_p->coord_ind, wargs, 1);
      XmStringFree(xs);  
      toolbranch(view_p, 0, 0, NONE, LEAVE);
    }
}



/* FUNCTION:       view_map_handler(Widget widget_id, struct view *view_p, 
                                    XEvent *call_data)
   DESCRIPTION:    Handler of DrawingArea ConfigureNotify events for the 
                   view_window,makes the window's view and picture current.
   PARAMETERS:
   Widget widget_id                        Widget_id of the widget that caused
                                             the callback.
   struct view *view_p                     Pointer to view structure.
   XmDrawingAreaCallbackStruct *call_data  Event structure.
   RETURNS:        Nothing

   Written by Vesa T|rm{nen, last modifications 22th August 1993.
*/
void view_map_handler(Widget widget_id, struct view *view_p, XEvent *call_data)
{
  char *name = NULL, *oldname = NULL;
  extern struct picture *cur_pic;  
  extern struct view *cur_view;
  int need_free = 0;
  if (call_data->type==ConfigureNotify)
    {
      if (cur_pic!=NULL&&cur_view!=NULL&&cur_view!=view_p)
	{
	  oldname = malloc(1+(strlen(cur_pic->name))*sizeof(char));
	  strcpy(oldname, cur_pic->name);
	  XStoreName(call_data->xany.display, XtWindow(cur_view->main),
		     oldname);
	  need_free = 1;
	}
      cur_view = view_p;
      cur_pic = view_p->pic_p;
      update_menu();

      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      sprintf(name, "ACTIVE: %s", view_p->pic_p->name);
      XStoreName(call_data->xany.display, XtWindow(view_p->main), name);
      name = malloc((10+strlen(view_p->pic_p->name))*sizeof(char));
      if (need_free)
	free(oldname);
      free(name);
    }
}


/* FUNCTION:       delete_canvas_handler(Widget widget_id,
                                         struct view *view_p, 
					 XmAnyCallbackStuct *call_data)
   DESCRIPTION:    Handles WM_DELETE_WINDOW ClientMessages.
   PARAMETERS:
   Widget widget_id                Widget_id of the widget that caused
                                     the callback.
   struct view *view_p             Pointer to view structure.
   XmAnyCallbackStruct *call_data  Event structure.
   RETURNS:        Nothing

   Written by Vesa T|rm{nen, last modifications 22th August 1993.
*/
void delete_canvas(Widget widget_id, struct view *view_p,
		   XmAnyCallbackStruct *call_data)
{
  if (call_data->event->xclient.data.l[0]==wm_delete_window)
    kill_window_q(view_p);
}



/* FUNCTION:       expose_canvas(Widget widget_id, struct view *view_p, 
                                 XmDrawingAreaCallbackStruct *call_data)
   DESCRIPTION:    Handler of DrawingArea expose events dor the view_window.
   PARAMETERS:
   Widget widget_id                        Widget_id of the widget that caused
                                             the callback.
   struct view *view_p                     Pointer to view structure.
   XmDrawingAreaCallbackStruct *call_data  Event structure.
   RETURNS:        Nothing

   Written by Vesa Tormanen, last modifications 4th August 1993.
*/   
void canvas_exposure_handler(Widget widget_id, struct view *view_p,
			     XEvent *call_data)
{
  extern int depth;
  XExposeEvent *ev = &call_data->xexpose;
  Display *disp = call_data->xany.display;
  Window win = ev->window;
  XImage *image;
  char *p;

  int x1 = MAX(view_p->x1+ev->x/view_p->scale, 0);
  int y1 = MAX(view_p->y1+ev->y/view_p->scale, 0);
  int x2 = MIN(view_p->x1+(ev->x+ev->width-1)/view_p->scale, 
	       view_p->pic_p->width-1);
  int y2 = MIN(view_p->y1+(ev->y+ev->height-1)/view_p->scale,
	       view_p->pic_p->height-1);
  all_busy();

  p = feed_block(view_p->pic_p, x1, y1, x2, y2, view_p->scale);
  
  image = XCreateImage(disp, view_vis, depth, ZPixmap, 0, p,
		       view_p->scale*(x2-x1+1), view_p->scale*(y2-y1+1), 
		       pad, 0);
  XPutImage(disp, win, view_p->gc, image, 0, 0,
	    (call_data->xexpose.x/view_p->scale)*view_p->scale,
	    (call_data->xexpose.y/view_p->scale)*view_p->scale,
	    view_p->scale*(x2-x1+1), view_p->scale*(y2-y1+1)); 
  XFlush(disp);
  free(p);
  XFree(image);
  all_busy_nomore();
}


/* FUNCTION:       resize_canvas(Widget widget_id, struct view *view_p, 
                                 XmDrawingAreaCallbackStruct *call_data)
   DESCRIPTION:    Handler of DrawingArea resize events dor the view_window.
   PARAMETERS:
   Widget widget_id                        Widget_id of the widget that caused
                                             the callback.
   struct view *view_p                     Pointer to view structure.
   XmDrawingAreaCallbackStruct *call_data  Event structure.
   RETURNS:        Nothing

   Written by Vesa Tormanen, last modifications 4th August 1993.
*/   
void resize_canvas(Widget widget_id, struct view *view_p,
		   XmDrawingAreaCallbackStruct *call_data)
{
  Arg wargs[10];
  Cardinal n;
  Dimension width, height;

  n = 0;
  XtSetArg(wargs[n], XmNwidth, &width); n++;
  XtSetArg(wargs[n], XmNheight, &height); n++;
  XtGetValues(widget_id, wargs, n);

  width = MIN(view_p->pic_p->width*view_p->scale, width);
  height = MIN(view_p->pic_p->height*view_p->scale, height);

  width = (width%view_p->scale) ? width/view_p->scale+1 :
    width/view_p->scale;
  height = (height%view_p->scale) ? height/view_p->scale+1 :
    height/view_p->scale;

  if (width>view_p->pic_p->width-view_p->x1)
    {
      view_p->x1 = MAX(view_p->pic_p->width-width, 0);
      XClearArea(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		 0, 0, 0, 0, True);
    }
  if (height>view_p->pic_p->height-view_p->y1)
    {
      view_p->y1 = MAX(view_p->pic_p->height-height, 0);
      XClearArea(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		 0, 0, 0, 0, True);
    }
  n = 0;
  XtSetArg(wargs[n], XmNvalue, view_p->x1); n++;
  XtSetArg(wargs[n], XmNsliderSize, width-1); n++;
  XtSetArg(wargs[n], XmNpageIncrement, width); n++;
  XtSetValues(view_p->h_scroll, wargs, n);

  n = 0;
  XtSetArg(wargs[n], XmNvalue, view_p->y1); n++;
  XtSetArg(wargs[n], XmNsliderSize, height-1); n++;
  XtSetArg(wargs[n], XmNpageIncrement, height); n++;
  XtSetValues(view_p->v_scroll, wargs, n);

  view_p->x2 = MIN(view_p->x1+width-1, view_p->pic_p->width-1);
  view_p->y2 = MIN(view_p->y1+height-1, view_p->pic_p->height-1);
}



/* FUNCTION:       h_scroll_canvas(Widget widget_id, struct view *view_p,
                                   XmScrollBarCallbackStruct *call_data)
   DESCRIPTION:    Function that updates view struct values according to
                   horizontal scroll bar movement.
   PARAMETERS:
   Widget widget_id                     ScrollBars widget id.
   struct view *view_p                  Pointer to view structure.
   XmScrollBarCallbackStruct *call_data Callback data.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 29th July 1993.
*/   
void h_scroll_canvas(Widget widget_id, struct view *view_p,
		     XmScrollBarCallbackStruct *call_data)
{
  extern int depth;
  XImage *image;
  char *p;
  Display *disp = XtDisplay(view_p->draw_area);
  Window win = XtWindow(view_p->draw_area);
  XEvent event;
  XGraphicsExposeEvent *ev;
  int delta, xsrc, copy_width, redraw_left, feed_left;
  int s_pos = call_data->value;
  int width = view_p->x2-view_p->x1;
  int height = view_p->y2-view_p->y1;
  int x1, y1, x2, y2;

  if (s_pos>view_p->x2||s_pos+width<view_p->x1)
    {
      view_p->x1 = s_pos;
      view_p->x2 = s_pos+width;
      XClearArea(disp, win, 0, 0, 0, 0, True);
      return;
    }

  delta = ABS(view_p->x1-s_pos);
  delta = MIN(delta, width);

  if (s_pos>=view_p->x1) /* scrolling right */
    {
      xsrc = delta;
      copy_width = width-delta;
      feed_left = view_p->x2;
      redraw_left = width-delta;
    }
  else                     /* scrolling left */
    {
      xsrc = redraw_left = 0;
      copy_width =width-delta+1;
      feed_left = view_p->x1-delta;
    }
  
  view_p->x1 = s_pos;
  view_p->x2 = s_pos+width;
  
  XCopyArea(disp, win, win, view_p->gc,
	    xsrc*view_p->scale, 0,
	    copy_width*view_p->scale, (height+1)*view_p->scale,
	    (delta-xsrc)*view_p->scale, 0);

  for (;;) {
    XIfEvent(disp, &event, predicate, (XPointer) win);
    if (event.type==GraphicsExpose) 
      {
	ev = &event.xgraphicsexpose;
	x1 = MAX(view_p->x1+ev->x/view_p->scale, 0);
	y1 = MAX(view_p->y1+ev->y/view_p->scale, 0);
	x2 = MIN(view_p->x1+(ev->x+ev->width-1)/view_p->scale, 
		 view_p->pic_p->width-1);
	y2 = MIN(view_p->y1+(ev->y+ev->height-1)/view_p->scale,
		 view_p->pic_p->height-1);
	
	p = feed_block(view_p->pic_p, x1, y1, x2, y2, view_p->scale);
	
	image = XCreateImage(disp, view_vis, depth, ZPixmap,
			     0, p, view_p->scale*(x2-x1+1),
			     view_p->scale*(y2-y1+1), pad, 0);
	XPutImage(disp, win, view_p->gc, image, 0, 0,
		  (ev->x/view_p->scale)*view_p->scale,
		  (ev->y/view_p->scale)*view_p->scale,
		  view_p->scale*(x2-x1+1), view_p->scale*(y2-y1+1)); 
	free(p);
	XFree(image);
	if (ev->count==0)
	  break;
      } 
    else /* NoExpose event */
      break;
  }

  p = feed_block(view_p->pic_p, feed_left, view_p->y1,
		 feed_left+delta, view_p->y2, view_p->scale);
  
  image = XCreateImage(disp, view_vis, depth, ZPixmap, 0, p,
		       (delta+1)*view_p->scale, (height+1)*view_p->scale, 
		       pad, 0);
  XPutImage(disp, win, view_p->gc, image, 0, 0, redraw_left*view_p->scale, 0,
	    (delta+1)*view_p->scale, (height+1)*view_p->scale); 
  XFlush(disp);
  free(p);
  XFree(image);
}



/* FUNCTION:       v_scroll_canvas(Widget widget_id, struct view *view_p,
                                   XmScrollBarCallbackStruct *call_data)
   DESCRIPTION:    Function that updates view struct values according to
                   vertical scroll bar movement.
   PARAMETERS:
   Widget widget_id                     ScrollBars widget id.
   struct view *view_p                  Pointer to view structure.
   XmScrollBarCallbackStruct *call_data Callback data.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 29th July 1993.
*/   
void v_scroll_canvas(Widget widget_id, struct view *view_p,
		     XmScrollBarCallbackStruct *call_data)
{
  extern int depth;
  XImage *image;
  char *p;
  Display *disp = XtDisplay(view_p->draw_area);
  Window win = XtWindow(view_p->draw_area);
  XEvent event;
  XGraphicsExposeEvent *ev;
  int delta, ysrc, copy_height, redraw_top, feed_top;
  int s_pos = call_data->value;
  int width = view_p->x2-view_p->x1;
  int height = view_p->y2-view_p->y1;
  int x1, x2, y1, y2;

  if (s_pos>view_p->y2||s_pos+height<view_p->y1)
    {
      view_p->y1 = s_pos;
      view_p->y2 = s_pos+height;
      XClearArea(disp, win, 0, 0, 0, 0, True);
      return;
    }

  delta = ABS(view_p->y1-s_pos);
  delta = MIN(delta, height);

  if (s_pos>=view_p->y1)
    {
      ysrc = delta;
      copy_height = height-delta;
      feed_top = view_p->y2;
      redraw_top = height-delta;
    }
  else
    {
      ysrc = redraw_top = 0;
      copy_height = height-delta+1;
      feed_top = view_p->y1-delta;
    }
      
  view_p->y1 = s_pos;
  view_p->y2 = s_pos+height;

  XCopyArea(disp, win, win, view_p->gc,
	    0, ysrc*view_p->scale,
	    (width+1)*view_p->scale, copy_height*view_p->scale,
	    0, (delta-ysrc)*view_p->scale);

  for (;;) {
    XIfEvent(disp, &event, predicate, (XPointer) win);
    if (event.type==GraphicsExpose) 
      {
	ev = &event.xgraphicsexpose;
	x1 = MAX(view_p->x1+ev->x/view_p->scale, 0);
	y1 = MAX(view_p->y1+ev->y/view_p->scale, 0);
	x2 = MIN(view_p->x1+(ev->x+ev->width-1)/view_p->scale, 
		 view_p->pic_p->width-1);
	y2 = MIN(view_p->y1+(ev->y+ev->height-1)/view_p->scale,
		 view_p->pic_p->height-1);
	
	p = feed_block(view_p->pic_p, x1, y1, x2, y2, view_p->scale);
	
	image = XCreateImage(disp, view_vis, depth, ZPixmap,
			     0, p, view_p->scale*(x2-x1+1),
			     view_p->scale*(y2-y1+1), pad, 0);
	XPutImage(disp, win, view_p->gc, image, 0, 0,
		  (ev->x/view_p->scale)*view_p->scale,
		  (ev->y/view_p->scale)*view_p->scale,
		  view_p->scale*(x2-x1+1), view_p->scale*(y2-y1+1)); 
	free(p);
	XFree(image);
	if (ev->count==0)
	  break;
      } 
    else /* NoExpose event */
      break;
  }
  p = feed_block(view_p->pic_p, view_p->x1, feed_top, view_p->x2,
		 feed_top+delta, view_p->scale);
      
  image = XCreateImage(disp, view_vis, depth, ZPixmap, 0, p,
		       (width+1)*view_p->scale, (delta+1)*view_p->scale,
		       pad, 0);

  XPutImage(disp, win, view_p->gc, image, 0, 0, 0, redraw_top*view_p->scale,
	    (width+1)*view_p->scale, (delta+1)*view_p->scale);
  XFlush(disp);
  free(p);
  XFree(image);
}


/* FUNCTION:       expose_ind(Widget widget_id, struct view *view_p, 
                              XmDrawingAreaCallbackStruct *call_data)
   DESCRIPTION:    Handler of DrawingArea expose events for the view_window
                   color indicators.
   PARAMETERS:
   Widget widget_id                        Widget_id of the widget that caused
                                             the callback.
   struct view *view_p                     Pointer to view structure.
   XmDrawingAreaCallbackStruct *call_data  Event structure.
   RETURNS:        Nothing

   Written by Vesa Tormanen, last modifications 14th July 1993.
*/   
void expose_ind(Widget widget_id, struct view *view_p,
		XmDrawingAreaCallbackStruct *call_data)
{
  show_colors(view_p);
}


/* FUNCTION:       predicate(Display *disp, XEvent *event, XPointer arg)
   DESCRIPTION:    Tests if the event is of type GraphicsExpose or NoExpose.
                   Used in the scrolling callbacks.
   PARAMETERS:
   Display *disp   Pointer to display structure.
   XEvent *event   Pointer to event that is being tested.
   XPointer arg    Event window.
   RETURNS:        Nothing

   Written by Mike Stroyan (mike_stroyan@fc.hp.com), last modifications
              (by wet) 9th August 1993.
*/   
Bool predicate(Display *disp, XEvent *event, XPointer arg)
{
    Window win = (Window) arg;
    return((event->type==GraphicsExpose
       &&event->xgraphicsexpose.drawable==win)
       ||(event->type==NoExpose
       &&event->xnoexpose.drawable==win));
}













