#include <stdlib.h>
#include <stdio.h>
#include "pixel.h"
#include "misc.h"
#include "convert.h"

static int dit_mat[256];
static int dit_mat2[256];

/* FUNCTION:     char *block_to_24bit (struct picture *pic_p, int x1, int y1,
                                       int x2, int y2, int scale)
   DESCRIPTION:  convert part of a block to 24-bit pixmap data. Coordinates
                 (x1, y1) and (x2, y2) are the opposite vertexes of the block.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   scale         scale value, 1=normal (1:1)
   RETURNS:      pointer to the char array which holds the converted picture
                 data. Each pixel is 4 chars, the first one is zero and
		 the next ones hold the RGB values.
   
   Written by Petri Kuittinen, last modifications 12nd July 1993
*/
char *block_to_24bit (struct picture *pic_p, int x1, int y1, int x2, int y2,
		      int scale)
{
  register int x, y, buf_offset, i, j;
  register pixel *buf1, *buf2, data;
  register char *p1, *p2;
  
  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;
  
  buf1 = pic_p->buf+x1+y1*pic_p->width;
  buf_offset = pic_p->width-(x2-x1+1);

  p1 = malloc((x2-x1+1)*(y2-y1+1)*scale*scale*4*sizeof(char));
  if (p1==NULL) return NULL;
  
  p2 = p1;
  for (y=y1; y<=y2; y++)
    {
      for (i=0; i<scale; i++)
	{
	  buf2 = buf1;
	  for (x=x1; x<=x2; x++)
	    {
	      data = *buf2++;
	      for (j=0; j<scale; j++)
		{
		  *p2++ = 0;      
		  *p2++ = GET_RED(data);
		  *p2++ = GET_GREEN(data);
		  *p2++ = GET_BLUE(data);
		}
	    }
	}
      buf1 = buf2;
      buf1 += buf_offset;
    }

  return p1;
}


/* FUNCTION:     char *block_to_8gs (struct picture *pic_p, int x1, int y1,
   int x2, int y2, int scale)
   DESCRIPTION:  convert part of a block to 8-bit gray scale pixmap data.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the block.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   scale         scale value, 1=normal (1:1)
   RETURNS:      pointer to the char array which holds the converted picture
                 data. Each pixel is 1 char.
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
char *block_to_8gs (struct picture *pic_p, int x1, int y1, int x2, int y2,
		    int scale)
{
  register int x, y, buf_offset, i, j;
  register pixel *buf1, *buf2, data;
  register char *p1, *p2, val;

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;
  
  buf1 = pic_p->buf+x1+y1*pic_p->width;
  buf_offset = pic_p->width-(x2-x1+1);

  p1 = malloc((x2-x1+1)*(y2-y1+1)*scale*scale*sizeof(char));
  if (p1==NULL) return NULL;
  
  p2 = p1;
  for (y=y1; y<=y2; y++)
    {
      for (i=0; i<scale; i++)
	{
	  buf2 = buf1;
	  for (x=x1; x<=x2; x++)
	    {
	      data = *buf2++;
	      val = 77*GET_RED(data)/256+150*GET_GREEN(data)/256+\
		29*GET_BLUE(data)/256;
	      for (j=0; j<scale; j++)
		*p2++ = val;
	    }
	}
      buf1 = buf2;
      buf1 += buf_offset;
    }

  return p1;
}


/* FUNCTION:     char *block_to_1bit (struct picture *pic_p)
   DESCRIPTION:  convert block to 1 bit pixmap data. Masked pixels are
   represented as 1, and non-masked as zero.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   RETURNS:      pointer to the char array which holds the converted picture
                 data. 8 pixels are packed into each char.
   
   Written by Petri Kuittinen, last modifications 12th August 1993
*/
char *mask_to_1bit (struct picture *pic_p)
{
  register int w, h, count;
  register pixel *src;
  register char *tar1, *tar2, data;
  
  src = pic_p->buf;
  w = pic_p->width;
  h = pic_p->height;

  tar1 = tar2 = malloc(h*(w/8+((w&7)!=0))*sizeof(char));
  if (tar1==NULL) return NULL;
  
  while (h--)
    {
      w = pic_p->width;
      count = data = 0;

      while (w--)
	{
	  if (GET_MASK(*src++))
	    data |= 1<<count;
	  if (count++==7)
	    {
	      *tar2++ = data;
	      count = 0;
	      data = 0;
	    }
	}
      if (count!=0) *tar2++ = data; /* Flush last bits of line */
   }

  return tar1;
}


/* FUNCTION:     int dither_value (int x, int y, int size)
   DESCRIPTION:  calculate dither value of certain point (x, y) in 
   2^size * 2^size ordered dither matrix.
   PARAMETERS:
   int x         x location in matrix [0,2^size-1] 
   int y
   int size
   RETURNS:      dither value
   
   Written by Petri Kuittinen, last modifications 18th August 1993
*/
int dither_value (int x, int y, int size)
{
  register int d = 0;

  while (size-->0)
    {
      d = (d<<1|((x&1)^(y&1)))<<1|(y&1);
      x >>= 1; y >>= 1;
    }
  return(d);
}

/* FUNCTION:     void init_4gs()
   DESCRIPTION:  initialises 8x8 dither matrix for 4-bit gray scale
   conversion routine. This routine must be called before calling
   block_to_4gs!
   PARAMETERS:
   none
   RETURNS:      none
   
   Written by Petri Kuittinen, last modifications 18th August 1993
*/
void init_4gs()
{
  register int *p, x, y;
  extern int dit_mat[256];

  p = &dit_mat[0];
  
  for (y = 0; y<8; y++)
    for (x = 0; x<8; x++)
      *p++ = dither_value(x, y, 4)/16;
}


/* FUNCTION:     char *block_to_4gs (struct picture *pic_p, int x1, int y1,
   int x2, int y2, int scale)
   DESCRIPTION:  convert part of a block to 4-bit gray scale pixmap data.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the block.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   scale         scale value, 1=normal (1:1)
   RETURNS:      pointer to the char array which holds the converted picture
                 data. 2 pixels are packed to 1 char.
   
   Written by Petri Kuittinen, last modifications 18th August 1993
*/
char *block_to_4gs (struct picture *pic_p, int x1, int y1, int x2, int y2,
		    int scale)
{
  register int x, y, buf_offset, i, j, data, count;
  register pixel *buf1, *buf2, src;
  register char *p1, *p2, val;

  extern int dit_mat[256];

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;
  
  buf1 = pic_p->buf+x1+y1*pic_p->width;
  buf_offset = pic_p->width-(x2-x1+1);

  p2 = p1 = malloc(((x2-x1+1)/2+(((x2-x1+1)&1)!=0))*(y2-y1+1)*scale*scale*\
		   sizeof(char));
  if (p1==NULL) return NULL;

  for (y=y1; y<=y2; y++)
    {
      for (i=0; i<scale; i++)
	{
	  buf2 = buf1;
	  data = count = 0;
	  for (x=x1; x<=x2; x++)
	    {
	      src = *buf2++;
	      val = 77*GET_RED(src)/256+150*GET_GREEN(src)/256+\
		29*GET_BLUE(src)/256;
	      val = (val>>4)+((val&15)>dit_mat[(x&7)+8*(y&7)]); LIMIT_15(val);
	      for (j=0; j<scale; j++)
		{
		  if (count++==0)
		    data |= val<<4;
		  else
		    {
		      data |= val;
		      *p2++ = data;
		      count = data = 0;
		    }
		}
	    }
	  if (count!=0) *p2++ = data;	
	}
      buf1 = buf2;
      buf1 += buf_offset;
    }

  return p1;
}


/* FUNCTION:     void init_1gs()
   DESCRIPTION:  initialises 8x8 dither matrix for 1-bit gray scale
   conversion routine. This routine must be called before calling
   block_to_1gs!
   PARAMETERS:
   none
   RETURNS:      none
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
void init_1gs()
{
  register int *p, x, y;
  extern int dit_mat[256];

  p = &dit_mat[0];
  
  for (y = 0; y<8; y++)
    for (x = 0; x<8; x++)
      *p++ = dither_value(x, y, 4);
}


/* FUNCTION:     char *block_to_1gs (struct picture *pic_p, int x1, int y1,
   int x2, int y2, int scale)
   DESCRIPTION:  convert part of a block to 1-bit gray scale pixmap data.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the block.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   scale         scale value, 1=normal (1:1)
   RETURNS:      pointer to the char array which holds the converted picture
                 data. 8 pixels are packed to 1 char.
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
char *block_to_1gs (struct picture *pic_p, int x1, int y1, int x2, int y2,
		    int scale)
{
  register int x, y, buf_offset, i, j, data, count;
  register pixel *buf1, *buf2, src;
  register char *p1, *p2;
  register unsigned char val;

  extern int dit_mat[256];

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;
  
  buf1 = pic_p->buf+x1+y1*pic_p->width;
  buf_offset = pic_p->width-(x2-x1+1);

  p2 = p1 = malloc(((x2-x1+1)/8+(((x2-x1+1)&7)!=0))*(y2-y1+1)*scale*scale*\
		   sizeof(char));
  if (p1==NULL) return NULL;

  for (y=y1; y<=y2; y++)
    {
      for (i=0; i<scale; i++)
	{
	  buf2 = buf1;
	  data = 0;
	  count = 7;
	  for (x=x1; x<=x2; x++)
	    {
	      src = *buf2++;
	      val = 77*GET_RED(src)/256+150*GET_GREEN(src)/256+\
		29*GET_BLUE(src)/256;
	      val = ((val)>dit_mat[(x&7)+8*(y&7)]);
	      /* if (val>1) val = 1; */
	      for (j=0; j<scale; j++)
		{
		  if (count)
		    {
		      data |= val<<count;
		      count--;	
		    }
		  else
		    {
		      data |= val;
		      *p2++ = ~data;
		      count = 7;
		      data = 0;
		    }
		}
	    }
	  if (count!=7) *p2++ = ~data;
	}
      buf1 = buf2;
      buf1 += buf_offset;
    }

  return p1;
}


/* FUNCTION:     void init_8c()
   DESCRIPTION:  initialises 8x8 dither matrix for 8-bit color display
   conversion routine. This routine must be called before calling
   block_to_8gs!
   PARAMETERS:
   none
   RETURNS:      none
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
void init_8c()
{
  register int *p, x, y;
  extern int dit_mat[256];
  extern int dit_mat2[256];

  p = &dit_mat[0];
  
  for (y = 0; y<8; y++)
    for (x = 0; x<8; x++)
      *p++ = dither_value(x, y, 3)/2;

  p = &dit_mat2[0];

  for (y = 0; y<8; y++)
    for (x = 0; x<8; x++)
      *p++ = dither_value(x, y, 3);
}


/* FUNCTION:     char *block_to_8c (struct picture *pic_p, int x1, int y1,
   int x2, int y2, int scale)
   DESCRIPTION:  convert part of a block to 8-bit color pixmap data.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the block.
   Assumese 3 bits for R and G and 2 bits for B component.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   scale         scale value, 1=normal (1:1)
   RETURNS:      pointer to the char array which holds the converted picture
                 data. 1 pixel is 1 char.
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
char *block_to_8c (struct picture *pic_p, int x1, int y1, int x2, int y2,
		   int scale)
{
  register int x, y, buf_offset, i, j;
  register pixel *buf1, *buf2, src;
  register char *p1, *p2;
  register unsigned char val1, val2;

  extern int dit_mat[256];
  extern int dit_mat2[256];

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;
  
  buf1 = pic_p->buf+x1+y1*pic_p->width;
  buf_offset = pic_p->width-(x2-x1+1);

  p2 = p1 = malloc((x2-x1+1)*(y2-y1+1)*scale*scale* sizeof(char));
  if (p1==NULL) return NULL;

  for (y=y1; y<=y2; y++)
    {
      for (i=0; i<scale; i++)
	{
	  buf2 = buf1;
	  for (x=x1; x<=x2; x++)
	    {
	      src = *buf2++;
	      val1 = GET_RED(src);
	      val1 = (val1>>5)+((val1&31)>dit_mat[(x&7)+8*(y&7)]);
	      if (val1>7) val1 = 7;
	      val2 = val1<<5;
	      val1 = GET_GREEN(src);
	      val1 = (val1>>5)+((val1&31)>dit_mat[(x&7)+8*(y&7)]);
	      if (val1>7) val1 = 7;
	      val2 |= val1<<2;
	      val1 = GET_BLUE(src);
	      val1 = (val1>>6)+((val1&63)>dit_mat2[(x&7)+8*(y&7)]);
	      if (val1>3) val1 = 3;
	      val2 |= val1;

	      for (j=0; j<scale; j++)
		{
		  *p2++ = val2;
		}
	    }
	}
      buf1 = buf2;
      buf1 += buf_offset;
    }

  return p1;
}


/* FUNCTION:     void init_12c()
   DESCRIPTION:  initialises 8x8 dither matrix for 12-bit color display
   conversion routine. This routine must be called before calling
   block_to_12gs!
   PARAMETERS:
   none
   RETURNS:      none
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
void init_12c()
{
  register int *p, x, y;
  extern int dit_mat[256];

  p = &dit_mat[0];
  
  for (y = 0; y<8; y++)
    for (x = 0; x<8; x++)
      *p++ = dither_value(x, y, 2);
}


/* FUNCTION:     char *block_to_12c (struct picture *pic_p, int x1, int y1,
   int x2, int y2, int scale)
   DESCRIPTION:  convert part of a block to 12-bit color pixmap data.
   Coordinates (x1, y1) and (x2, y2) are the opposite vertexes of the block.
   Assumes 4-bits for R, G and B components.
   PARAMETERS:
   struct picture pic_p pointer to picture struct
   int x1        
   int y1
   int x2
   int y2
   scale         scale value, 1=normal (1:1)
   RETURNS:      pointer to the char array which holds the converted picture
                 data. 1 pixel is 2 chars.
   
   Written by Petri Kuittinen, last modifications 19th August 1993
*/
char *block_to_12c (struct picture *pic_p, int x1, int y1, int x2, int y2,
		    int scale)
{
  register int x, y, buf_offset, i, j;
  register pixel *buf1, *buf2, src;
  register char *p1, *p2;
  register unsigned int val1, val2;

  extern int dit_mat[256];

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  /* Test out of screen conditions */
  if ((x1<0 && x2<0) || (x1>=pic_p->width && x2>=pic_p->width) ||
      (y1<0 && y2<0) || (y1>=pic_p->height && y2>=pic_p->height))
    return NULL;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=pic_p->width) x2 = pic_p->width-1;
  if (y2>=pic_p->height) y2 = pic_p->height-1;
  
  buf1 = pic_p->buf+x1+y1*pic_p->width;
  buf_offset = pic_p->width-(x2-x1+1);

  p2 = p1 = malloc(2*(x2-x1+1)*(y2-y1+1)*scale*scale*sizeof(char));
  if (p1==NULL) return NULL;

  for (y=y1; y<=y2; y++)
    {
      for (i=0; i<scale; i++)
	{
	  buf2 = buf1;
	  for (x=x1; x<=x2; x++)
	    {
	      src = *buf2++;
	      val1 = GET_RED(src);
	      val1 = (val1>>4)+((val1&15)>dit_mat[(x&7)+8*(y&7)]);
	      if (val1>15) val1 = 15;
	      val2 = val1<<8;
	      val1 = GET_GREEN(src);
	      val1 = (val1>>4)+((val1&15)>dit_mat[(x&7)+8*(y&7)]);
	      if (val1>15) val1 = 15;
	      val2 |= val1<<4;
	      val1 = GET_BLUE(src);
	      val1 = (val1>>4)+((val1&15)>dit_mat[(x&7)+8*(y&7)]);
	      if (val1>15) val1 = 15;
	      val2 |= val1;
	      for (j=0; j<scale; j++)
		{
		  *p2++ = val2>>8;
		  *p2++ = val2;
		}
	    }
	}
      buf1 = buf2;
      buf1 += buf_offset;
    }

  return p1;
}
