/*
 X_misc.c: Miscellaneous routines used in the X/Motif module. All the X 
           specific routines that are called from the main module are in this 
	   file. 

 $Header: X_misc.c,v 2.13 93/09/03 14:46:19 wet Exp $
 */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/ToggleB.h>
#include <stdlib.h>
#include <stdio.h>
#include "list.h"
#include "callb.h"
#include "tools1.h"
#include "X_misc.h"
#include "window.h"
#include "dialogw.h"
#include "bitmaps.h"

/* FUNCTION:       putimage(int x, int y, int width, int height,
                            struct view *view_p, char *data)
   DESCRIPTION:    Dumps the imagedata in char *data to the window in view.
   PARAMETERS:
   int x                Image x-coordinate.
   int y                Image y-coordinate.
   int width            Image width.
   int height           Image height.
   struct view *view_p  Pointer to view structure containing the widget the
                          image is supposed to be drawn.
   char *data           Image data.			  
   RETURNS:             Nothing.

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void putimage(int x, int y, int width, int height,
	      struct view *view_p, char *data)
{
  extern int depth;
  XImage *image;
  Display *disp;
  Window win;
  int scr;

  disp = XtDisplay(view_p->draw_area);
  win = XtWindow(view_p->draw_area);
  scr = DefaultScreen(disp);

  image = XCreateImage(disp, view_vis, depth, ZPixmap, 0, data,
		       width, height, pad, 0);
  XPutImage(disp, win, view_p->gc, image, 0, 0, x-view_p->x1*view_p->scale,
	    y-view_p->y1*view_p->scale, width, height);
  XFree(image);
}


/* FUNCTION:       handle_resize(struct view *view_p)
   DESCRIPTION:    Handles picture resize situations.
   PARAMETERS:
   struct view *view_p  Pointer to view structure.
   RETURNS:             Nothing.

   Written by Vesa T|rm{nen, last modifications 9th August 1993.
*/   
void handle_resize(struct view *view_p)
{
  /* destroy all X related stuff in the view struct and initilize them */
  XFreeGC(XtDisplay(view_p->draw_area), view_p->gc);
  XDestroyWindow(XtDisplay(view_p->main), view_p->busy_view);
  XtDestroyWidget(view_p->coord_ind);
  XtDestroyWidget(view_p->prim_ind);
  XtDestroyWidget(view_p->second_ind);
  XtDestroyWidget(view_p->h_scroll);
  XtDestroyWidget(view_p->v_scroll);
  XtDestroyWidget(view_p->zoom);
  XtDestroyWidget(view_p->draw_area);
  XtDestroyWidget(view_p->main);
  view_p->gc = (GC) NULL;
  view_p->busy_view = (Window) NULL;
  view_p->coord_ind = (Widget) NULL;
  view_p->prim_ind = (Widget) NULL;
  view_p->second_ind = (Widget) NULL;
  view_p->h_scroll = (Widget) NULL;
  view_p->v_scroll = (Widget) NULL;
  view_p->zoom = (Widget) NULL;
  view_p->draw_area = (Widget) NULL;
  view_p->main = (Widget) NULL;
  view_p->x1 = 0;
  view_p->y1 = 0;
  view_p->x2 = view_p->pic_p->width-1;
  view_p->y2 = view_p->pic_p->height-1;
  /* create new window */
  make_view_window(view_p);
}


/* FUNCTION:       void show_colors(struct view *view_p)
   DESCRIPTION:    Updates the view window color indicators.
   PARAMETERS:
   struct view *view_p  Pointer to view structure containing data about the
                          window and the colors.
   RETURNS:             Nothing.

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void show_colors(struct view *view_p)
{
  extern int depth;
  int r, g, b;
  GC prim_gc = XtGetGC(view_p->prim_ind, (XtGCMask)NULL, NULL);
  GC second_gc = XtGetGC(view_p->second_ind, (XtGCMask)NULL, NULL);
  XImage *prim_image, *second_image;
  Display *disp = XtDisplay(view_p->prim_ind);
  Window prim_win = XtWindow(view_p->prim_ind);
  Window second_win = XtWindow(view_p->second_ind);

  give_color(view_p, &r, &g, &b, PRIMARY);
  prim_image = XCreateImage(disp, view_vis, depth, ZPixmap, 0,
			    colored_block(r, g, b, ind_w, ind_h), 
			    ind_w, ind_h, pad, 0);
  XPutImage(disp, prim_win, prim_gc, prim_image, 0, 0, 0, 0,
	    ind_w, ind_h);

  give_color(view_p, &r, &g, &b, SECONDARY);
  second_image = XCreateImage(disp, view_vis, depth, ZPixmap,
			      0, colored_block(r, g, b, ind_w, ind_h), 
			      ind_w, ind_h, pad, 0);
  XPutImage(disp, second_win, second_gc, second_image, 0, 0, 0, 0,
	    ind_w, ind_h);

  XFree(prim_image);
  XFree(second_image);
}


/* FUNCTION:       show_state(char *msg)
   DESCRIPTION:    Displays info in the toolbox label, doesn't free the
                   character string.
   PARAMETERS:
   char *msg       Message string, maximum length xx
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 29th July 1993.
*/ 
void show_state(char *msg)
{
  Arg wargs[2];
  XmString xs = XmStringLtoRCreate(msg, XmSTRING_DEFAULT_CHARSET);

  XtSetArg(wargs[0], XmNlabelString, xs);
  XtSetValues(tool_label, wargs, 1);
  XmStringFree(xs);  
}


/* FUNCTION:       show_zoom(struct view *view_p)
   DESCRIPTION:    Shows information in the view window zoom label.
   PARAMETERS:
   struct view *view_p Pointer to view struct.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 5th August 1993.
*/ 
void show_zoom(struct view *view_p)
{
  Arg wargs[2];
  char zoom[10];
  XmString xs;
  sprintf(zoom, " %d00%% (%s)", view_p->scale, 
	  (is_touched(view_p)) ? "*" : " ");

  xs = XmStringLtoRCreate(zoom, XmSTRING_DEFAULT_CHARSET);
  XtSetArg(wargs[0], XmNlabelString, xs);
  XtSetValues(view_p->zoom, wargs, 1);
  XmStringFree(xs);  
}


/* FUNCTION:       kill_window(struct view *view_p)
   DESCRIPTION:    Kills the window in the view structure.
   PARAMETERS:
   struct view *view_p  Pointer to view structure containing data about the
                          window.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications th July 1993.
*/   
void kill_window(struct view *view_p)
{
  XFreeGC(XtDisplay(view_p->draw_area), view_p->gc);
  XDestroyWindow(XtDisplay(view_p->main), view_p->busy_view);
  XtDestroyWidget(view_p->coord_ind);
  XtDestroyWidget(view_p->prim_ind);
  XtDestroyWidget(view_p->second_ind);
  XtDestroyWidget(view_p->h_scroll);
  XtDestroyWidget(view_p->v_scroll);
  XtDestroyWidget(view_p->zoom);
  XtDestroyWidget(view_p->draw_area);
  XtDestroyWidget(view_p->main);
}



/* FUNCTION:       busy(void)
   DESCRIPTION:    Displays a busy cursor and blocks all events for the 
                   toplevel shell.
   PARAMETERS:
   void            None.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 14th August 1993.
*/
void busy(void)
{
  XMapWindow(XtDisplay(toplevel), busy_top);
}



/* FUNCTION:       busy_nomore(void)
   DESCRIPTION:    Return to normal operation after the busy mode.
   PARAMETERS:
   void            None.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 14th August 1993.
*/
void busy_nomore(void)
{
  XUnmapWindow(XtDisplay(toplevel), busy_top);
}


/* FUNCTION:       view_busy(struct view *view_p)
   DESCRIPTION:    Displays a busy cursor and blocks all events for a view
                   window.
   PARAMETERS:
   struct view *view_p Pointer to view structure.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 18th August 1993.
*/
void view_busy(struct view *view_p)
{
  XMapWindow(XtDisplay(view_p->main), view_p->busy_view);
}


/* FUNCTION:       view_busy_nomore(struct view *view_p)
   DESCRIPTION:    Return to normal operation after the busy mode.
   PARAMETERS:
   struct view *view_p Pointer to view structure.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 18th August 1993.
*/
void view_busy_nomore(struct view *view_p)
{
  XUnmapWindow(XtDisplay(view_p->main), view_p->busy_view);
}


/* FUNCTION:       change_cursor(struct view *view_p, int c)
   DESCRIPTION:    Changes the cursor in a view window.
   PARAMETERS:
   struct view *view_p Pointer to view structure.
   int c               Cursor to use.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 18th August 1993.
*/
void change_cursor(struct view *view_p, int c)
{
  switch (c)
    {
    case DRAW_C:
      XDefineCursor(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		    draw_c);
      break;
    case AIRBRUSH_C:
      XDefineCursor(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		    airbrush_c);
      break;
    case FILL_C:
      XDefineCursor(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		    fill_c);
      break;
    case SELECT_C:
      XDefineCursor(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		    select_c);
      break;
    case EXTRACT_C:
      XDefineCursor(XtDisplay(view_p->draw_area), XtWindow(view_p->draw_area),
		    colorext_c);
      break;
    }
  XmUpdateDisplay(view_p->draw_area);
}



/* FUNCTION:       ask_question(char *text, char *ok_label, char *cancel_label,
                                void (*func)(int))
   DESCRIPTION:    Manages a question dialog with the char *parameters as
                   labels and sets question_func to point to func. This 
		   function is then called from the dialog callback in 
		   callb.c.
   PARAMETERS:
   char *text         Dialog text.
   char *ok_label     Dialog ok button label.
   char *cancel_label Dialog cancel button label.
   void (*func)(int)) Function pointer of the function that will be called
                        when the user presses either button. The parameter
			identifies the button (B_OK, B_CANCEL).
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications th July 1993.
*/   
void ask_question(char *text, char *ok_label, char *cancel_label,
		  void (*func)(int))
{
  XmString question, cancel, ok;
  Arg wargs[5];
  Cardinal n;

  question = XmStringCreateLtoR(text, XmSTRING_DEFAULT_CHARSET);
  ok = XmStringCreateLtoR(ok_label, XmSTRING_DEFAULT_CHARSET);
  cancel = XmStringCreateLtoR(cancel_label, XmSTRING_DEFAULT_CHARSET);
  question_func = func;
  n = 0;
  XtSetArg(wargs[n], XmNmessageString, question); n++;
  XtSetArg(wargs[n], XmNokLabelString, ok); n++;
  XtSetArg(wargs[n], XmNcancelLabelString, cancel); n++;
  XtSetValues(question_dialog, wargs, n);
      
  XmStringFree(question);
  XmStringFree(ok);
  XmStringFree(cancel);
  
  center_window(question_dialog);      
  XtManageChild(question_dialog);      
}

/* FUNCTION:       ask_question3(char *text, char *ok_label, char *no_label,
                                 char *cancel_label, void (*func)(int))
   DESCRIPTION:    Manages a question dialog with the char *parameters as
                   labels and sets question_func to point to func. This 
		   function is then called from the dialog callback in 
		   callb.c.
   PARAMETERS:
   char *text         Dialog text.
   char *ok_label     Dialog ok button label.
   char *no_label     Dialog no button label.
   char *cancel_label Dialog cancel button label.
   void (*func)(int)) Function pointer of the function that will be called
                        when the user presses either button. The parameter
			identifies the button (B_OK, B_NO, B_CANCEL).
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 4th August 1993.
*/   
void ask_question3(char *text, char *ok_label, char *no_label, 
		   char *cancel_label, void (*func)(int))
{
  XmString question, ok, no, cancel;
  Arg wargs[5];
  Cardinal n;
  if (text!=NULL)
    {
      question = XmStringCreateLtoR(text, XmSTRING_DEFAULT_CHARSET);
      ok = XmStringCreateLtoR(ok_label, XmSTRING_DEFAULT_CHARSET);
      no = XmStringCreateLtoR(no_label, XmSTRING_DEFAULT_CHARSET);
      cancel = XmStringCreateLtoR(cancel_label, XmSTRING_DEFAULT_CHARSET);
      question_func = func;
      n = 0;
      XtSetArg(wargs[n], XmNmessageString, question); n++;
      XtSetArg(wargs[n], XmNokLabelString, ok); n++;
      XtSetArg(wargs[n], XmNcancelLabelString, no); n++;
      XtSetArg(wargs[n], XmNhelpLabelString, cancel); n++;
      XtSetValues(question3_dialog, wargs, n);
      
      XmStringFree(question);
      XmStringFree(ok);
      XmStringFree(no);
      XmStringFree(cancel);
      
      center_window(question3_dialog);      
      XtManageChild(question3_dialog);      
    }
  else
    (*func)(B_EMPTY);
}

/* FUNCTION:       show_errormsg(char *text, char *ok_label,
                                 void (*func)(int))
   DESCRIPTION:    Manages an error dialog with the char *parameters as
                   labels and sets error_func to point to func. This 
		   function is then called from the dialog callback in 
		   callb.c.
   PARAMETERS:
   char *text         Dialog text.
   char *ok_label     Dialog ok button label.
   char *cancel_label Dialog cancel button label.
   void (*func)(int)) Function pointer of the function that will be called
                        when the user presses the ok button. The parameter
			identifies the button (B_OK).
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications th July 1993.
*/   
void show_errormsg(char *text, char *ok_label, void (*func)(int))
{
  XmString error, ok;
  Arg wargs[5];
  Cardinal n;
  error = XmStringCreateLtoR(text, XmSTRING_DEFAULT_CHARSET);
  ok = XmStringCreateLtoR(ok_label, XmSTRING_DEFAULT_CHARSET);
  error_func = func;
  n = 0;
  XtSetArg(wargs[n], XmNmessageString, error); n++;
  XtSetArg(wargs[n], XmNokLabelString, ok); n++;
  XtSetValues(error_dialog, wargs, n);

  XmStringFree(error);
  XmStringFree(ok);

  center_window(error_dialog);      
  XtManageChild(error_dialog);      
}


/* FUNCTION:       ask_paste(void (*func)(int, int))
   DESCRIPTION:    Pops up a dialog that queries the user about the brush
                   location in the toolbox user brushes.
   PARAMETERS:
   void (*func)(int, int) Function in the main module to be called after
                            the user presses aither Paste or Cancel. The
			    first argument is the brush #define in tools.h
			    and the and the other is either B_OK or B_CANCEL,
			    #defined in X_misc.h
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 13th July 1993.
*/   
void ask_paste(void (*func)(int, int))
{
  /* position #1 is the default */
  ptool = U1;
  XmToggleButtonSetState(ptool_btns[U1], True, False);
  XmToggleButtonSetState(ptool_btns[U2], False, False);
  XmToggleButtonSetState(ptool_btns[U3], False, False);

  paste_func = func;
  center_window(ed_ptool_dialog);
  XtManageChild(ed_ptool_dialog);
}



/* FUNCTION:       deactivate_menu_entry(char *entry)
   DESCRIPTION:    Greys out a menu entry with label entry.
   PARAMETERS:
   char *entry     Menu button label. (entries in menu.h)
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 19th July 1993.
*/   
void deactivate_menu_entry(char *entry)
{ 
  Arg wargs[2];
  Widget button = get_menu_widget(entry);
  XtSetArg(wargs[0], XmNsensitive, FALSE);
  XtSetValues(button, wargs, 1);
}


/* FUNCTION:       activate_menu_entry(char *entry)
   DESCRIPTION:    Makes a menu entry with label entry active again.
   PARAMETERS:
   char *entry     Menu button label. (entries in menu.h)
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 19th July 1993.
*/   
void activate_menu_entry(char *entry)
{
  Arg wargs[2];
  Widget button = get_menu_widget(entry);
  XtSetArg(wargs[0], XmNsensitive, TRUE);
  XtSetValues(button, wargs, 1);
}


/* FUNCTION:       change_menu_entry(char *old_entry, char *new_entry)
   DESCRIPTION:    Changes the label of a menu button, used with undo
                   menu entry.
   PARAMETERS:
   char *old_entry Menu button label to be changed. (entries in menu.h)
   char *new_entry New button label.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 19th July 1993.
*/   
void change_menu_entry(char *old_entry, char *new_entry)
{
  Arg wargs[2];
  XmString new_label = XmStringCreateLtoR(new_entry, XmSTRING_DEFAULT_CHARSET);
  Widget button = get_menu_widget(old_entry);
  XtSetArg(wargs[0], XmNlabelString, new_label);
  XtSetValues(button, wargs, 1);
}



/* FUNCTION:       center_window(Widget widget_id)
   DESCRIPTION:    Adjusts widget resouces so that the window is located in
                   center of the screen, doesn't really work with message-
		   boxes.
   PARAMETERS:
   Widget widget_id Widget_id of the shell to be realized.
   RETURNS:        Nothing.

   Written by Vesa T|rm{nen, last modifications 3rd August 1993.
*/   
void center_window(Widget widget_id)
{
  Arg wargs[10];
  Cardinal n;
  Display *disp = XtDisplay(widget_id);
  int scr = DefaultScreen(disp);
  Dimension w, h;
  unsigned int x, y;

  /* get the dialog dimensions */
  n = 0; 
  XtSetArg(wargs[n], XmNwidth, &w); n++;
  XtSetArg(wargs[n], XmNheight, &h); n++;
  XtGetValues(widget_id, wargs, n);
  
  /* count the center if previous XGetValues was succesful, otherwise use
     fixed values */
  x = XDisplayWidth(disp, scr)/2-((w!=0) ? w/2 : 150);
  y = XDisplayHeight(disp, scr)/2-((h!=0) ? h/2 : 50);

  n = 0;
  XtSetArg(wargs[n], XmNdefaultPosition, FALSE); n++;
  XtSetArg(wargs[n], XtNx, x); n++; 
  XtSetArg(wargs[n], XtNy, y); n++;
  XtSetValues(widget_id, wargs, n);
}



/***********************************************************************
 * Retrieve the character string from a possibly composite compound string,
 * such  as created by xs_concat_words()
 *         From:
 *                   The X Window System, 
 *            Programming and Applications with Xt
 *                   OSF/Motif Edition
 *         by
 *                Douglas Young
 *              Prentice Hall, 1990
 *
 *                 Example described on pages:  ??
 *
 *
 *  Copyright 1989 by Prentice Hall
 *  All Rights Reserved
 *
 * This code is based on the OSF/Motif widget set and the X Window System
 *
 * Permission to use, copy, modify, and distribute this software for 
 * any purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation.
 *
 * Prentice Hall and the author disclaim all warranties with regard to 
 * this software, including all implied warranties of merchantability and fitne
 ss.
 * In no event shall Prentice Hall or the author be liable for any special,
 * indirect or cosequential damages or any damages whatsoever resulting from 
 * loss of use, data or profits, whether in an action of contract, negligence 
 * or other tortious action, arising out of or in connection with the use 
 * or performance of this software.
 *
 * Open Software Foundation is a trademark of The Open Software Foundation, Inc
 .
 * OSF is a trademark of Open Software Foundation, Inc.
 * OSF/Motif is a trademark of Open Software Foundation, Inc.
 * Motif is a trademark of Open Software Foundation, Inc.
 * DEC is a registered trademark of Digital Equipment Corporation
 * HP is a registered trademark of the Hewlett Packard Company
 * DIGITAL is a registered trademark of Digital Equipment Corporation
 * X Window System is a trademark of the Massachusetts Institute of Technology
 *****************************************************************************/
char *xs_get_string_from_xmstring(XmString string) 
{ 
  caddr_t context; 
  char *text; 
  XmStringCharSet charset; 
  XmStringDirection dir; 
  Boolean separator; 
  char *buf = NULL; 
  int done = FALSE;
  
  XmStringInitContext(&context, string);
  while (!done)
    if(XmStringGetNextSegment (context, &text, &charset,
			       &dir, &separator)){ 
      if(separator) /* Stop when next segment is a separator */
        done = TRUE;
      
      if(buf){
        buf = XtRealloc(buf, strlen(buf) + strlen(text) + 2);
        strcat(buf, text);                      
      }
      else {
        buf = (char *) XtMalloc(strlen(text) +1);
        strcpy(buf, text);
      }
      XtFree(text);
    }
    else
      done = TRUE;
  
  XmStringFreeContext (context);
  return buf;
}



/* FUNCTION:       get_global_bounds (Widget widget_id, 
                                      unsigned int *width_return,
				      unsigned int *height_return)
   DESCRIPTION:    Calculate total window width with the window manager
                   decorations included. Doesn't work with virtual desktops
		   (tvtwm).
   PARAMETERS:
   Widget widget_id            Widget id of the window, whose boundaries are
                                 calculated.
   unsigned int *width_return  Returned width.
   unsigned int *height_return Returned height.
   RETURNS:        Width and height of the total window area in pointer
                   arguments, type is void.
   Written by Henry Michael Lassen <henryml@daimi.aau.dk>, last modifications
   (by Vesa T|rm{nen) 17th July 1993.
*/   
void get_global_bounds (Widget widget_id, unsigned int *width_return,
			unsigned int *height_return)
{
  XSetWindowAttributes attr;
  Display *disp;
  unsigned int nchildren;
  Window *children;
  Window win;
  Window parent;
  Window root;
  Window whole;
  unsigned int default_depth;
  int x;
  int y;
  unsigned int width;
  unsigned int height;
  unsigned int border;

  disp = XtDisplay(widget_id);
  win = XtWindow(widget_id);
  attr.override_redirect = True;
  XChangeWindowAttributes(disp, win,CWOverrideRedirect, &attr);
  whole = win;
  XQueryTree(disp, win, &root, &parent, &children, &nchildren);
  while (parent!=root) {
    whole = parent;
    XQueryTree(disp,whole, &root, &parent, &children, &nchildren);
  }
  XGetGeometry(disp, whole, &root, &x, &y, &width, &height, &border,
               &default_depth);
  *width_return = width + 2*border;
  *height_return = height + 2*border;
  attr.override_redirect = False;
  XChangeWindowAttributes(disp, win,CWOverrideRedirect, &attr);
}


