 /*
  * Khoros: $Id: lvconvert.c,v 1.3 1992/03/20 23:04:59 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lvconvert.c,v 1.3 1992/03/20 23:04:59 dkhoros Exp $";
#endif

 /*
  * $Log: lvconvert.c,v $
 * Revision 1.3  1992/03/20  23:04:59  dkhoros
 * VirtualPatch5
 *
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1992, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1992 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvconvert.c
 >>>>
 >>>>      Program Name: vconvert
 >>>>
 >>>> Date Last Updated: Sun Feb 23 11:51:25 1992 
 >>>>
 >>>>          Routines: lvconvert - the library call for vconvert
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvconvert - library call for vconvert
*
* Purpose:
*    
*    Convert a Khoros image from one data storage type to another
*    
*    

* Input:
*    
*    image          pointer to xvimage structure to be processed
*    
*    k              VIFF data storage type to convert input image into
*    
*    sflg           if  non-zero  then  perform  scaling   using   the
*                   scale_factor
*    
*    nflg           if  non-zero   then   perform   normalization   to
*                   norm_factor
*    
*    norm_factor    value to normalize output image data to
*    
*    scale_factor   value to scale image data by
*    
*    bflg           logic switch to request  each  band  be  processed
*                   independently
*    
*    

* Output:
*    
*    image          receives the output image after processing
*    
*    Return Value   1 on success, 0 on failure.
*    
*    

*
* Written By: Scott Wilson
*    
*            Scott Wilson and Tom Sauer 11/23/88
*                Added cmap type checking and support for color models other
*                than VFF_CM_GREY.
*            Tom Sauer -- Tue Dec 27 15:15:26 MST 1988
*                Updated lvcset_sizes to handle generic color models
*            Scott Wilson -- updated to XV III 18-Mar-89
*            Mark Young and Tait Cyrus 11/9/89
*                lvconvert was not converting BIT images correctly by not
*                incrementing past the padding at the end of each scanline.
*            Scott Wilson -- 8-Feb-90
*                Fixed bug that caused BIT to * to ignore scale and
*                normalization factors.
*            Scott Wilson -- 6-Jun-90
*                Fixed missing free() calls
*            Scott Wilson -- 15-Aug-90
*                Renamed lvconvert as lvcvt, for use as a toolbox routine for
*                the new lvconvert, which is new code that can do multiband
*                processing with each band being normalized or scaled independently.*    
*                If the band processing is chosen to be global, then it just falls
*                through to the old lvconvert code.
*            Scott Wilson -- 8-Oct-90
*                Modified the normalization procedure so that if the max is zero
*                then you get back a zero image. Before, you would get an error
*                message and no image. This really screws up editimage.
*            Jeremy Worley Tue Feb 12 14:32:01 MST 1991
*                Added support for double and double complex data types.
*            Jeremy Worley Fri Feb 15 16:24:36 MST 1991
*                complex to complex case indexing bug fixed.  It was indexing (i,2*i*    +1)
*                when it should have been indexing (i,i+1)
*            Jeremy Worley Sun Feb 23 11:50:03 MST 1992
*                fixed scale_factor usage so that is is set to 1.0 is sflg is 0.
*                also fixed messages that indicate invalid source or destination
*                data types.
*    
*    

****************************************************************/


/* -library_def */
int
lvconvert(image,k,nflg,sflg,norm_factor,scale_factor,bflg)
struct xvimage *image;
int  k,nflg,sflg,bflg;
float norm_factor,scale_factor;
/* -library_def_end */

/* -library_code */
  {
    struct xvimage *it;  /* Work image area for multiband */
    int i;
    int insize,outsize,trash;
    char *data;

    if (bflg)
      {
        /* User requests independent band processing */
        /* Fix up a 1-band image for a work area */
        it = (struct xvimage *)malloc(sizeof(struct xvimage));
        if (it == NULL)
          {
            fprintf(stderr,"lvconvert: Unable to allocate work area\n");
            return(0);
          }
        copyheader(image,it);
        it->num_data_bands = 1;

        /* Attach space for one band only */
        if (!imagesize(image,&insize,&trash,&trash,&trash,&trash,&trash))
          {
            fprintf(stderr,"lvconvert: Unknown image configuration\n");
            freeimage(it);
            return(0);
          }
        insize /= image->num_data_bands;
        it->data_storage_type = k;
        if (!imagesize(it,&outsize,&trash,&trash,&trash,&trash,&trash))
          {
            fprintf(stderr,"lvconvert: Unknown image configuration\n");
            freeimage(it);
            return(0);
          }
        it->data_storage_type = image->data_storage_type;
        data = malloc(outsize*image->num_data_bands);
        if (data == NULL)
          {
            fprintf(stderr,"lvconvert: Not enough space for new image\n");
            freeimage(it);
            return(0);
          }
        
        /* Process each band separately */
        for (i=0; i<image->num_data_bands; i++)
          {
            /* Copy the i'th band into the work image */
            it->imagedata = malloc(insize);
            if (it->imagedata == NULL)
              {
                fprintf(stderr,"lvconvert: Unable to allocate work data area\n");
                freeimage(it);
                return(0);
              }
            bcopy(image->imagedata+i*insize,it->imagedata,insize);     
            lvcvt(it,k,nflg,sflg,norm_factor,scale_factor);
            bcopy(it->imagedata,data+i*outsize,outsize);
            it->data_storage_type = image->data_storage_type;
          }
        free(image->imagedata);
        image->imagedata = data;
        image->data_storage_type = k;
        freeimage(it);
        return(1);
      }
    else
      {
        /* User requests processing all bands at once */
        if (!lvcvt(image,k,nflg,sflg,norm_factor,scale_factor))
          {
            fprintf(stderr,"lvconvert: Unable to convert image\n");
            return(0);
          }
        return(1);
      }
  }
            
int
lvcvt(image,k,nflg,sflg,norm_factor,scale_factor)
struct xvimage *image;
int  k,nflg,sflg;
float norm_factor,scale_factor;
{
    register int i, j, it;
    register short s;
    register float f;
    register unsigned char c;
    int    bmsize,rows,cols,r;
    int    *iptr,*iptr1,imax,imin;
    short  *sptr,*sptr1,smax,smin;
    float  *fptr,*fptr1,fmax,fmin;
    double *dptr; /* new for double conversions */
    unsigned char *bptr,*b,bit;
    unsigned char *cptr,*cptr1,cmax;
    int  size,sz;
    float  multiplier=1.0; /* initial value; jw */
    char   *malloc();
    int really_to_double; /* new for double conversions */
    rows = image->col_size;
    cols = image->row_size;
    size = rows*cols;

/*
** this mod to allow for double and double complex...
*/

    if(image->data_storage_type==VFF_TYP_DOUBLE){
       if((fptr = (float *)malloc(size*image->num_data_bands
           *sizeof(float)))==NULL){
          fprintf(stderr,"vconvert:  Can't malloc memory..\n");
          return(0);
       }
       dptr = (double*)(image->imagedata);
       for(i=0;i<size*image->num_data_bands;i++)fptr[i] = (float)dptr[i];  
       free((char *)dptr);
       image->imagedata = (char *)fptr;
       image->data_storage_type = VFF_TYP_FLOAT;
    }else if(image->data_storage_type==VFF_TYP_DCOMPLEX){
       if((fptr = (float *)malloc(2*size*image->num_data_bands
           *sizeof(float)))==NULL){
          fprintf(stderr,"vconvert:  Can't malloc memory..\n");
          return(0);
       }
       dptr = (double*)(image->imagedata);
       for(i=0;i<2*size*image->num_data_bands;i++)fptr[i] = (float)dptr[i];
       image->imagedata = (char *)fptr;
       image->data_storage_type = VFF_TYP_COMPLEX;
    }

    if(k==VFF_TYP_DOUBLE){
       k = VFF_TYP_FLOAT;
       really_to_double = TRUE;
    }else if(k==VFF_TYP_DCOMPLEX){
       k = VFF_TYP_COMPLEX;
       really_to_double = TRUE;
    }else{
       really_to_double = FALSE;
    }
 
/*
** the original lvcvt code ...
*/

    switch(k)
      {
        case VFF_TYP_BIT : /* Target is bit */
          {
            switch(image->data_storage_type)
              {
                case VFF_TYP_BIT: /* Bit to Bit */
                  {
                    /* NOTHING TO CONVERT! */
                    break;
                  }
                case VFF_TYP_1_BYTE: /* CHAR to bit */
                  {
                    if (image->map_scheme != VFF_MS_NONE)
                      {
                        fprintf(stderr,"lvconvert: Source must be unmapped for conversion to BIT target\n");
                        return(0);
                      }
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    bmsize = ((cols + 7)/8)*r;
                    bptr = (unsigned char *)malloc((unsigned int)bmsize);
                    b = bptr;
                    cptr = (unsigned char *)(image->imagedata);
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to set or clear */
                            if (*cptr++ > 127) *b = *b | bit;
                            else *b = *b & ~bit;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                          {
                            for (j=k; j<8; j++) *b = *b & (~(1 << j));
                            b++;
                          }
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)bptr;
                    image->data_storage_type =  VFF_TYP_BIT;
                    break;
                  }
                case VFF_TYP_2_BYTE: /* SHORT to bit */
                  {
                    if (image->map_scheme != VFF_MS_NONE)
                      {
                        fprintf(stderr,"lvconvert: Source must be unmapped for conversion to BIT target\n");
                        return(0);
                      }
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    bmsize = ((cols + 7)/8)*r;
                    bptr = (unsigned char *)malloc((unsigned int)bmsize);
                    b = bptr;
                    sptr = (short *)(image->imagedata);
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to set or clear */
                            if (*sptr++) *b = *b | bit;
                            else *b = *b & ~bit;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                          {
                            for (j=k; j<8; j++) *b = *b & (~(1 << j));
                            b++;
                          }
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)bptr;
                    image->data_storage_type =  VFF_TYP_BIT;
                    break;
                  }
                case VFF_TYP_4_BYTE: /* INT to bit */
                  {
                    if (image->map_scheme != VFF_MS_NONE)
                      {
                        fprintf(stderr,"lvconvert: Source must be unmapped for conversion to BIT target\n");
                        return(0);
                      }
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    bmsize = ((cols + 7)/8)*r;
                    bptr = (unsigned char *)malloc((unsigned int)bmsize);
                    b = bptr;
                    iptr = (int *)(image->imagedata);
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to set or clear */
                            if (*iptr++) *b = *b | bit;
                            else *b = *b & ~bit;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                          {
                            for (j=k; j<8; j++) *b = *b & (~(1 << j));
                            b++;
                          }
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)bptr;
                    image->data_storage_type =  VFF_TYP_BIT;
                    break;
                  }
                case VFF_TYP_FLOAT: /* FLOAT to bit */
                  {
                    if (image->map_scheme != VFF_MS_NONE)
                      {
                        fprintf(stderr,"lvconvert: Source must be unmapped for conversion to BIT target\n");
                        return(0);
                      }
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    bmsize = ((cols + 7)/8)*r;
                    bptr = (unsigned char *)malloc((unsigned int)bmsize);
                    b = bptr;
                    fptr = (float *)(image->imagedata);
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to set or clear */
                            if (*fptr++) *b = *b | bit;
                            else *b = *b & ~bit;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                          {
                            for (j=k; j<8; j++) *b = *b & (~(1 << j));
                            b++;
                          }
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)bptr;
                    image->data_storage_type =  VFF_TYP_BIT;
                    break;
                  }
                case VFF_TYP_COMPLEX: /* COMPLEX to bit */
                  {
                    if (image->map_scheme != VFF_MS_NONE)
                      {
                        fprintf(stderr,"lvconvert: Source must be unmapped for conversion to BIT target\n");
                        return(0);
                      }
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    bmsize = ((cols + 7)/8)*r;
                    bptr = (unsigned char *)malloc((unsigned int)bmsize);
                    b = bptr;
                    fptr = (float *)(image->imagedata);
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to set or clear */
                            if (*fptr) *b = *b | bit; /* REAL PART ONLY! */
                            else *b = *b & ~bit;
                            fptr++; fptr++;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                          {
                            for (j=k; j<8; j++) *b = *b & (~(1 << j));
                            b++;
                          }
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)bptr;
                    image->data_storage_type =  VFF_TYP_BIT;
                    break;
                  }
                default :
                  {
                    (void) fprintf(stderr,"lvconvert: unknown source conversion type\n");
                    return(0);
                    /* break; */
                  }
              } /* switch */
            break;
          }
        case VFF_TYP_1_BYTE : /* Target is char */
          {
            switch(image->data_storage_type)
              {
                case VFF_TYP_BIT : /* Bit to char */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    cptr = (unsigned char *)malloc(sz);
                    cptr1 = cptr;
                    bptr = (unsigned char *)image->imagedata;
                    b = bptr;
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to look at */
                            if (*b & bit) 
                              {
                                if (sflg) *cptr1++ = scale_factor;
                                else if (nflg) *cptr1++ = norm_factor;
                                else *cptr1++ = 255;
                              }
                            else *cptr1++ = 0;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                            b++;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)cptr;
                    image->data_storage_type =  VFF_TYP_1_BYTE;
                    image->map_scheme =  VFF_MS_NONE;
                    break;
                  }
                case VFF_TYP_1_BYTE : /* Char to char */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    cptr = (unsigned char *)malloc(sz);
                    if (nflg)              /* Get normalization factor */
                      {
                        if (norm_factor > 255)
                          {  
                            (void) fprintf(stderr,"lvconvert: Normalization target too large!\n");
                            return(0);
                          }
                        cmax = 0;
                        for (i=0; i<sz; i++)
                          {
                            c = ((unsigned char *)(image->imagedata))[i];
                            if (c > cmax) cmax = c;
                          }
                        if (cmax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/cmax; 
                      }
                    else
                      {
                           multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0; i<sz; i++)
                      {
                            c = ((unsigned char *)(image->imagedata))[i];
                            cptr[i] = c*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)cptr;
                    image->data_storage_type =  VFF_TYP_1_BYTE;
                    break;
                  }
                case VFF_TYP_2_BYTE : /* short to char */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    cptr = (unsigned char *)malloc(sz);
                    smin = 0;
                    if (nflg)              /* Get normalization factor */
                      {
                        if (norm_factor > 255)
                          {  
                            (void) fprintf(stderr,"lvconvert: Normalization target too large!\n");
                            return(0);
                          }
                        smax = ((short *)(image->imagedata))[0];
                        smin = smax;
                        for (i=0; i<sz; i++)
                          {
                            s = ((short *)(image->imagedata))[i];
                            if (s > smax) smax = s;
                            else if (s < smin) smin = s;
                          }
                        if (smin == 0 && smax == 0) multiplier = 0.0;
                        else if (smin == smax)
                          {
                            multiplier = norm_factor/smin;
                            smin = 0;
                          }
                        else multiplier = norm_factor/(smax-smin); 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0; i<sz; i++)
                      {
                        s = ((short *)(image->imagedata))[i];
                        s = (s-smin)*multiplier;
                        if (s > 255) s = 255;
                        else if (s < 0) s = 0;
                        cptr[i] = s;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)cptr;
                    image->data_storage_type =  VFF_TYP_1_BYTE;
                    break;
                  }
                case VFF_TYP_4_BYTE : /* int to char */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    cptr = (unsigned char *)malloc(sz);
                    imin = 0;
                    if (nflg)              /* Get normalization factor */
                      {
                        if (norm_factor > 255)
                          {  
                            (void) fprintf(stderr,"lvconvert: Normalization target too large!\n");
                            return(0);
                          }
                        imax = ((int *)(image->imagedata))[0];
                        imin = imax;
                        for (i=0; i<sz; i++)
                          {
                            it = ((int *)(image->imagedata))[i];
                            if (it > imax) imax = it;
                            else if (it < imin) imin = it;
                          }
                        if (imin == 0 && imax == 0) multiplier = 0.0;
                        else if (imin == imax)
                          {
                            multiplier = norm_factor/imin;
                            imin = 0;
                          }
                        else multiplier = norm_factor/(imax-imin); 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        it = ((int *)(image->imagedata))[i];
                        it = (it-imin)*multiplier;
                        if (it > 255) it = 255;
                        else if (it < 0) it = 0;
                        cptr[i] = it;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)cptr;
                    image->data_storage_type =  VFF_TYP_1_BYTE;
                    break;
                  }
                case VFF_TYP_FLOAT : /* float to char */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    cptr = (unsigned char *)malloc(sz);
                    fmin = 0.0;
                    if (nflg)              /* Get normalization factor */
                      {
                        if (norm_factor > 255)
                          {  
                            (void) fprintf(stderr,"lvconvert: Normalization target too large!\n");
                            return(0);
                          }
                        fmax = ((float *)(image->imagedata))[0];
                        fmin = fmax;
                        for (i=0; i<sz; i++)
                          {
                            f = ((float *)(image->imagedata))[i];
                            if (f > fmax) fmax = f;
                            else if (f < fmin) fmin = f;
                          }
                        if (fmin == 0.0 && fmax == 0.0) multiplier = 0.0;
                        else if (fmin == fmax)
                          {
                            multiplier = norm_factor/fmin;
                            fmin = 0;
                          }
                        else multiplier = norm_factor/(fmax-fmin); 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        f = ((float *)(image->imagedata))[i];
                        f = (f-fmin)*multiplier;
                        if (f > 255.0) f=255.0;
                        else if (f < 0.0) f=0.0;
                        cptr[i] = f;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)cptr;
                    image->data_storage_type =  VFF_TYP_1_BYTE;
                    break;
                  }
                case VFF_TYP_COMPLEX : /* complex to char */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    cptr = (unsigned char *)malloc(sz);
                    fmin = 0.0;
                    if (nflg)              /* Get normalization factor */
                      {
                        if (norm_factor > 255)
                          {  
                            (void) fprintf(stderr,"lvconvert: Normalization target too large!\n");
                            return(0);
                          }
                        fmax = ((float *)(image->imagedata))[0];
                        fmin = fmax;
                        for (i=0; i<2*sz; i += 2)
                          {
                            f = ((float *)(image->imagedata))[i];
                            if (f > fmax) fmax = f;
                            else if (f < fmin) fmin = f;
                          }
                        if (fmin == 0.0 && fmax == 0.0) multiplier = 0.0;
                        else if (fmin == fmax)
                          {
                            multiplier = norm_factor/fmin;
                            fmin = 0;
                          }
                        else multiplier = norm_factor/(fmax-fmin); 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }

                    for (i=0,j=0; i<2*sz; i += 2)
                      {
                            f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY */
                            f = (f-fmin)*multiplier;
                            if (f > 255.0) f=255.0;
                            else if (f < 0.0) f=0.0;
                            cptr[j++] = (unsigned char)f;
                          }
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)cptr;
                    image->data_storage_type =  VFF_TYP_1_BYTE;
                    break;
                default :
                  {
                    (void) fprintf(stderr,"lvconvert: unknown source conversion type\n");
                    return(0);
                    /* break; */
                  }
              } /* switch */
            break;
          } /* case */
        case VFF_TYP_2_BYTE: /* Target is short */
          {
            switch(image->data_storage_type)
              {
                case VFF_TYP_BIT : /* Bit to short */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    sptr = (short *)malloc(sz*sizeof(short));
                    sptr1 = sptr;
                    bptr = (unsigned char *)image->imagedata;
                    b = bptr;
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to look at */
                            if (*b & bit) 
                              {
                                if (sflg) *sptr1++ = scale_factor;
                                else if (nflg) *sptr1++ = norm_factor;
                                else *sptr1++ = 1;
                              }
                            else *sptr1++ = 0;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                            b++;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)sptr;
                    image->data_storage_type =  VFF_TYP_2_BYTE;
                    image->map_scheme =  VFF_MS_NONE;
                    break;
                  }
                case VFF_TYP_1_BYTE : /* Char to short */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    sptr = (short *)malloc(sz*sizeof(short));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        cmax = 0;
                        for (i=0; i<sz; i++)
                          {
                            c = ((unsigned char *)(image->imagedata))[i];
                            if (c > cmax) cmax = c;
                          }
                        if (cmax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/cmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        c = ((unsigned char *)(image->imagedata))[i];
                        sptr[i] = multiplier*c;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)sptr;
                    image->data_storage_type =  VFF_TYP_2_BYTE;
                    break;
                  }
                case VFF_TYP_2_BYTE : /* Short to short */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    sptr = (short *)malloc(sz*sizeof(short));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        smax = ((short *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            s = ((short *)(image->imagedata))[i];
                            if (s > smax) smax = s;
                          }
                        if (smax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/smax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        s = ((short *)(image->imagedata))[i];
                        sptr[i] = s*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)sptr;
                    image->data_storage_type =  VFF_TYP_2_BYTE;
                    break;
                  }
                case VFF_TYP_4_BYTE : /* int to short */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    sptr = (short *)malloc(sz*sizeof(short));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        imax = ((int *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            it = ((int *)(image->imagedata))[i];
                            if (it > imax) imax = it;
                          }
                        if (imax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/imax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        it = ((int *)(image->imagedata))[i];
                        sptr[i] = it*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)sptr;
                    image->data_storage_type =  VFF_TYP_2_BYTE;
                    break;
                  }
                case VFF_TYP_FLOAT : /* float to short */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    sptr = (short *)malloc(sz*sizeof(short));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            f = ((float *)(image->imagedata))[i];
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        f = ((float *)(image->imagedata))[i];
                        sptr[i] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)sptr;
                    image->data_storage_type =  VFF_TYP_2_BYTE;
                    break;
                  }
                case VFF_TYP_COMPLEX : /* complex to short */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    sptr = (short *)malloc(sz*sizeof(short));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz*2; i += 2)
                          {
                            f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY! */
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0,j=0; i<sz*2; i+= 2)
                      {
                        f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY */
                        sptr[j++] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)sptr;
                    image->data_storage_type =  VFF_TYP_2_BYTE;
                    break;
                  }
                default :
                  {
                    (void) fprintf(stderr,"lvconvert: unknown source conversion type\n");
                    return(0);
                    /* break; */
                  }
              } /* switch */
            break;
          } /* case */
        case VFF_TYP_4_BYTE: /* Target is int */
          {
            switch(image->data_storage_type)
              {
                case VFF_TYP_BIT : /* Bit to int */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    iptr = (int *)malloc(sz*sizeof(int));
                    iptr1 = iptr;
                    bptr = (unsigned char *)image->imagedata;
                    b = bptr;
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to look at */
                            if (*b & bit) 
                              {
                                if (sflg) *iptr1++ = scale_factor;
                                else if (nflg) *iptr1++ = norm_factor;
                                else *iptr1++ = 1;
                              }
                            else *iptr1++ = 0;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                            b++;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)iptr;
                    image->data_storage_type =  VFF_TYP_4_BYTE;
                    image->map_scheme =  VFF_MS_NONE;
                    break;
                  }
                case VFF_TYP_1_BYTE : /* Char to int */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    iptr = (int *)malloc(sz*sizeof(int));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        cmax = 0;
                        for (i=0; i<sz; i++)
                          {
                            c = ((unsigned char *)(image->imagedata))[i];
                            if (c > cmax) cmax = c;
                          }
                        if (cmax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/cmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        c = ((unsigned char *)(image->imagedata))[i];
                        iptr[i] = multiplier*c;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)iptr;
                    image->data_storage_type =  VFF_TYP_4_BYTE;
                    break;
                  }
                case VFF_TYP_2_BYTE : /* Short to int */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    iptr = (int *)malloc(sz*sizeof(int));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        smax = ((short *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            s = ((short *)(image->imagedata))[i];
                            if (s > smax) smax = s;
                          }
                        if (smax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/smax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        s = ((short *)(image->imagedata))[i];
                        iptr[i] = s*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)iptr;
                    image->data_storage_type =  VFF_TYP_4_BYTE;
                    break;
                  }
                case VFF_TYP_4_BYTE : /* int to int */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    iptr = (int *)malloc(sz*sizeof(int));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        imax = ((int *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            it = ((int *)(image->imagedata))[i];
                            if (it > imax) imax = it;
                          }
                        if (imax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/imax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        it = ((int *)(image->imagedata))[i];
                        iptr[i] = it*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)iptr;
                    image->data_storage_type =  VFF_TYP_4_BYTE;
                    break;
                  }
                case VFF_TYP_FLOAT : /* float to int */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    iptr = (int *)malloc(sz*sizeof(int));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            f = ((float *)(image->imagedata))[i];
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        f = ((float *)(image->imagedata))[i];
                        iptr[i] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)iptr;
                    image->data_storage_type =  VFF_TYP_4_BYTE;
                    break;
                  }
                case VFF_TYP_COMPLEX : /* complex to int */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    iptr = (int *)malloc(sz*sizeof(int));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz*2; i += 2)
                          {
                            f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY! */
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0,j=0; i<sz*2; i += 2)
                      {
                        f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY */
                        iptr[j++] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)iptr;
                    image->data_storage_type =  VFF_TYP_4_BYTE;
                    break;
                  }
                default :
                  {
                    (void) fprintf(stderr,"lvconvert: unknown source conversion type\n");
                    return(0);
                    /* break; */
                  }
              } /* switch */
            break;
          } /* case */
        case VFF_TYP_FLOAT: /* Target is float */
          {
            switch(image->data_storage_type)
              {
                case VFF_TYP_BIT : /* Bit to FLOAT */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float));
                    fptr1 = fptr;
                    bptr = (unsigned char *)image->imagedata;
                    b = bptr;
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to look at */
                            if (*b & bit) 
                              {
                                if (sflg) *fptr1++ = scale_factor;
                                else if (nflg) *fptr1++ = norm_factor;
                                else *fptr1++ = 1.0;
                              }
                            else *fptr1++ = 0.0;
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                            b++;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_FLOAT;
                    image->map_scheme =  VFF_MS_NONE;
                    break;
                  }
                case VFF_TYP_1_BYTE : /* Char to float */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        cmax = 0;
                        for (i=0; i<sz; i++)
                          {
                            c = ((unsigned char *)(image->imagedata))[i];
                            if (c > cmax) cmax = c;
                          }
                        if (cmax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/cmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        c = ((unsigned char *)(image->imagedata))[i];
                        fptr[i] = c*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_FLOAT;
                    break;
                  }
                case VFF_TYP_2_BYTE : /* Short to float */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        smax = ((short *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            s = ((short *)(image->imagedata))[i];
                            if (s > smax) smax = s;
                          }
                        if (smax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/smax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        s = ((short *)(image->imagedata))[i];
                        fptr[i] = s*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_FLOAT;
                    break;
                  }
                case VFF_TYP_4_BYTE : /* int to float */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        imax = ((int *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            it = ((int *)(image->imagedata))[i];
                            if (it > imax) imax = it;
                          }
                        if (imax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/imax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        it = ((int *)(image->imagedata))[i];
                        fptr[i] = it*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_FLOAT;
                    break;
                  }
                case VFF_TYP_FLOAT : /* float to float */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            f = ((float *)(image->imagedata))[i];
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0 ;
                      }
                    for (i=0; i<sz; i++)
                      {
                        f = ((float *)(image->imagedata))[i];
                        fptr[i] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_FLOAT;
                    break;
                  }
                case VFF_TYP_COMPLEX : /* complex to float */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float));
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz*2; i += 2)
                          {
                            f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY! */
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0,j=0; i<sz*2; i += 2)
                      {
                        f = ((float *)(image->imagedata))[i]; /* REAL PART ONLY */
                        fptr[j++] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_FLOAT;
                    break;
                  }
                default :
                  {
                    (void) fprintf(stderr,"lvconvert: unknown source conversion type\n");
                    return(0);
                    /* break; */
                  }
              } /* switch */
            break;
          } /* case */
        case VFF_TYP_COMPLEX: /* Target is complex */
          {
            switch(image->data_storage_type)
              {
                case VFF_TYP_BIT : /* Bit to complex */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float)*2);
                    fptr1 = fptr;
                    bptr = (unsigned char *)image->imagedata;
                    b = bptr;
                    for (i=0; i<r; i++)
                      {
                        k = 0;
                        for (j=0; j<cols; j++)
                          {
                            bit = 1 << k;      /* Bit to look at */
                            if (*b & bit) 
                              { /* NOTE: This only sets the REAL PART */
                                if (sflg) *fptr1++ = scale_factor;
                                else if (nflg) *fptr1++ = norm_factor;
                                else *fptr1++ = 1.0;
                              }
                            else *fptr1++ = 0.0;
                            fptr++;
                            *fptr++ = 0.0; /* IMAGINARY PART */
                            if (k == 7)
                              {
                                k = 0;
                                b++;
                              }
                            else k++;
                          }
                        if (k != 0)
                            b++;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_COMPLEX;
                    image->map_scheme = VFF_MS_NONE;
                    break;
                  }
                case VFF_TYP_1_BYTE : /* Char to complex */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float)*2);
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        cmax = 0;
                        for (i=0; i<sz; i++)
                          {
                            c = ((unsigned char *)(image->imagedata))[i];
                            if (c > cmax) cmax = c;
                          }
                        if (cmax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/cmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0; i<sz; i++)
                      {
                        c = ((unsigned char *)(image->imagedata))[i];
                        fptr[2*i] = c*multiplier;
                        fptr[2*i+1] = 0.0;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_COMPLEX;
                    break;
                  }
                case VFF_TYP_2_BYTE : /* short to complex */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float)*2);
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        smax = ((short *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            s = ((short *)(image->imagedata))[i];
                            if (s > smax) smax = s;
                          }
                        if (smax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/smax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0; i<sz; i++)
                      {
                        s = ((short *)(image->imagedata))[i];
                        fptr[2*i] = s*multiplier;
                        fptr[2*i+1] = 0.0;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_COMPLEX;
                    break;
                  }
                case VFF_TYP_4_BYTE : /* int to complex */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float)*2);
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        imax = ((int *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            it = ((int *)(image->imagedata))[i];
                            if (it > imax) imax = it;
                          }
                        if (imax == 0) multiplier = 0.0;
                        else multiplier = norm_factor/imax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0; i<sz; i++)
                      {
                        it = ((int *)(image->imagedata))[i];
                        fptr[2*i] = it*multiplier;
                        fptr[2*i+1] = 0.0;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_COMPLEX;
                    break;
                  }
                case VFF_TYP_FLOAT : /* float to complex */
                  {
                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float)*2);
                    if (nflg != 0)              /* Get normalization factor */
                      {
                        fmax = ((float *)(image->imagedata))[0];
                        for (i=0; i<sz; i++)
                          {
                            f = ((float *)(image->imagedata))[i];
                            if (f > fmax) fmax = f;
                          }
                        if (fmax == 0.0) multiplier = 0.0;
                        else multiplier = norm_factor/fmax; 
                      }
                    else
                      {
                        multiplier = sflg ? scale_factor : 1.0;
                      }
                    for (i=0; i<sz; i++)
                      {
                        f = ((float *)(image->imagedata))[i];
                        fptr[2*i] = f*multiplier;
                        fptr[2*i+1] = 0.0;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_COMPLEX;
                    break;
                  }
                case VFF_TYP_COMPLEX : /* complex to complex */
                  {
                    if (nflg)
                      {
                        (void) fprintf(stderr,"lvconvert: Cannot normalize COMPLEX.\n");
                        return(0);
                      }
                    if (sflg) multiplier = scale_factor;

                    if (!lvcset_sizes(image,rows,size,&r,&sz)) return(0);
                    fptr = (float *)malloc(sz*sizeof(float)*2);
                    for (i=0; i<sz*2; i += 2) 
                      {
                        f = ((float *)(image->imagedata))[i];
                        fptr[i] = f*multiplier;
                        f = ((float *)(image->imagedata))[i+1]; /*was 2*i+1-jw*/
                        fptr[i+1] = f*multiplier;
                      }
                    free(image->imagedata);
                    image->imagedata = (char *)fptr;
                    image->data_storage_type =  VFF_TYP_COMPLEX;
                    break;
                  }
                default :
                  {
                    (void) fprintf(stderr,"lvconvert: unknown source conversion type\n");
                    return(0);
                    /* break; */
                  }
              } /* switch */
            break;
          } /* case */
        default :
          {
            (void)fprintf(stderr,"lvconvert: Unknown target data type: %d\n",k);
            return(0);
            /* break; */
          }
      } /* switch */

/*
** now for a little more kludge code for double and double complex...
*/

   if(really_to_double){
      if(k==VFF_TYP_FLOAT){
         if((dptr = (double *)malloc(size*image->num_data_bands*
            sizeof(double)))==NULL){
            fprintf(stderr,"lvconvert: failure to malloc stuff.\n");
            return(0);
         }
         for(i=0;i<size*image->num_data_bands;i++){
             dptr[i] = (double)fptr[i];
         }
         free(image->imagedata);
         image->imagedata = (char *)dptr;
         image->data_storage_type =  VFF_TYP_DOUBLE;
         image->map_scheme =  VFF_MS_NONE;
      }else{
         if((dptr = (double *)malloc(2*size*image->num_data_bands
            *sizeof(double)))==NULL){
            fprintf(stderr,"lvconvert: failure to malloc stuff.\n");
            return(0);
         }
         for(i=0;i<2*size*image->num_data_bands;i++){
             dptr[i] = (double)fptr[i];
         }
         free(image->imagedata);
         image->imagedata = (char *)dptr;
         image->data_storage_type =  VFF_TYP_DCOMPLEX;
         image->map_scheme =  VFF_MS_NONE;
      }
   }

/*
** the end .... whew!
*/

    return(1);
}

lvcset_sizes(image,rows,size,r,s)
struct xvimage *image;
int *r,*s,rows,size;
  {
    *r = rows*image->num_data_bands;
    *s = size*image->num_data_bands;
    return(1);
  }
/* -library_code_end */
