/* $Id: misc_math.c,v 4.0 89/06/06 15:38:54 mbp Exp $
 *
 * misc_math.c: mathematical utility procedures
 */

/***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include "heisenberg.h"

#define SQR(x) ((x)*(x))

/*-----------------------------------------------------------------------
 * Function:     chain_v_coord
 * Description:  computes the v coord of a point on a nonvertical chain
 * Arguments IN: *c: the center of the chain
 *               x,y: the point in the xy-plane over which the v
 *                coord is to be computed
 * Returns:      the v coord of the point on the chain over (x,y)
 * Notes:        
 */
double chain_v_coord(c,x,y)
Hpoint *c;
double x,y;
{
  double answer;
  answer = c->ver - (c->hor.im)*x + (c->hor.re)*y;
  return( answer );
}

/*-----------------------------------------------------------------------
 * Function:     compute_standard_rcircle
 * Description:  compute point on standard rcircle corresponding to
 *                a given angle and sign
 * Arguments IN: th: parameter (angle in radians)
 *               sign: tells which half to compute (+1 or -1)
 *          OUT: *p: the point on the rcircle corresponding to th
 * Returns:      nothing
 * Notes:        The 'standard' rcircle is the one with center (0,0,0)
 *               and radius 1.
 */
compute_standard_rcircle(p,th,sign)
Hpoint *p;
double th;
int sign;
{
  double r;
  r = sign * sqrt( fabs( cos( 2*th) ) );
  p->hor.re  = r * cos(th);
  p->hor.im  = r * sin(th);
  p->hor.inf = NO;
  p->ver     = - (p->hor.re)/(2* (p->hor.im)) * (1+r*r) ;
}

/*-----------------------------------------------------------------------
 * Function:     hpoint_neg
 * Description:  negative of a point in Heisenberg space
 * Arguments IN: *p2: the Hpoint to be negated
 *          OUT: *p1: -(*p2)
 * Returns:      nothing
 */
hpoint_neg(p1,p2)
Hpoint *p1,*p2;
{
	C_NEG(p1->hor,p2->hor);
	p1->ver = - p2->ver;
}

/*-----------------------------------------------------------------------
 * Function:	compute_spinal_t0
 * Description:	compute t0 in preparation for drawing a spinal by chains
 * Args  IN:	Z1,Z2: NORMALIZED Cvectors corresponding to
 *		  the vertices of the spinal
 *      OUT:	*t0: the computed t0 value
 * Returns:	nothing
 * Notes:	Z1 and Z2 must be normalized to have 3rd coord 1.
 *
 *		         Im[ << Z1_aff, Z2_aff >> ]
 *		t0 = i ------------------------------
 *		       Re[ << Z1_aff, Z2_aff >> ] - 1,
 *
 *		where Z1_aff and Z2_aff are Z1 and Z2 in affine coordinates,
 *		and << , >> is the positive definite Hermitian product on C2.
 *
 *		This t0 is equal to -i w, where w is from the formula in
 *		Bill's "Notes" which gives the geodesic between two points
 *		at infinity.
 */
compute_spinal_t0(t0, Z1,Z2)
     Complex *t0;
     Cvector Z1, Z2;
{
  Complex Z1Z2_aff_herm;
  int j;

  /* Z1Z2_aff_herm = <<Z1_aff, Z2_aff>> */
  for (j=0; j<2; j++){
    C_PUSH_S(Z1[j]); C_PUSH_S(Z2[j]); c_bar_s(); c_mul_s();
  }
  c_add_s(); C_POP_S(Z1Z2_aff_herm);

  t0->re = 0;
  t0->im = Z1Z2_aff_herm.im / (Z1Z2_aff_herm.re - 1);
  t0->inf = NO;
}

/*-----------------------------------------------------------------------
 * Function:	compute_spinal_slice
 * Description:	compute a slice of a spinal
 * Args  IN:	latitude: angle of latitude for this slice
 *		t0: The 't0' value for this spinal, as computed by
 *		  compute_spinal_t0
 *		Z1,Z2: NORMALIZED Cvectors corresponding to the vertices
 *		  of the spinal
 *      OUT:	*slice: the computed slice
 *		*success: whether computation was successful
 * Returns:	nothing
 * Notes:	Z1 and Z2 must be normalized to have 3rd coord 1.
 *
 *		This should always return success = 1; there is no
 *		excuse for failure.  Flag is there just in case.
 *
 *		Each slice is perpendicular to the spine, and hence
 *		corresponds to a point p on the "dual curve" of the 
 *		spine (the extension of the spine outside the unit
 *		ball in C2).  We compute the slice by computing the
 *		Cvector corresponding to p.  The spine's dual curve
 *		is parametrized by the upper unit semicircle in the
 *		complex plane; the latitude parameter is used as the
 *		argument of a point z on this semicircle.  z in turn
 *		determines p.
 *
 *		The parametrization of the spine's dual curve is
 *		achieved by the following sequence of maps:
 *		
 * disk_to_	unit disk to the	This takes the upper unit
 *  t0_upper_	upper-half-plane	semicircle to the ray t0 + x
 *  half_plane	Im[z-t0] > 0		(x > 0).
 *		a	|------->	b
 * 
 * left_half_	left-half-plane		This takes the line t0 + x (all x)
 *  plane_to_	Re[z] < 0 to the	to a circle perpendicular to the
 *  disk	unit disk		unit circle, passing through -1 and
 *					(1 + t0) / (1 - t0). The ray t0 + x
 *					(x > 0) is mapped to the part of
 *					this circle which is outside the
 *					unit circle.
 *		b	|------->	c
 * 
 * disk_to_	Poincare disk to	This maps the circle obtained in the
 *  C2_ball	complex geodesic	previous step to its image in the
 *		in B2 containing	(affine) complex line spanned by Z1
 *		Z1 and Z2.		and Z2.  The coordinates s[0] and
 *					s[1] are the coefficients of Z1 and
 *					Z2, respectively.
 *		c	|------->	(s[0], s[1])
 * 
 * The Cvector corresponding to the slice is then s[0] Z1 + s[1] Z2.
 */
compute_spinal_slice(slice,latitude,t0,Z1,Z2,success)
     Chain *slice;
     double latitude;
     Complex *t0;
     Cvector Z1,Z2;
     int *success;
{
  Cvector slice_cv;
  Complex a,b,c,s[2];
  int k;
  
  a.re = cos(latitude);        
  a.im = sin(latitude);
  a.inf = NO;
  disk_to_t0_upper_half_plane(&b, t0, &a);
  left_half_plane_to_disk(&c, &b);
  disk_to_C2_ball(s, t0, &c);
  for (k=0; k<3; ++k) {
    C_PUSH_S(Z1[k]); C_PUSH_S(s[0]); c_mul_s();
    C_PUSH_S(Z2[k]); C_PUSH_S(s[1]); c_mul_s();
    c_add_s(); C_POP_S(slice_cv[k]);
  }
  cvector_to_chain(slice, slice_cv, success);
}
