/* $Id: complex.c,v 4.0 89/06/06 15:38:29 mbp Exp $
 *
 * complex.c: complex arithmetic module
 */

/***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

/* (This is a general complex numbers arithmetic package, which may be
 * used separately from HEISENBERG.  The package consists of this file
 * and the file "complex.h".)
 *
 * SYNOPSIS:
 *   #include "complex.h"
 *
 * DESCRIPTION:
 *   This module implements arithmetic with extended complex numbers.
 *   An extended complex number is represented by a structure type
 *   named "Complex" (defined in "complex.h"), which has three fields:
 *   "re" and "im" are double fields which represent the real and
 *   imaginary parts, and "inf" is a char which serves as a flag to
 *   indicate whether the quantity is infinite or not; its value is
 *   always one of the predefined (in "complex.h") constants INFINITE
 *   or FINITE.  In the INFINITE case, the values of the "re" and "im"
 *   fields are meaningless.
 *
 *   The global variables C_ZERO, C_ONE, C_I, C_INFINITY, C_NEG_ONE,
 *   and C_NEG_I are available for use by application programs.  Note
 *   that these are variables, not constants.  They should be used as
 *   constants, however.
 *
 *   This module provides a stack for doing complex arithmetic.  The
 *   basic stack operations c_push_s and c_pop_s can be used to push
 *   onto and pop off of this stack, and arithmetic operations can be
 *   performed on the stack.  Each of these operations removes
 *   operands from the top of the stack and replaces them with the
 *   result of the operation.
 *
 *   All procedures in this module begin with "c_".  Procedures which
 *   deal with the stack end in "_s".
 *
 *   A set of macros defined in "complex.h" facilitates calling
 *   procedures in this module which take pointers to Complexes as
 *   arguments.  For every procedure which takes a pointer to a
 *   Complex as an argument, there is a corresponding macro which
 *   calls the procedure after taking the address of its arguments.
 *   The names of the macros are exactly the same as the names of the
 *   procedures, but are spelled with upper case letters. So for
 *   example, C_ADD(z1,z2,z3) is the same as c_add(&z1,&z2,&z3).
 *   
 *   The following conventions regarding arithmetic with infinity are
 *   used:
 *	infinity + z = z + infinity = infinity
 * 	infinity - z = z - infinity = infinity
 *	infinity * z = z * infinity = infinity
 *	infinity / w = infinity 
 *	z / 0 = infinity
 *	w / infinity = 0
 *	infinity / infinity = 1,
 *   where z denotes any extended complex number (including infinity),
 *   and w denotes any finite complex number.
 */

#include "complex.h"
#include <stdio.h>
#include "malloc.h"
#include <math.h>

#define	SQR(x)	((x)*(x))
#define YES 1
#define NO 0
#define RE(z) (z->re)
#define IM(z) (z->im)
#define INF(z) (z->inf)
#define FINITE ((char)NO)
#define INFINITE ((char)YES)
#define IS_FINITE(z) (INF(z)==FINITE)
#define IS_INFINITE(z) (INF(z)==INFINITE)
#define IS_ZERO(z) (c_isequal(z,&C_ZERO))

Complex
  C_ZERO = {0.0, 0.0, FINITE},
  C_ONE = {1.0, 0.0, FINITE},
  C_I = {0.0, 1.0, FINITE},
  C_NEG_ONE = {-1.0, 0.0, FINITE},
  C_NEG_I = {0.0, -1.0, FINITE},
  C_INFINITY = {0.0, 0.0, INFINITE};

char *c_error=NULL;

typedef struct c_node_s {
  Complex         c;
  struct c_node_s *prev;
} c_node;

static c_node *top_s=NULL;

/*-----------------------------------------------------------------------
 * Function:     c_copy
 * Description:  copy one complex to another
 * Arguments IN: *z2: source
 *          OUT: *z1: destination
 * Returns:      nothing
 */
c_copy(z1, z2)
     Complex        *z1, *z2;
{
  RE(z1) = RE(z2);
  IM(z1) = IM(z2);
  INF(z1) = INF(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_enorm_sq
 * Description:  compute Euclidean norm-squared of a complex
 * Arguments IN: *z: the number to compute norm of
 * Returns:      the norm of *z
 * Notes:        *z must be finite; the returned value is meaningless
 *               if *z is infinite.
 */
double c_enorm_sq(z)
     Complex        *z;
{
  double answer;
  answer = SQR(RE(z)) + SQR(IM(z));
  return ( answer );
}

/*-----------------------------------------------------------------------
 * Function:     c_enorm
 * Description:  compute Euclidean norm of a complex
 * Arguments IN: *z: the number to compute norm of
 * Returns:      the norm of *z
 * Notes:        *z must be finite; the returned value is meaningless
 *               if *z is infinite.
 */
double c_enorm(z)
     Complex        *z;
{
  double answer;
  answer = sqrt(fabs(c_enorm_sq(z)));
  return ( answer );
}

/*-----------------------------------------------------------------------
 * Function:     c_edist
 * Description:  compute Euclidean distance between two complexes
 * Arguments IN: *z1,*z2: the two complexes
 * Returns:      Euclidean distance from *z1 to *z2
 * Notes:        *z1 and *z2 must be finite; the returned value is
 *               meaningless if either is infinite.
 */
double c_edist(z1, z2)
     Complex        *z1, *z2;
{
  double d1,d2,answer;
  d1 = RE(z1) - RE(z2);
  d2 = IM(z1) - IM(z2);
  answer = sqrt( fabs(SQR(d1) + SQR(d2)) );
  return(answer);
}

/*-----------------------------------------------------------------------
 * Function:     c_arg
 * Description:  compute the arg of a complex
 * Arguments IN: *z: the comples
 * Returns:      arg(*z)
 * Notes:        
 */
double c_arg(z)
Complex *z;
{
  double answer;
  answer = atan2( IM(z), RE(z) );
  return( answer );
}

/*-----------------------------------------------------------------------
 * Function:     c_isequal
 * Description:  determine if two complexes are equal
 * Arguments IN: *z1,*z2: the two numbers
 * Returns:      YES or NO
 * Notes:        All infinite numbers are considered equal
 */
c_isequal(z1, z2)
Complex        *z1, *z2;
{
  int z1inf, z2inf;

  z1inf = IS_INFINITE(z1);
  z2inf = IS_INFINITE(z2);
  if (z1inf && z2inf)
    return(YES);
  if ( (z1inf && (!z2inf)) || ((!z1inf) && z2inf) )
    return(NO);
  if ( (RE(z1)==RE(z2)) && (IM(z1)==IM(z2)) )
    return(YES);
  else
    return(NO);
}

/*-----------------------------------------------------------------------
 * Function:     c_bar
 * Description:  take complex conjugate
 * Arguments IN: *z2: original complex number
 *          OUT: *z1: complex conjugate of *z2
 * Returns:      nothing
 * Notes:        *z2 is left unchanged
 */
c_bar(z1,z2)
Complex *z1,*z2;
{
  RE(z1) =  RE(z2);
  IM(z1) = -IM(z2);
  INF(z1) = INF(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_neg
 * Description:  take the negative of a complex
 * Arguments IN: *z2: the complex to be negated
 *          OUT: *z1: = - (*z2)
 * Returns:      nothing
 * Notes:        
 */
c_neg(z1,z2)
Complex *z1,*z2;
{
  RE(z1) = -RE(z2);
  IM(z1) = -IM(z2);
  INF(z1) = INF(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_sca_mul
 * Description:  multiply a complex by a scalar
 * Arguments IN: s: the scalar
 *               *z2: the compex
 *          OUT: *z1: the result s * (*z2)
 * Returns:      nothing
 * Notes:        
 */
c_sca_mul(z1,s,z2)
Complex *z1,*z2;
double s;
{
  RE(z1) = s * RE(z2);
  IM(z1) = s * IM(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_add
 * Description:  add 2 complexes
 * Arguments IN: *z1,*z2: the two summands
 *          OUT: *w: the sum of *z1 and *z2
 * Returns:      nothing
 * Notes:        if either *z1 or *z2 is infinite, *w is set to
 *               infinity
 */
c_add(w, z1, z2)
     Complex        *w, *z1, *z2;
{
  if (INF(w) = (INF(z1) || INF(z2)))
    return;
  RE(w) = RE(z1) + RE(z2);
  IM(w) = IM(z1) + IM(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_sub
 * Description:  subtract 2 complexes
 * Arguments IN: *z1,*z2: the numbers to operate on
 *          OUT: *w: the difference *z1 - *z2
 * Returns:      nothing
 * Notes:        if either *z1 or *z2 is infinite, *w is set to
 *               infinity
 */
c_sub(w, z1, z2)
     Complex        *w, *z1, *z2;
{
  if (INF(w) = (INF(z1) || INF(z2)))
    return;
  RE(w) = RE(z1) - RE(z2);
  IM(w) = IM(z1) - IM(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_mul
 * Description:  multiply 2 complexes
 * Arguments IN: *z1,*z2: the factors
 *          OUT: *w: the product (*z1) * (*z2)
 * Returns:      nothing
 * Notes:        If either *z1 or *z2 is infinite, *w is set to
 *               infinity.  This is true even if one of the factors
 *               is 0.
 */
c_mul(w, z1, z2)
     Complex        *w, *z1, *z2;
{
  if (INF(w) = (INF(z1) || INF(z2)))
    return;
  RE(w) = RE(z1) * RE(z2) - IM(z1) * IM(z2);
  IM(w) = RE(z1) * IM(z2) + IM(z1) * RE(z2);
}

/*-----------------------------------------------------------------------
 * Function:     c_div
 * Description:  divide two complexes
 * Arguments IN: *z1: dividend
 *               *z2: divisor
 *          OUT: *w: the quotient (*z1) / (*z2)
 * Returns:      nothing
 * Notes:        the following conventions regarding infinity are used:
 *                 infinity / w = infinity
 *                 z / 0 = infinity
 *                 w / infinity = 0
 *                 infinity / infinity = 1
 *               where z denotes any extended complex number
 *               (including infinity), and w denotes any finite
 *               complex number.
 */
c_div(w, z1, z2)
     Complex        *w, *z1, *z2;
{
  double          s;
  
  if ((INF(z1) && (!INF(z2)))
      || (IS_ZERO(z2))) {
    INF(w) = INFINITE;
    return;
  }
  
  if (INF(z1) && INF(z2))
    c_copy(w, &C_ONE);
  else if (INF(z2))
    c_copy(w, &C_ZERO);
  else {
    s = SQR(RE(z2)) + SQR(IM(z2));
    RE(w) = (RE(z1) * RE(z2) + IM(z1) * IM(z2)) / s;
    IM(w) = (IM(z1) * RE(z2) - RE(z1) * IM(z2)) / s;
    INF(w) = FINITE;
  }
}

/*-----------------------------------------------------------------------
 * Function:     c_cross_ratio
 * Description:  compute the cross ratio of 4 complexes
 * Arguments IN: *z1,*z2,*z3,*z4: the 4 complexes
 *          OUT: *w: the cross ratio [*z1,*z2,*z3,*z4]
 * Returns:      nothing
 * Notes:        Works properly with infinite values
 */
c_cross_ratio(w, z1, z2, z3, z4)
Complex        *w, *z1, *z2, *z3, *z4;
{
  c_push_s(z1);
  c_push_s(z2);
  c_push_s(z3);
  c_push_s(z4);
  c_cross_ratio_s();
  c_pop_s(w);
}

/*-----------------------------------------------------------------------
 * Function:     c_push_s
 * Description:  push an entry onto the stack
 * Arguments IN: *z: the entry to be pushed
 * Returns:      nothing
 */
c_push_s(z)
     Complex *z;
{
  c_node *old_top_s;

  old_top_s = top_s;
  top_s = (c_node*)malloc(sizeof(c_node));
  top_s->prev = old_top_s;
  c_copy( &(top_s->c), z );
}

/*-----------------------------------------------------------------------
 * Function:     c_pop_s
 * Description:  pop an entry from the top of the stack
 * Arguments OUT: *z: the popped entry
 * Returns:      nothing
 *               If a pop is attempted when the stack is empty,
 *               the contents of *z are left unchanged.  The present
 *               version does not signal any error in this case.
 */
c_pop_s(z)
     Complex *z;
{
  c_node *old_top_s;

  if (top_s!=NULL) {
    c_copy( z, &(top_s->c) );
    old_top_s = top_s;
    top_s = top_s->prev;
    free((char*)old_top_s);
  }
  else {
    c_error = "Attempt to pop empty stack";
  }
}
    
/*-----------------------------------------------------------------------
 * Function:     c_bar_s
 * Description:  take complex conjugate of top of stack
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        The number on the top of the stack is replaced by
 *               its complex conjugate.
 */
c_bar_s()
{
  if (top_s!=NULL)
    top_s->c.im = - top_s->c.im;
}

/*-----------------------------------------------------------------------
 * Function:     c_add_s
 * Description:  add top two entries on stack
 * Arguments     (none)
 * Returns:      nothing
 * Notes:        If z2=top, z1=next under top, then z1 and z2
 *               are popped off and z1+z2 is pushed on
 */
c_add_s()
{
  Complex z1, z2, w;

  c_pop_s(&z2);
  c_pop_s(&z1);
  c_add(&w,&z1,&z2);
  c_push_s(&w);
}

/*-----------------------------------------------------------------------
 * Function:     c_sub_s
 * Description:  add top two entries on stack
 * Arguments     (none)
 * Returns:      nothing
 * Notes:        If z2=top, z1=next under top, then z1 and z2
 *               are popped off and z1-z2 is pushed on
 */
c_sub_s()
{
  Complex z1, z2, w;

  c_pop_s(&z2);
  c_pop_s(&z1);
  c_sub(&w,&z1,&z2);
  c_push_s(&w);
}

/*-----------------------------------------------------------------------
 * Function:     c_mul_s
 * Description:  multiply top two entries on stack
 * Arguments     (none)
 * Returns:      nothing
 * Notes:        If z2=top, z1=next under top, then z1 and z2
 *               are popped off and z1*z2 is pushed on
 */
c_mul_s()
{
  Complex z1, z2, w;

  c_pop_s(&z2);
  c_pop_s(&z1);
  c_mul(&w,&z1,&z2);
  c_push_s(&w);
}

/*-----------------------------------------------------------------------
 * Function:     c_div_s
 * Description:  divide the top two entries on the stack
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        If z2=top, z1=next under top, then z1 and z2
 *               are popped off and z1/z2 is pushed on
 */
c_div_s()
{
  Complex z1, z2, w;

  c_pop_s(&z2);
  c_pop_s(&z1);
  c_div(&w,&z1,&z2);
  c_push_s(&w);
}

/*-----------------------------------------------------------------------
 * Function:     c_cross_ratio
 * Description:  compute cross ratio of top 4 entries on stack
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        Stack {top=z4,z3,z2,z1} is replaced with
 *               {top=[z1,z2,z3,z4]}
 */
c_cross_ratio_s()
{
  Complex z1,z2,z3,z4;

  c_pop_s(&z4);
  c_pop_s(&z3);
  c_pop_s(&z2);
  c_pop_s(&z1);
  c_push_s(&z1); c_push_s(&z4); c_sub_s();
  c_push_s(&z1); c_push_s(&z2); c_sub_s(); c_div_s();
  c_push_s(&z3); c_push_s(&z2); c_sub_s();
  c_push_s(&z3); c_push_s(&z4); c_sub_s(); c_div_s();
  c_mul_s();
}

/*-----------------------------------------------------------------------
 * Function:     c_neg_s
 * Description:  take negative of top of stack
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        top of stack is replaced with its neg
 */
c_neg_s()
{
  if (top_s!=NULL) {
    top_s->c.re = - top_s->c.re;
    top_s->c.im = - top_s->c.im;
  }
}

/*-----------------------------------------------------------------------
 * Function:     c_sca_mul_s
 * Description:  multiply top of stack by a scalar
 * Arguments IN: s: the scalar
 * Returns:      nothing
 * Notes:        top of stack is replaced by itself times s
 */
c_sca_mul_s(s)
double s;
{
  if (top_s!=NULL) {
    top_s->c.re *= s;
    top_s->c.im *= s;
  }
}

/*-----------------------------------------------------------------------
 * Function:     c_sqrt
 * Description:  compute a square-root of a complex number
 * Arguments IN: *z: the complex
 *           OUT: *s: a root
 * Returns:      
 * Notes:        added by wmg 8/14/88
 */

c_sqrt(s,z)
     Complex *s,*z;
{
  double a,r;
  
  if (INF(z)) { 
    INF(s) = INFINITE; 
    return;
  }
  else {
    a = c_arg(z)/2;
    r = sqrt(fabs(c_enorm(z)));
    s->re = r * cos(a);
    s->im = r * sin(a);
    s->inf = NO;
  }
}

/*-----------------------------------------------------------------------
 * Function:     c_sqrt_s
 * Description:  replace stack top with its sqrt
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        
 */
c_sqrt_s()
{
  Complex z,s;

  C_POP_S(z);
  C_SQRT(s,z);
  C_PUSH_S(s);
}
