/*
 * clipper.c: 3D clipping filter for quad lists
 *
 * usage: clipper < infile > outfile
 *
 *  infile should be a quad list
 *  outfile will be another quad list
 *
 * this version always clips to coord sys [-10,10] X [-10,10] X [-10,10]
 */

/* Note: this program uses ungetc on stdin, hence it will only
   work if stdin is buffered (this is usually the case) */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

#define YES 1
#define NO  0

#define BITSPERBYTE 8

typedef struct ProjectivePoint_s {
  double coord[4];
} ProjectivePoint;

typedef ProjectivePoint HalfSpace;

typedef struct AffinePoint_s {
  double coord[3];
} AffinePoint;

/* Our Polygon type can have at most 10 vertices; this is because the
   intersection of a quadrilateral with a cube can have no more than
   10 edges (probably no more than 8 ??) */
#define MAXVERTICES 10

typedef struct Polygon_s {
  AffinePoint vertex[MAXVERTICES];
  int n;
} Polygon;

typedef struct Quad_s {
  AffinePoint vertex[4];
} Quad;

double xmin = -10;
double ymin = -10;
double zmin = -10;
double xmax =  10;
double ymax =  10;
double zmax =  10;

HalfSpace boundary[6];

static char *c1 = "-.0123456789";

double dotprod4();

main()
{
  Quad q;
  Polygon poly;

  initialize_boundaries();

  printf("QUAD\n");
  skip_chars(c1);
  while (!feof(stdin)) {
    if (read_quad(&q)) {
      clip_quad(&poly, &q);
      print_poly(&poly);
    }
    skip_chars(c1);
  }
}

/*-----------------------------------------------------------------------
 * Function:	read_quad
 * Description:	read a quad from stdin
 * Args OUT:	*q: the quad that was read
 * Returns:	1 if a complete quad was read
 *		0 if EOF or other problems found before complete
 *		  quad is read
 */
read_quad(q)
     Quad *q;
{
  int v,i;

  if (!feof(stdin))
    for (v=0; v<4; ++v)
      for (i=0; i<3; ++i)
	if (scanf("%lf", &(q->vertex[v].coord[i])) == 0) {
	  fprintf(stderr, "incomplete quadrilateral (ignored)\n");
	  return(0);
	}
  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	skip_chars
 * Description:	read chars from stdin until a char from *s
 *		  is read or EOF; then unread that char if not EOF.
 * Args  IN:	*s: chars to stop for
 * Returns:	nothing
 */
skip_chars(s)
     char *s;
{
  int c;

  while (((c=getchar()) != EOF) && (strchr(s, c) == NULL));
  if (c != EOF) ungetc(c, stdin);
}

/*-----------------------------------------------------------------------
 * Function:	print_poly
 * Description:	Print a 'polygon' on standard output
 * Args  IN:	*poly: the polygon to print
 * Returns:	nothing
 * Notes:	We only print quadrilaterals. If the polygon is not a
 *		quadrilateral, it is broken up into a sequence of
 *		quadrilaterals.
 *
 *		If the number of vertices in the polygon is less than
 *		  3, we return immediately without printing anything.
 *
 */
print_poly(poly)
     Polygon *poly;
{
  int nsave;
  Quad q;

  if ((nsave=poly->n) < 3) return;
  while (poly->n) {
    if (strip_quad(&q, poly))
      printf("%f %f %f  %f %f %f\n\t%f %f %f  %f %f %f\n",
	     q.vertex[0].coord[0],
	     q.vertex[0].coord[1],
	     q.vertex[0].coord[2],

	     q.vertex[1].coord[0],
	     q.vertex[1].coord[1],
	     q.vertex[1].coord[2],

	     q.vertex[2].coord[0],
	     q.vertex[2].coord[1],
	     q.vertex[2].coord[2],

	     q.vertex[3].coord[0],
	     q.vertex[3].coord[1],
	     q.vertex[3].coord[2]);
  }
  poly->n = nsave;
}

strip_quad(q, poly)
     Quad *q;
     Polygon *poly;
{
  int i;

  if (poly->n < 3) {
    fprintf(stderr, "strip_quad: n = %1d\n", poly->n);
    return(0);
  }

  if (poly->n > 4) {
    q->vertex[0] = poly->vertex[0];
    q->vertex[1] = poly->vertex[poly->n-3];
    q->vertex[2] = poly->vertex[poly->n-2];
    q->vertex[3] = poly->vertex[poly->n-1];
    poly->n -= 2;
  }
  else {
    for (i=0; i<poly->n; ++i)
      q->vertex[i] = poly->vertex[i];
    if (poly->n == 3)
      q->vertex[3] = poly->vertex[2];
    poly->n = 0;
  }
  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	add_vertex_to_poly
 * Description:	add a vertex to a polygon
 * Args  IN:	*p: the vertex to be added
 *		*poly: the polygon to add it to
 * Returns:	nothing
 * Notes:	Does nothing if polygon is full
 */
add_vertex_to_poly(p, poly)
     AffinePoint *p;
     Polygon *poly;
{
  if (poly->n<MAXVERTICES) {
    poly->vertex[poly->n] = *p;
    ++(poly->n);
  }
}

clip_quad(poly, q)
     Polygon *poly;
     Quad *q;
{
  int j;

  for (j=0; j<4; ++j)
    poly->vertex[j] = q->vertex[j];
  poly->n = 4;

  for (j=0; j<6; ++j)
    clip_poly(poly, &(boundary[j]));
}

/*-----------------------------------------------------------------------
 * Function:	clip_poly
 * Description:	clip a polygon to a half-space
 * Args  IN:	*poly: the original polygon
 *		hs: the clipping halfspace
 *      OUT:	*poly: the new, clipped polygon
 * Returns:	nothing
 */
clip_poly(poly, hs)
     Polygon *poly;
     HalfSpace *hs;
{
  int j;
  AffinePoint *s, *p, i;
  Polygon origPoly;

  copy_poly(&origPoly, poly);

  poly->n = 0;
  s = &(origPoly.vertex[origPoly.n-1]);
  for(j=0; j<origPoly.n; ++j) {
    p = &(origPoly.vertex[j]);
    if (in_halfspace(p, hs))
      if (in_halfspace(s, hs))
	add_vertex_to_poly(p, poly);
      else {
	intersect(&i, s, p, hs);
	add_vertex_to_poly(&i, poly);
	add_vertex_to_poly(p, poly);
      }
    else
      if (in_halfspace(s, hs)) {
	intersect(&i, s, p, hs);
	add_vertex_to_poly(&i, poly);
      }
    s = p;
  }
}

in_halfspace(p, hs)
     AffinePoint *p;
     HalfSpace *hs;
{
  ProjectivePoint pp;

  affine_to_projective(&pp, p);
  return(dotprod4(pp.coord,hs->coord) >= 0);
}

affine_to_projective(p, a)
     ProjectivePoint *p;
     AffinePoint *a;
{
  int i;

  for (i=0; i<3; ++i)
    p->coord[i] = a->coord[i];
  p->coord[3] = 1;
}

double
  dotprod4(u,v)
double u[4],v[4];
{
  return( u[0]*v[0] + u[1]*v[1] + u[2]*v[2] + u[3]*v[3] );
}

intersect(a, p1, p2, hs)
     AffinePoint *a, *p1, *p2;
     HalfSpace *hs;
{
  double t0d, t0;
  int j;

  t0d = 
    - hs->coord[0]*p1->coord[0]
      - hs->coord[1]*p1->coord[1] 
	- hs->coord[2]*p1->coord[2] 
	  + hs->coord[0]*p2->coord[0]
	    + hs->coord[1]*p2->coord[1]
	      + hs->coord[2]*p2->coord[2];
  if (t0d == 0) {
    fprintf(stderr, "intersection error\n");
    return;
  }

  t0 = - (  hs->coord[3]
	  + hs->coord[0]*p1->coord[0]
	  + hs->coord[1]*p1->coord[1]
	  + hs->coord[2]*p1->coord[2] ) / t0d;

  for (j=0; j<3; ++j)
    a->coord[j] = p1->coord[j] + t0 * (p2->coord[j] - p1->coord[j]);
}

copy_poly(dest, source)
     Polygon *dest, *source;
{
  int j;
  dest->n = source->n;
  for (j=0; j<source->n; ++j)
    dest->vertex[j] = source->vertex[j];
}

initialize_boundaries()
{
  boundary[0].coord[0] = -1;
  boundary[0].coord[1] =  0;
  boundary[0].coord[2] =  0;
  boundary[0].coord[3] =  xmax;		/* x <= xmax */

  boundary[1].coord[0] =  0;
  boundary[1].coord[1] = -1;
  boundary[1].coord[2] =  0;
  boundary[1].coord[3] =  ymax;		/* y <= ymax */

  boundary[2].coord[0] =  0;
  boundary[2].coord[1] =  0;
  boundary[2].coord[2] = -1;
  boundary[2].coord[3] =  zmax;		/* z <= zmax */

  boundary[3].coord[0] =  1;
  boundary[3].coord[1] =  0;
  boundary[3].coord[2] =  0;
  boundary[3].coord[3] =  -xmin;	/* x =< xmin */

  boundary[4].coord[0] =  0;
  boundary[4].coord[1] =  1;
  boundary[4].coord[2] =  0;
  boundary[4].coord[3] =  -ymin;	/* y =< ymin */

  boundary[5].coord[0] =  0;
  boundary[5].coord[1] =  0;
  boundary[5].coord[2] =  1;
  boundary[5].coord[3] =  -zmin;	/* z =< zmin */
}
