/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include <math.h>
#include <gl.h>
#include <device.h>
#include "../lgd.h"
#include "../internal.h"
#include "../vector.h"
#include "gr.h"
#include "dpu.h"
#include GLGHEADER

#define ESC	'\033'

static double ymousefactor, xmousefactor;

static double min_focal_dist, min_window_width;

#define WXLEN(x) ((x)*xmousefactor)
#define WYLEN(y) ((y)*ymousefactor)

#define RADTODEG(a) ((a)*180/M_PI)

int
  GrViewInit()
{
  lgd_View3 view;

  dpu_get_default_view( &view );
  xmousefactor = (view.u2 - view.u1)/GrWindowWidth;
  ymousefactor = (view.v2 - view.v1)/GrWindowHeight;

  min_focal_dist = dpu_default_clipping_distance()/100;
  min_window_width = dpu_default_clipping_distance()/10000;
}

int
  GrRotate(view, dx, dy)
lgd_View3 *view;
int dx,dy;
{
  double M[3][3], E[3], U[3], U1[3], E1[3];
  double dir[3], wdx, wdy, dv[3], angle, wl;
  int i;

  if ( (dx == 0) && (dy == 0) )
    return;

  /* Compute rotation matrix */
  wdx = WXLEN(dx);
  wdy = WYLEN(dy);
  for (i=0; i<3; ++i)
    dv[i] = wdx * dpu_unit_ivec[i] + wdy * dpu_unit_jvec[i];
  LGD_cross_vec(dir, dpu_unit_kvec, dv);
  wl = sqrt(wdx*wdx + wdy*wdy);
  angle = - atan2( wl, dpu_focal_dist/2 );
  LGD_compute_3d_rot_mat( M, dir, RADTODEG(angle) );

  /* Move eye point */
  LGD_sub_vec( E, view->eye, view->focus );
  LGD_matmul_vec( E1, M, E );
  LGD_add_vec( view->eye, view->focus, E1 );  

  /* Move up point */
  LGD_sub_vec( U, view->up,  view->focus );
  LGD_matmul_vec( U1, M, U );
  LGD_add_vec( view->up, view->focus, U1 );  
}

/*-----------------------------------------------------------------------
 * Function:     GrReset
 * Description:  Reset to default view
 * Arguments:    (none)
 * Returns:      nothing
 */
int
  GrReset()
{
  lgd_View3 view;

  lgd_get_default_view( &view );
  lgd_set_view( &view );
}

GrZoom(view, dx, dy)
     lgd_View3 *view;
     int dx,dy;
{
  double dv[3];

  if (dx == 0) return;

  LGD_scamul_vec( dv, -WXLEN(dx), dpu_unit_kvec  );
  LGD_add_vec( view->eye, view->eye, dv );
  LGD_add_vec( view->up, view->up, dv );
  LGD_add_vec( view->focus, view->focus, dv );
}

GrTranslate(view, dx, dy)
     lgd_View3 *view;
     int dx,dy;
{
  double dv[3], wdx, wdy;
  int i;

  if ( (dx == 0) && (dy == 0) ) return;

  wdx = WXLEN(dx);
  wdy = WYLEN(dy);
  for (i=0; i<3; ++i)
    dv[i] = -wdx * dpu_unit_ivec[i] - wdy * dpu_unit_jvec[i];
  LGD_add_vec( view->eye, view->eye, dv );
  LGD_add_vec( view->up, view->up, dv );
  LGD_add_vec( view->focus, view->focus, dv );
}


int
  GrScale(view, dx, dy)
lgd_View3 *view;
int dx,dy;
{
  double uc, us, vc, vs, wdx;

  if (dx == 0) return;

  wdx = WXLEN(dx);
  uc = (view->u2 + view->u1)/2;
  vc = (view->v2 + view->v1)/2;
  us = (view->u2 - view->u1)/2;
  vs = (view->v2 - view->v1)/2;
  us += -wdx; if (us <= min_window_width) us = min_window_width;
  vs += -wdx; if (vs <= min_window_width) vs = min_window_width;
  view->u1 = uc - us;
  view->u2 = uc + us;
  view->v1 = vc - vs;
  view->v2 = vc + vs;
}

int
  GrFov(view, dx, dy)
lgd_View3 *view;
int dx,dy;
{
  double de[3], fd;

  if (dx == 0) return;

  fd = dpu_focal_dist - WXLEN(dx);
  if (fd <= min_focal_dist) fd = min_focal_dist;
  LGD_scamul_vec( de, fd, dpu_unit_kvec );
  LGD_add_vec( view->eye, view->focus, de );
}


int
  GrTwist( view, dx, dy )
lgd_View3 *view;
int dx,dy;
{
  double U[3], M[3][3], U1[3], angle;

  if (dx == 0) return;
  
  /* Compute rotation matrix */
  angle = - atan2( WXLEN(dx), dpu_focal_dist/2 );
  LGD_compute_3d_rot_mat( M, dpu_unit_kvec, RADTODEG(angle) );
  LGD_sub_vec( U, view->up, view->focus );
  LGD_matmul_vec( U1, M, U );
  LGD_add_vec( view->up, view->focus, U1 );
}

printvector(s,v)
     char *s;
     double v[3];
{
  printf("%s(%f, %f, %f)\n", s, v[0], v[1], v[2]);
}
