/* $Id: fsa.c,v 1.3 90/07/11 13:10:09 mbp Exp Locker: mbp $
 *
 * fsa.c: finite state automaton for matching a finite set of strings
 *
 * Mark B. Phillips
 * April 17, 1989
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

#include <stdio.h>

/* Maximum number of states: */
#define MAX_STATES 300

/* Special states and flags */
#define ACCEPT -1
#define REJECT -2
#define ERROR  -3

/* Operation flags */
#define PROGRAM 1
#define PARSE   2

typedef struct Trule_s {
  char			c;	/* char for this rule */
  int  			ns;	/* state to switch to */
  struct Trule_s 	*next;	/* ptr to next rule for this state */
} Trule;

typedef struct State_s {
  Trule *tlist;		/* list of transition rules for this state */
  int	return_value;	/* value to return if we end in this state */
} State;

static State *state[MAX_STATES];/* ptrs to state notes */
static int state_count = 0;	/* number of states in current use */
static int reject_value;	/* value to return upon rejection */

static Trule *new_trule_node();
static int initial_state, return_value;

/*-----------------------------------------------------------------------
 * Function:	fsa_initialize
 * Description:	Initialize the FSA
 * Args  IN:	reject: value to return when parsing unacceptable
 *		  string
 */
fsa_initialize(reject)
int reject;
{
  /* Clear out current program: */
  while (state_count--) {
    delete_trule_list(state[state_count]->tlist);
    free((char*)(state[state_count]));
  }
  state_count = 0;

  reject_value = reject;
  initial_state = new_state();
}

/*-----------------------------------------------------------------------
 * Function:	fsa_install
 * Description:	Install a string in the FSA
 * Args  IN:	s: the string to install
 *		v: the value to return when s is parsed
 * Returns:	v, unless there is not enough room in the FSA, in
 *		which case -v is returned.
 */
fsa_install(s,v)
char *s;
int v;
{
  return(fsa_execute(s, v, PROGRAM));
}

/*-----------------------------------------------------------------------
 * Function:	fsa_parse
 * Description:	Parse a string
 * Args  IN:	s: the string to parse
 * Returns:	If s is acceptable, returns the value v that was
 *		specified when s was installed with fsa_install().  If
 *		v is not acceptable, returns the value of reject given
 *		in the most recent call to fsa_initialize().
 * Notes:	
 */
fsa_parse(s)
char *s;
{
  return(fsa_execute(s, 0, PARSE));
}

/*-----------------------------------------------------------------------
 * Function:	fsa_execute
 * Description:	parse or program (install) a string
 * Args  IN:	s: the string
 *		v: value to correspond to string
 *		op: PROGRAM or PARSE
 * Returns:	Result of parsing or installing.  If op==PROGRAM, this
 *		is always v, unless there is no more room in the FSA,
 *		in which case it is -v.  If op==PARSE, it is either v
 *		or the reject value.

 *		The philosophy of this procedure and procedure
 *		next_state() is that programming the FSA and parsing a
 *		string using the FSA are very similar processes; the
 *		only difference is in what is done when there is no
 *		rule for the current state corresponding to the
 *		current input char.  When parsing, we return a
 *		rejection.  When programming, we add a rule, and
 *		possibly a new state.
 *
 *		If op==PROGRAM, s is added to list of acceptable
 *		strings, and the FSA is programmed to return v for s.
 *		If op==PARSE, the FSA is executed and returns value
 *		corresponding to s, or reject value if s is not
 *		acceptable.
 */
static
  fsa_execute(s, v, op)
char *s;
int v, op;
{
  int cs;			/* current state */
  int ns;			/* next state */

  cs = initial_state;
  return_value = reject_value;
  while ( (cs != ACCEPT) && (cs != REJECT) && (cs != ERROR) ) {
    cs = next_state(cs, *s, v, op);
    ++s;
  }
  return( (cs == ERROR) ? -v : return_value );
}

/*-----------------------------------------------------------------------
 * Function:	next_state
 * Description:	Get the next state based on current state and char
 * Args  IN:	cs: current state
 *		c: current char
 *		v: return value when programing
 *		op: PROGRAM or PARSE
 * Returns:	index of next state, or ERROR
 * Notes:	If op==PROGRAM and no transition rule for c exists for
 *		the current state, a new state is created and a
 *		transition rule to that state is added to the current
 *		state.
 *
 *		See also comments for fsa_execute().
 */
static
  next_state(cs, c, v, op)
int cs, v, op;
char c;
{
  Trule *t;

  /* Check tlist of current state for rule for c */
  t = state[cs]->tlist;
  while (t != NULL) {
    if (t->c == c) {
      /* We found rule for c; if next state is ACCEPT, set return value. */
      /* Then return next state. */
      if (t->ns == ACCEPT)
	return_value = state[cs]->return_value;
      return(t->ns);
    }
    t = t->next;
  }
  /* No rule for c present; if just parsing, return rejection */
  if (op == PARSE) return(REJECT);

  /* Otherwise, add a rule for c to current state */

  /* Install a rule node for c in current state's tlist */
  t = new_trule_node(cs);
  if (t == NULL) return(ERROR);
  t->c = c;

  /* Now specify the next state for this rule */
  if (c == '\0') {
    /* '\0' means end of a word, so set return value for this state */
    /* to v and set the next state for this rule to ACCEPT */
    return_value = state[cs]->return_value = v;
    t->ns = ACCEPT;
  }
  else {
    /* Not '\0' means another char in word, so create a new state */
    t->ns = new_state();
    if (t->ns == ERROR) return(ERROR);
  }

  return(t->ns);
}

/*-----------------------------------------------------------------------
 * Function:	new_trule_node
 * Description:	return a ptr to a new trule node in the tlist of a state
 * Args  IN:	n: index of state to add node to
 * Returns:	ptr to newly allowcated rule node
 */
static Trule *
  new_trule_node(n)
int n;
{
  Trule *t, *tnew;

  /* Allocate and initialize the node */
  tnew = (Trule *)malloc(sizeof(Trule));
  if (tnew == NULL) return(NULL);
  tnew->c = '\1';
  tnew->ns = REJECT;
  tnew->next = NULL;

  /* Install it in tlist of state n */
  if (state[n]->tlist == NULL) {
    /* List is empty: */
    state[n]->tlist = tnew;
  }
  else {
    /* List is not empty; set t = last node in list, then add tnew to end */
    t = state[n]->tlist;
    while (t->next != NULL)  t = t->next;
    t->next = tnew;
  }

  /* Return ptr to the new trule node */
  return(tnew);
}

/*-----------------------------------------------------------------------
 * Function:	new_state
 * Description:	get a new state
 * Args:	(none)
 * Returns:	index (an int) of new state
 */
static
  new_state()
{
  if (state_count == MAX_STATES) return(ERROR);
  state[state_count] = (State*)malloc(sizeof(State));
  if (state[state_count] == NULL) return(ERROR);
  state[state_count]->return_value = reject_value;
  state[state_count]->tlist = NULL;
  ++state_count;
  return(state_count-1);
}

/*-----------------------------------------------------------------------
 * Function:	delete_trule_list
 * Description:	Delete a trule list, freeing up its nodes
 * Args  IN:	tlist: ptr to first node in list
 */
static
  delete_trule_list(tlist)
Trule *tlist;
{
  Trule *t;

  while (tlist != NULL) {
    t = tlist;
    tlist = tlist->next;
    free((char*)t);
  }
}
