/*ScianVisContours.c
  June 14, 1991
  Eric Pepke
  Routines for contour visualization object

  The contour object takes a scalar field defined over a data form with
  topological dimension 2.  The spatial dimension of the dataset may be 
  either 2 or 3.  If the spatial dimension is 2, the Z is assumed to be
  zero.  The contour visualization object works for regular data forms,
  curvilinear data forms, and nonstructured data forms.

*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianArrays.h"
#include "ScianWindows.h"
#include "ScianTextBoxes.h"
#include "ScianObjWindows.h"
#include "ScianIcons.h"
#include "ScianColors.h"
#include "ScianControls.h"
#include "ScianLists.h"
#include "ScianSpaces.h"
#include "ScianSliders.h"
#include "ScianIDs.h"
#include "ScianDatasets.h"
#include "ScianErrors.h"
#include "ScianVisObjects.h"
#include "ScianVisWindows.h"
#include "ScianVisContours.h"
#include "ScianStyle.h"
#include "ScianPictures.h"
#include "ScianTitleBoxes.h"
#include "ScianButtons.h"
#include "ScianMethods.h"
#include "ScianDraw.h"

ObjPtr contoursClass;
ObjPtr contourControlClass;		/*Class for a contour control*/

/*Values for color cells.  Don't change these values.*/
#define COLORCELLSNONE		0
#define COLORCELLSGRID		1
#define COLORCELLSDATA		2
#define COLROCELLSCONTOURS	3

static ObjPtr ContoursInit(object)
ObjPtr object;
/*Initializes a contours object*/
{
    ObjPtr minMax;
    real bounds[6];
    real xySize, zSize;
    ObjPtr colorObj, deformObj;

    MakeVar(object, CONTOURFIELD);
    SetVar(object, COLOROBJ, colorObj = GetVar(object, CONTOURFIELD));
    if (colorObj)
    {
	SetVar(object, COLORS, ObjTrue);
    }
    SetVar(object, COLORCELLS, NewInt(1));
    SetVar(object, DEFORMOBJ, deformObj = GetVar(object, CONTOURFIELD));

    /*Heuristically determine whether to displace Z*/
    GetBounds(object, bounds);
    xySize = bounds[1] - bounds[0];
    xySize = MAX(xySize, bounds[3] - bounds[2]);
    xySize = MAX(xySize, bounds[5] - bounds[4]);
    if (GetVar(colorObj, DATA))
    {
	minMax = GetMinMax(deformObj);
	zSize = ((real *) ELEMENTS(minMax))[1] - ((real *) ELEMENTS(minMax))[0];
	if (zSize < xySize * 0.5)
	{
	    SetVar(object, DEFORMSWITCH, NewInt(1));
	}
	else
	{
	    SetVar(object, DEFORMSWITCH, NewInt(0));
	}
    }
}

static ObjPtr DrawContourControl(control)
ObjPtr control;
/*Draws a contour control*/
{
    int left, right, bottom, top;	/*Size of the control*/
    int l, r, b, t;			/*Temporary bounds*/
    int start, diff, comp;
    real min, max;
    ObjPtr value;
    double majorWidth, minorWidth;	/*Width of a major step*/
    double ddiff;			/*Data difference*/
    int nTics;				/*Number of tics*/
    long temp;
    double curValue;
    int k, pixel;

#ifdef GRAPHICS
    Get2DIntBounds(control, &left, &right, &bottom, &top);

    /*Draw the slider portion of the control*/
    l = left;
    r = right;
    b = bottom;
    t = top - 3 * MINORBORDER - 3 * EDITBOXHEIGHT;

    DrawBumpEdge(l, r, b, t);
    FillUIRect(l + 3, r - 3, b + 3, t - 3, UIBACKGROUND);

    /*Draw the real number line*/
    b = bottom + CCBBORDER + CCTEXTUP + CCTEXTSEP;
    l = left + CCLBORDER;
    r = right - CCRBORDER;
    DrawUILine(l, b, r, b, UIBLACK);

    /*Calculate good steps for the stuff*/
    value = GetRealVar("DrawContourControl", control, LOVALUE);
    if (!value)
    {
	return ObjFalse;
    }
    min = GetReal(value);

    value = GetRealVar("DrawContourControl", control, HIVALUE);
    if (!value)
    {
	return ObjFalse;
    }
    max = GetReal(value);

    ddiff = max - min;

    CalcGoodSteps(ddiff,
		  r - l,
		  CCSTEPPIXELS,
		  &majorWidth, &nTics);
    minorWidth = majorWidth / nTics;

    SetUIColor(UIBLACK);
    SetupFont(CCTEXTFONT, CCTEXTSIZE);

    /*Minor and major tics first*/
    temp = min / majorWidth;
    curValue = temp * majorWidth;

    while (curValue > min)
    {
	curValue -= majorWidth;
    }
    k = 0;
    while (curValue < min)
    {
	++k;
	if (k >= nTics) k = 0;

	curValue += minorWidth;
    }

    /*Now actually draw them*/
    if (ABS(curValue) < ddiff * 1.0E-6) curValue = 0.0;
    while (curValue <= max + ddiff * 1.0E-6)
    {
	pixel = l + (curValue - min) * (r - l) / (ddiff);
	if (k == 0)
	{
	    /*Major tic*/
	    DrawUILine(pixel, b,
		       pixel, b - CCMAJORTICLEN,
		       UIBLACK);
	    sprintf(tempStr, "%lg", curValue);
	    DrawString(pixel - StrWidth(tempStr) / 2, b - CCTEXTSEP, tempStr);
	}
	else
	{
	    /*Minor tic*/
	    DrawUILine(pixel, b,
		       pixel, b - CCMAJORTICLEN / 2,
		       UIBLACK);
	}

	curValue += minorWidth;
	if (ABS(curValue) < ddiff * 1.0E-6) curValue = 0.0;
	++k;
	if (k >= nTics) k = 0;
    }
#endif
    return ObjTrue;
}

ObjPtr NewContourControl(left, right, bottom, top, name, min, max)
int left, right, bottom, top;
char *name;
real min, max;
/*Makes a new contour control in left, right, bottom, top*/
{
    ObjPtr retVal;
    
    retVal = NewObject(contourControlClass, 0);
    Set2DIntBounds(retVal, left, right, bottom, top);
    SetVar(retVal, NAME, NewString(name));
    SetVar(retVal, TYPESTRING, NewString("contour control"));
    SetVar(retVal, VALUE, NULLOBJ);
    SetVar(retVal, HIVALUE, NewReal(max));
    SetVar(retVal, LOVALUE, NewReal(min));
    return retVal;
}

static ObjPtr SetContoursMainDataset(visObj, dataSet)
ObjPtr visObj, dataSet;
/*Sets the main data set of visObj to dataSet*/
{
    SetVar(visObj, CONTOURFIELD, dataSet);
    SetVar(visObj, MAINDATASET, dataSet);
    return ObjTrue;
}

static ObjPtr MakeContoursColored(visObject)
ObjPtr visObject;
/*Makes the contours colored*/
{
    SetVar(visObject, PICCOLORED, ObjTrue);
    if (GetPredicate(visObject, COLORS))
    {
	ObjPtr contourField, contourForm;
	ObjPtr colorField, colorForm;
	real brightness;
	ObjPtr var;
	ObjPtr surface;
	ObjPtr palette;

	/*Get the contour field and its form*/
	contourField = GetObjectVar("MakeContoursColored", visObject, CONTOURFIELD);
	if (!contourField) return ObjFalse;
	contourForm = GetObjectVar("MakeContoursColored", contourField, DATAFORM);
	if (!contourForm) return ObjFalse;

	/*Get the color field and its form*/
	colorField = GetObjectVar("MakeContoursColored", visObject, COLOROBJ);
	if (!colorField) return ObjFalse;
	colorForm = GetObjectVar("MakeContoursColored", colorField, DATAFORM);
	if (!colorForm) return ObjFalse;

	/*Get the color palette*/
	MakeVar(colorField, CPALETTE);
	palette = GetPaletteVar("MakeContoursColored", colorField, CPALETTE);
	if (!palette)
	{
	    return ObjFalse;
	}
	SetPalette(palette);

	/*Get the brightness for coloring*/
	var = GetVar(visObject, BRIGHTNESS);
	if (var)
	{
	    brightness = GetReal(var);
	}
	else
	{
	    brightness = 1.0;
	}

	/*Get the surface to color*/
	surface = GetPictureVar("MakePicColored", visObject, SURFACE);
	if (!surface) return ObjFalse;

	/*See if the contour form is the same as the color form*/
	if (EqualForm(colorForm, contourForm))
	{
	    /*Yes!  It's easy and fast to traverse*/
	    long info;
	    info = GetDatasetInfo(colorField);

	    if (info & DS_UNSTRUCTURED)
	    {
		/*It's an unstructured data form*/
		ObjPtr nodes;			/*Nodes of the dataform*/
		ObjPtr faces;			/*Faces of the dataform*/
		threeArray *nodesArray;		/*Array of nodes*/
		PolysPtr polys;			/*The polygons in the surface*/
		PolyPtr polyRunner;		/*The runner through the polys*/
		ThingListPtr cellRunner;	/*The runner through the cells*/
		ObjPtr field;			/*The field itself*/

		nodes = GetDatasetNodes(contourField);
		if (!nodes)
		{
		    return ObjFalse;
		}
		nodesArray = (threeArray *) ArrayMeat(nodes);

		faces = GetDatasetKEdges(contourField, 2);
		if (!faces)
		{
		    return ObjFalse;
		}

		/*Look in the polys of the picture*/
		polys = (PolysPtr) ((PicPtr) surface) -> items;
		if (!polys)
		{
		    return ObjTrue;
		}
		if (polys -> item . type != POLYGONS)
		{
		    ReportError("MakeContoursColored", "Malformed picture");
		    return ObjFalse;
		}

		/*Get the data*/
		if (!SetCurField(FIELD1, contourField))
		{
		    return ObjFalse;
		}

		cellRunner = LISTOF(faces);
		polyRunner = polys -> polygons;  
    
		while (cellRunner)
		{
		    ObjPtr array;
		    int k, d, i, index;
		    real *a;

		    if (!polyRunner)
		    {
			ReportError("MakeContoursColored", "Picture too short");
			break;
		    }
		    array = cellRunner -> thing;
		    d = DIMS(array)[0];
		    a = ArrayMeat(array);

		    if (d == polyRunner -> nVertices)
		    {
			for (k = 0; k < d; ++k)
			{
			    /*Color the vertices of the polygon*/
			    long indices[1];
			    i = *a;
			    indices[0] = i;
			    index = GetRealColorIndex(
				SelectFieldComponent(FIELD1, 0, indices));
			    polyRunner -> vertices[k] . colorIndex = index;
			    ++a;
			}
		    }
		    else
		    {
			ReportError("MakeContoursColored", "Malformed polygon");
		    }

		    cellRunner = cellRunner -> next;
		    polyRunner = (PolyPtr) polyRunner -> item . next;
		}
		if (cellRunner)
		{
		    ReportError("MakeContoursColored", "Picture too long");
		}
	    }
	    else
	    {
		/*Same structured dataset*/
		int topDim;
		long curDim[2], iDims[2];
		ObjPtr dims;
		RectMeshPtr rectMesh;
		real temp[3];
		ObjPtr var;
		int colorCells;

		/*It must have dimension 2*/
		topDim = GetTopDim(contourField);
		if (topDim != 2)
		{
		    ReportError("MakeContoursColored", "Topological dimension must be 2.");
		    return ObjFalse;
		}

		/*Get the actual topological dimensions dimension*/
		dims = GetDatasetFormDims(contourField);
		if (!dims || !IsArray(dims) || RANK(dims) != 1 || DIMS(dims)[0] != 2)
		{
		    return ObjFalse;
		}

		iDims[0] = ((real *) ELEMENTS(dims))[0];
		iDims[1] = ((real *) ELEMENTS(dims))[1];

		/*Register the contour field*/
		if (!SetCurField(FIELD1, contourField))
		{
		    return ObjFalse;
		}

		/*See what kind of color cells it has*/
		var = GetIntVar("MakeContoursColored", visObject, COLORCELLS);
		if (var)
		{
		    colorCells = GetInt(var);
 		}
		else
		{
		    colorCells = 1;
		}
		if (colorCells)
		{
		    real sample;
		    int index;

		    int iDim, jDim;

		    if (GetPredicate(visObject, REVERSESENSE))
		    {
			iDim = 1;
			jDim = 0;
		    }
		    else
		    {
			iDim = 0;
			jDim = 1;
		    }

		    /*Get the rectangular mesh*/
		    rectMesh = (RectMeshPtr) ((PicPtr) surface) -> items;
		    if (!rectMesh)
		    {
			return ObjFalse;
		    }

		    if (colorCells == COLORCELLSGRID)
		    {
			/*Color cells between the grid lines*/

			for (curDim[iDim] = 0; curDim[iDim] < iDims[iDim]; ++curDim[iDim])
			{
			    for (curDim[jDim] = 0; curDim[jDim] < iDims[jDim]; ++curDim[jDim])
			    {
				sample = SelectFieldComponent(FIELD1, 0, curDim);
				index = GetRealColorIndex(sample);
				SetRectMeshVColorIndex(rectMesh, 
					curDim[iDim], curDim[jDim], 
					index);
			    }
			}

			InterpRectCenters(rectMesh);

		    }
		    else if (colorCells == COLORCELLSDATA)
		    {
			/*Color cells around the data points*/

			for (curDim[iDim] = 0; curDim[iDim] < iDims[iDim]; ++curDim[iDim])
			{
			    for (curDim[jDim] = 0; curDim[jDim] < iDims[jDim]; ++curDim[jDim])
			    {
				sample = SelectFieldComponent(FIELD1, 0, curDim);
				index = GetRealColorIndex(sample);
				SetRectMeshCColorIndex(rectMesh, 
					curDim[iDim], curDim[jDim], 
					index);
			    }
			}

			InterpRectVertices(rectMesh);
		    }
		}
	    }
	}
	else
	{
	    /*Have to make it colored by the object*/
	    ColorPictureByObject(surface, colorField, GetPredicate(visObject, INTERPCOLORS));
	}
    }
    return ObjTrue;
}

static ObjPtr MakeContoursSurface(visObject)
ObjPtr visObject;
/*Makes the surface in a contours object.  Also colors it.*/
{
    ObjPtr dataset;		/*The dataset the vis object represents*/
    long datasetFlags;		/*Flags of the dataset*/
    ObjPtr var;			/*Random variable*/
    int colorCells;		/*What kind of color cells*/
    ObjPtr picture;		/*The picture to be made*/
    ObjPtr deformObj;		/*Deformed object*/
    Bool deformAtAll;		/*True iff deform at all*/
    Bool deformByField;		/*True iff deform by field*/
    Bool deformBySame;		/*True iff deform by same object*/
    real deformFixed;		/*Fixed deformation*/
    real deformFactor;		/*Deformation factor*/
    real deformOffset;		/*Deformation offset*/

    dataset = GetObjectVar("MakeContoursSurface", visObject, CONTOURFIELD);
    if (!dataset)
    {
	return ObjFalse;
    }

    datasetFlags = GetDatasetInfo(dataset);

    if (0 == datasetFlags & DS_HASFORM)
    {
	ReportError("MakeContoursSurface", "No data form");
	return ObjFalse;
    }

    /*See what kind of color cells it has*/
    var = GetIntVar("MakeContoursSurface", visObject, COLORCELLS);
    if (var)
    {
	colorCells = GetInt(var);
    }
    else
    {
	colorCells = 1;
    }

    var = GetVar(visObject, DEFORMSWITCH);
    if (var)
    {
	deformAtAll = true;
	deformByField = GetInt(var) ? true : false;
    }
    else
    {
	deformAtAll = true;
	deformByField = false;
    }
    if (deformByField)
    {
	deformObj = GetVar(visObject, DEFORMOBJ);
	if (!deformObj)
	{
	    deformByField = false;
	    deformAtAll = false;
	}
    }
    if (deformAtAll)
    {
	var = GetVar(visObject, DEFFACTOR);
	if (var)
	{
	    deformFactor = GetReal(var);
	}
	else
	{
	    deformFactor = 0.0;
	}
	var = GetVar(visObject, DEFOFFSET);
	if (var)
	{
	    deformOffset = GetReal(var);
	}
	else
	{
	    deformOffset = 0.0;
	}
	if (deformOffset == 0.0 && deformFactor == 0.0)
	{
	    deformAtAll = false;
	}
    }

    if (deformAtAll)
    {
    if (deformByField)
    {
	deformBySame = (dataset == deformObj) ? true : false;
    }
    else
    {
	var = GetVar(visObject, DEFCONSTANT);
	if (var)
	{
	    deformFixed = GetReal(var);
	}
	else
	{
	    deformFixed = 0.0;
	}
    }
    }

    /*Make the new picture*/
    picture = NewPicture();
    if (!picture) return ObjFalse;

    if (datasetFlags & DS_UNSTRUCTURED)
    {
	/*It's an unstructured dataset.*/
	ObjPtr nodes;
	ObjPtr edges;			/*Edges of the dataform*/
	ObjPtr faces;			/*Faces of the dataform*/
	threeArray *nodesArray;		/*Array of nodes*/
	twoArray *edgesArray;		/*Array of edges*/
	PolysPtr polys;			/*The polygons in the surface*/
	ThingListPtr runner;		/*The runner through the cells*/
	VertexPtr vertices;		/*Temporary holding place for vertices*/
	long allocVertices;		/*Number of vertices allocated*/

	nodes = GetDatasetNodes(dataset);
	if (!nodes)
	{
	    return ObjFalse;
	}
	nodesArray = (threeArray *) ArrayMeat(nodes);

	edges = GetDatasetKEdges(dataset, 1);
	if (!edges)
	{
	    return ObjFalse;
	}
	edgesArray = (twoArray *) ArrayMeat(edges);

	faces = GetDatasetKEdges(dataset, 2);
	if (!faces)
	{
	    return ObjFalse;
	}

	if (colorCells)
	{
	    /*Make the picture and set it up with one polys*/
	    polys = NewPolygons();
	    if (!polys) return ObjFalse;
	    AppendItemToPicture((PicPtr) picture, (PicItemPtr) polys);

	    allocVertices = 20;
	    vertices = (VertexPtr) malloc(20 * sizeof(VertexTuple));
	    if (!vertices)
	    {
		return ObjFalse;
	    }
	
	    runner = LISTOF(faces);    
	    while (runner)
	    {
		ObjPtr array;
		int k, d, i;
		real *a;

		array = runner -> thing;
		d = DIMS(array)[0];
		a = ArrayMeat(array);

		for (k = 0; k < d; ++k)
		{
		    i = *a;
		    
		    if (k >= allocVertices)
		    {
			allocVertices += 20;
			vertices = (VertexPtr) realloc(vertices, allocVertices * sizeof(VertexTuple));
			if (!vertices)
			{
			    return ObjFalse;
			}
		    }
		    vertices[k] . position[0] = nodesArray[i][0];
		    vertices[k] . position[1] = nodesArray[i][1];
		    vertices[k] . position[2] = nodesArray[i][2];

		    /****UPDATE*** assume normal in z direction only*/
		    vertices[k] . normal[0] = 0.0;
		    vertices[k] . normal[1] = 0.0;
		    vertices[k] . normal[2] = 1.0;

		    ++a;
		}
		AppendPolyToPolys(polys, d, vertices);

		runner = runner -> next;
	    }
	    free(vertices);
	}
    }
    else
    {
	/*It's a structured dataset.*/
	int topDim, nComponents;
	long curDim[2], iDims[2];
	ObjPtr dims;
	RectMeshPtr rectMesh;
	real temp[3];

	/*It must have dimension 2*/
	topDim = GetTopDim(dataset);
	if (topDim != 2)
	{
	    ReportError("MakeContoursSurface", "Topological dimension must be 2.");
	    return ObjFalse;
	}

	/*Get the actual topological dimensions*/
	dims = GetDatasetFormDims(dataset);
	if (!dims || !IsArray(dims) || RANK(dims) != 1 || DIMS(dims)[0] != 2)
	{
	    ReportError("MakeContoursSurface", "No topological dimensions");
	    return ObjFalse;
	}

	iDims[0] = ((real *) ELEMENTS(dims))[0];
	iDims[1] = ((real *) ELEMENTS(dims))[1];

	/*Register the dataset and its dataform*/
	if (!SetCurField(FIELD1, dataset))
	{
	    return ObjFalse;
	}
	if (!SetCurForm(FIELD2, dataset))
	{
	    return ObjFalse;
	}

	if (deformByField && !deformBySame)
	{
	    /*Register the deformation field*/
	    if (!SetCurField(FIELD3, deformObj))
	    {
		return ObjFalse;
	    }
	    if (!SetCurForm(FIELD4, deformObj))
	    {
		return ObjFalse;
	    }
	}

	if (colorCells)
	{
	    int iDim, jDim;

	    if (GetPredicate(visObject, REVERSESENSE))
	    {
		iDim = 1;
		jDim = 0;
	    }
	    else
	    {
		iDim = 0;
		jDim = 1;
	    }

	    if (colorCells == COLORCELLSGRID)
	    {
		/*Color cells between the grid lines*/
	
		/*Create a new rectangular mesh*/
		rectMesh = NewRectMesh(iDims[iDim], iDims[jDim], false);
		if (!rectMesh)
		{
		    return ObjFalse;
		}

		nComponents = GetNComponents(FIELD2);

		for (curDim[iDim] = 0; curDim[iDim] < iDims[iDim]; ++curDim[iDim])
		{
		    for (curDim[jDim] = 0; curDim[jDim] < iDims[jDim]; ++curDim[jDim])
		    {
			SetRectMeshVPosition(rectMesh, curDim[iDim], curDim[jDim],
				SelectFieldComponent(FIELD2, 0, curDim),
				SelectFieldComponent(FIELD2, 1, curDim),
				nComponents >= 3 ? SelectFieldComponent(FIELD2, 2, curDim) : 0.0);
		    }
		}

		InterpRectCenters(rectMesh);

		CalcRectNormals(rectMesh);

		if (deformAtAll)
		{
		    /*Must deform it*/

		    for (curDim[iDim] = 0; curDim[iDim] < iDims[iDim]; ++curDim[iDim])
		    {
		        for (curDim[jDim] = 0; curDim[jDim] < iDims[jDim]; ++curDim[jDim])
		        {
			    real p[3], n[3], sample;
			    GetRectMeshVPosition(rectMesh, curDim[iDim], curDim[jDim], &(p[0]), &(p[1]), &(p[2]));
			    GetRectMeshVNormal(rectMesh, curDim[iDim], curDim[jDim], &(n[0]), &(n[1]), &(n[2]));
			    if (deformByField)
			    {
				if (deformBySame)
				{
				    sample = SelectFieldComponent(FIELD1, 0, curDim);
				}
				else
				{
				    SampleSpatField(FIELD3, FIELD4, 1, &sample,
					3, p, false);
				}
			    }
			    else
			    {
				sample = deformFixed;
			    }
			    if (sample != missingData)
			    {
				/*Deform the sample*/
				sample = sample * deformFactor + deformOffset;
				p[0] += n[0] * sample;
				p[1] += n[1] * sample;
				p[2] += n[2] * sample;
				SetRectMeshVPosition(rectMesh, curDim[iDim], curDim[jDim], p[0], p[1], p[2]);
			    }

		        }
		    }
		    InterpRectCenters(rectMesh);

		    CalcRectNormals(rectMesh);
		}

		/*Append rectangular mesh to the picture*/
		AppendItemToPicture((PicPtr) picture, (PicItemPtr) rectMesh);
	    }
	    else if (colorCells == COLORCELLSDATA)
	    {
		/*Color cells around the data points*/

		/*Create a new rectangular mesh*/
		rectMesh = NewRectMesh(iDims[iDim] + 1, iDims[jDim] + 1, true);
		if (!rectMesh)
		{
		    return ObjFalse;
		}

		nComponents = GetNComponents(FIELD2);

		for (curDim[iDim] = 0; curDim[iDim] < iDims[iDim]; ++curDim[iDim])
		{
		    for (curDim[jDim] = 0; curDim[jDim] < iDims[jDim]; ++curDim[jDim])
		    {
			SetRectMeshCPosition(rectMesh, curDim[iDim], curDim[jDim],
				SelectFieldComponent(FIELD2, 0, curDim),
				SelectFieldComponent(FIELD2, 1, curDim),
				(nComponents >= 3 ? SelectFieldComponent(FIELD2, 2, curDim) : 0.0));
		    }
		}
		InterpRectVertices(rectMesh);
		CalcRectNormals(rectMesh);

		if (deformAtAll)
		{
		    /*Must deform it*/

		    for (curDim[iDim] = 0; curDim[iDim] < iDims[iDim]; ++curDim[iDim])
		    {
		        for (curDim[jDim] = 0; curDim[jDim] < iDims[jDim]; ++curDim[jDim])
		        {
			    real p[3], n[3], sample;
			    GetRectMeshCPosition(rectMesh, curDim[iDim], curDim[jDim], &(p[0]), &(p[1]), &(p[2]));
			    GetRectMeshCNormal(rectMesh, curDim[iDim], curDim[jDim], &(n[0]), &(n[1]), &(n[2]));
			    if (deformByField)
			    {
				if (deformBySame)
				{
				    sample = SelectFieldComponent(FIELD1, 0, curDim);
				}
				else
				{
				    SampleSpatField(FIELD3, FIELD4, 1, &sample,
					3, p, false);
				}
			    }
			    else
			    {
				sample = deformFixed;
			    }
			    if (sample != missingData)
			    {
				/*Deform the sample*/
				sample = sample * deformFactor + deformOffset;
				p[0] += n[0] * sample;
				p[1] += n[1] * sample;
				p[2] += n[2] * sample;
				SetRectMeshCPosition(rectMesh, curDim[iDim], curDim[jDim], p[0], p[1], p[2]);
			    }

		        }
		    }
		    InterpRectVertices(rectMesh);

		    CalcRectNormals(rectMesh);
		}

		/*Append rectangular mesh to the picture*/
		AppendItemToPicture((PicPtr) picture, (PicItemPtr) rectMesh);
	    }
	}
    }

    SetVar(visObject, SURFACE, picture);
    SetVar(picture, REPOBJ, visObject);
    return ObjTrue;
}

static ObjPtr ChangeContourColorCells(button)
ObjPtr button;
/*Changes the color cells as a result of a change in the value of button*/
{
    ObjPtr repObj;
    ObjPtr value;

    repObj = GetObjectVar("ChangeContourColorCells", button, REPOBJ);
    if (!repObj)
    {
	return ObjFalse;
    }

    value = GetValue(button);
    SetVar(repObj, COLORCELLS, NewInt(GetInt(value)));
    ImInvalid(repObj);
    return ObjTrue;
}

#if 0
static ObjPtr DropInContourCorral(corral, object, x, y)
ObjPtr corral, object;
int x, y;
/*Drops an icon in a contour field corral*/
{
    ObjPtr visObj;
    ObjPtr fieldObj;
    ObjPtr icon;
    ObjPtr name;
    ObjPtr defaultIcon;
    ObjPtr contents;

    /*Find the visualization object*/
    visObj = GetObjectVar("DropInContourCorral", corral, REPOBJ);
    if (!visObj)
    {
	return ObjFalse;
    }

    /*Get the field object*/
    fieldObj = GetObjectVar("DropInContourCorral", object, REPOBJ);
    if (!fieldObj)
    {
	return ObjFalse;
    }

    /*Create an icon for it*/
    name = GetStringVar("DropInContourCorral", fieldObj, NAME);
    if (!name)
    {
	return ObjFalse;
    }

    defaultIcon = GetVar(fieldObj, DEFAULTICON);
    if (defaultIcon)
    {
	ObjPtr locArray;
	real loc[2];
	icon = NewObject(defaultIcon, 0);
	SetVar(icon, NAME, name);
	loc[0] = x;
	loc[1] = y;
	locArray = NewRealArray(1, 2L);
	CArray2Array(locArray, loc);
	SetVar(icon, ICONLOC, locArray);
    }
    else
    {
	icon = NewIcon(x, y, ICONQUESTION, GetString(name));
    }
    
    /*Make the icon point to the field*/
    SetVar(icon, REPOBJ, fieldObj);

    /*Make it the only icon in the corral*/
    contents = NewList();
    PrefixList(contents, icon);
    SetVar(corral, CONTENTS, contents);
    SetVar(contents, PARENT, corral);
    SetVar(icon, PARENT, corral);
    RecalcScroll(corral);

    /*Make this object the colored object*/
    SetVar(visObj, CONTOURFIELD, fieldObj);
    ImInvalid(visObj);

    ImInvalid(corral);
}
#endif

static ObjPtr AddContourControls(contour, panelContents)
ObjPtr contour, panelContents;
/*Adds controls appropriate to a contour object to panelContents*/
{
    ObjPtr titleBox, button, radio, var, corral, icon, name, contourField, mainDataset;
    ObjPtr textBox, defaultIcon;
    int width, left, top, bottom, right;
    ObjPtr control;
    ObjPtr minMax;
    real min, max, *elements;

    width = CWINWIDTH - 2 * CORRALBORDER - CWINCORRALWIDTH;

    /*Get the contour field*/
    contourField = GetObjectVar("AddContourControls", contour, CONTOURFIELD);
    if (!contourField) return ObjFalse;
    while (mainDataset = GetVar(contourField, MAINDATASET))
    {
	contourField = mainDataset;
    }

    /*Find its min and max*/
    minMax = GetMinMax(contourField);
    if (minMax)
    {
	elements = ELEMENTS(minMax);
	min = elements[0];
	max = elements[1];
    }
    else
    {
	min = 0.0;
	max = 1.0;
    }

    /*Create the color cells title box*/
    right = width;
    left = right - (CCBUTTONLENGTH + 2 * MINORBORDER);
    top = CWINHEIGHT - MINORBORDER;
    bottom = top - (2 * MINORBORDER + 4 * CHECKBOXHEIGHT + 3 * CHECKBOXSPACING + TITLEBOXTOP);

    titleBox = NewTitleBox(left, right, bottom, top, "Color Cells");
    PrefixList(panelContents, titleBox);
    SetVar(titleBox, PARENT, panelContents);

    /*Create the radio button group*/
    radio = NewRadioButtonGroup("Color Cells Radio");
    PrefixList(panelContents, radio);
    SetVar(radio, PARENT, panelContents);
    SetVar(radio, REPOBJ, contour);

    /*Create the buttons*/
    left += MINORBORDER;
    right -= MINORBORDER;
    top -= TITLEBOXTOP + MINORBORDER;
    bottom = top - CHECKBOXHEIGHT;
    button = NewRadioButton(left, right, bottom, top, "None");
    AddRadioButton(radio, button);

    top = bottom - CHECKBOXSPACING;
    bottom = top - CHECKBOXHEIGHT;
    button = NewRadioButton(left, right, bottom, top, "Between Grid Lines");
    AddRadioButton(radio, button);

    top = bottom - CHECKBOXSPACING;
    bottom = top - CHECKBOXHEIGHT;
    button = NewRadioButton(left, right, bottom, top, "Around Data Points");
    AddRadioButton(radio, button);

    top = bottom - CHECKBOXSPACING;
    bottom = top - CHECKBOXHEIGHT;
    button = NewRadioButton(left, right, bottom, top, "Between Contours");
    AddRadioButton(radio, button);
    ActivateButton(button, false);

    /*Set the radio according to COLORCELLS*/
    var = GetIntVar("AddContourControls", contour, COLORCELLS);
    if (var)
    {
	SetValue(radio, var);
    }

    SetMethod(radio, CHANGEDVALUE, ChangeContourColorCells);

    /*Create a new contour control*/
    left = MAJORBORDER;
    right = width;
    bottom = MINORBORDER;
    top = CWINHEIGHT - 2 * MINORBORDER - (2 * MINORBORDER + 4 * CHECKBOXHEIGHT + 3 * CHECKBOXSPACING + TITLEBOXTOP);
    control = NewContourControl(left, right, bottom, top, "Line Contours", min, max);
    PrefixList(panelContents, control);
    SetVar(control, PARENT, panelContents);
    SetVar(control, REPOBJ, contour);
    SetVar(control, HELPSTRING,
	NewString("This control allows you to create and edit line contours.  \
I haven't quite decided how it should work just yet."));  /***UPDATE***/

    /*Put in the contour corral at the top left*/
    left = MAJORBORDER;
    top = MAJORBORDER;
    corral = NewIconCorral(NULLOBJ,
			   left, left + ONECORRALWIDTH,
			   CWINHEIGHT - MAJORBORDER - ONECORRALHEIGHT,
			   CWINHEIGHT - MAJORBORDER, 0);
    SetVar(corral, SINGLECORRAL, ObjTrue);
    SetVar(corral, TOPDOWN, ObjTrue);
    SetVar(corral, NAME, NewString("Contour Field"));
    SetVar(corral, HELPSTRING,
	NewString("This corral shows the dataset that is being used to make \
the contour display.  The locations of color cells and contours are calculated \
using this field.  The color of color cells is calculated using the Color Field, \
available in the Color control set."));
    PrefixList(panelContents, corral);
    SetVar(corral, PARENT, panelContents);
    SetVar(corral, REPOBJ, contour);
    SetMethod(corral, DROPINCONTENTS, DropInMainDatasetCorral);

    /*Create the contour source text box*/
    textBox = NewTextBox(left, left + ONECORRALWIDTH, 
			 CWINHEIGHT - MAJORBORDER - ONECORRALHEIGHT - TEXTBOXSEP - TEXTBOXHEIGHT,
			 CWINHEIGHT - MAJORBORDER - ONECORRALHEIGHT - TEXTBOXSEP,
			 0, "Contour Field Text", "Contour Field");
    PrefixList(panelContents, textBox);
    SetVar(textBox, PARENT, panelContents);
    SetTextAlign(textBox, CENTERALIGN);

    /*Put in an icon that represents the field*/
    name = GetVar(contourField, NAME);
    defaultIcon = GetVar(contourField, DEFAULTICON);
    if (defaultIcon)
    {
	icon = NewObject(defaultIcon, 0);
	SetVar(icon, NAME, name);
    }
    else
    {
	icon = NewIcon(0, 0, ICONQUESTION, GetString(name));
    }
    SetVar(icon, ICONLOC, NULLOBJ);
    SetVar(icon, REPOBJ, contourField);
    DropIconInCorral(corral, icon);

    return ObjTrue;
}

void InitContours()
/*Initializes the contour objects*/
{
    ObjPtr icon;

    /*Class for a color contour control*/
    contourControlClass = NewObject(controlClass, 0);
    AddToReferenceList(contourControlClass);
#ifdef GRAPHICS
    SetMethod(contourControlClass, DRAW, DrawContourControl);
#endif
#if 0
#ifdef INTERACTIVE
    SetMethod(colorBarClass, PRESS, PressColorBar);
#endif
    SetMethod(colorBarClass, SETVAL, SetColorBarVal);
#endif

    /*Class for a contour object*/
    contoursClass = NewObject(visDeformed, 0);
    AddToReferenceList(contoursClass);
    SetVar(contoursClass, NAME, NewString("Contours"));
    SetMethod(contoursClass, INITIALIZE, ContoursInit);
    SetVar(contoursClass, SHINVAL, NewReal(80.0));
    SetVar(contoursClass, SPECVAL, NewReal(0.2));
    SetVar(contoursClass, DEFAULTICON, icon = NewObject(visIcon, 0));
    SetVar(icon, WHICHICON, NewInt(ICONCONTOURS));
    SetVar(icon, NAME, NewString("Contours"));
    SetVar(icon, HELPSTRING,
	NewString("This icon represents a contour visualization object.  \
The contour object shows contours and shaded color surfaces of 2-dimensional \
scalar fields defined over structured or nonstructured grids."));
    DeclareIndirectDependency(contoursClass, SURFACE, CONTOURFIELD, DATA);
    DeclareIndirectDependency(contoursClass, SURFACE, CONTOURFIELD, CURDATA);
    DeclareDependency(contoursClass, SURFACE, COLORCELLS);
    SetMethod(contoursClass, SURFACE, MakeContoursSurface);
    DeclareDependency(contoursClass, SURFACE, REVERSESENSE);
    SetMethod(contoursClass, PICCOLORED, MakeContoursColored);
    SetMethod(contoursClass, SETMAINDATASET, SetContoursMainDataset);

    SetMethod(contoursClass, ADDCONTROLS, AddContourControls);
    icon = NewIcon(0, 0, ICONCONTOURS, "Contours");
    SetMethod(icon, CHANGEDVALUE, ChangedControlIcon);
    SetVar(contoursClass, CONTROLICON, icon);

    DefineVisMapping(DS_HASFORM | DS_HASFIELD, 2, 3, 1, contoursClass);
    DefineVisMapping(DS_HASFORM | DS_HASFIELD, 2, 2, 1, contoursClass);
    DefineVisMapping(DS_HASFORM | DS_HASFIELD | DS_UNSTRUCTURED, 2, 3, 1, contoursClass);
    DefineVisMapping(DS_HASFORM | DS_HASFIELD | DS_UNSTRUCTURED, 2, 2, 1, contoursClass);
}

void KillContours()
/*Kills the contours*/
{
    DeleteThing(contoursClass);
    DeleteThing(contourControlClass);
}
