
#include "rtep.h"

/***************************************************************************/
/* FUNCTION: int seqpal_io()
/*
/* DESCRIPTION:
/*    Reads in a "seq" format color look up table with 256
/*    RGB entries.
/*
/* USAGE:
/*    error_ret = seqpal_io(fname,palio_cmd,palette);
/*
/* ARGUMENTS:
/*    char           *fname        (in)     : filename of color palette
/*    char           *palio_cmd    (in)     : i/o operation (r,w,rb,wb)
/*    unsigned char  palette[][3]  (in/out) : array of intensities for
/*                                          : each of the 256 color entries
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*    1 : i/o open error with palio_cmd = "rb"
/*    2 : i/o read error with palio_cmd = "rb"
/*    3 : i/o open error with palio_cmd = "wb"
/*    4 : i/o write error with palio_cmd = "wb"
/*    5 : i/o open error with palio_cmd = "r"
/*    6 : i/o read error with palio_cmd = "r"
/*    7 : i/o open error with palio_cmd = "w"
/*    8 : i/o write error with palio_cmd = "w"
/*    9 : invalid palio_cmd argument
/*
/* COMMENTS:
/*    'palio_cmd' can have values (r,w,rb,wb) where
/*        r  - read ascii     w  - write ascii
/*        rb - read binary    wb - write binary
/*    Binary format consists of 256 bytes of red intensities followed
/*    by 256 bytes of green intensities and 256 bytes of blue intensities.
/*    Ascii format contains 256 lines of clear text encoded integers,
/*    three integers per line: R G B.  Intensities are 0 - minimum,
/*    255 - maximum.  palette[][0] = red, palette[][1] = green, and
/*    palette[][2] = blue.
/*
/* EXAMPLE:
/* SEE ALSO:
/* INFO:
/*    Author          : Ray Idaszak
/*    Date            : March 3, 1989
/*    Last Modified   : May 15, 1989
/*    email           : rayi@ncsa.uiuc.edu
/***************************************************************************/

#define BUFF_SIZE 768
#define PALBUFSIZE 256
#define ONE 1

int seqpal_io(fname,palio_cmd,palette)
char *fname,*palio_cmd;
unsigned char palette[][3];
{
	register int i;
	FILE *fd1;
	int num_rec,num_vals,err_ret;

	err_ret = 0;
	if (!strcmp(palio_cmd,"rb")){
		unsigned char buff[BUFF_SIZE];
		unsigned char *buf_ptr,*r2,*g2,*b2;
		if ((fd1=fopen(fname,palio_cmd))==NULL){
			err_ret = 1;
			goto error;
		}
		num_rec=fread(buff,sizeof(buff),ONE,fd1);
		if (num_rec<ONE){
			err_ret = 2;
			fclose(fd1);
			goto error;
		}
		fclose(fd1);
		buf_ptr = &palette[0][0];
		r2 = &buff[0];
		g2 = &buff[256];
		b2 = &buff[512];
		for(i=0; i<PALBUFSIZE; i++){
			*buf_ptr++ = *r2++;
			*buf_ptr++ = *g2++;
			*buf_ptr++ = *b2++;
		}
	}
	else if (!strcmp(palio_cmd,"wb")){
		unsigned char buff[BUFF_SIZE];
		unsigned char *buf_ptr,*r2,*g2,*b2;
		buf_ptr = &palette[0][0];
		r2 = &buff[0];
		g2 = &buff[256];
		b2 = &buff[512];
		for(i=0; i<PALBUFSIZE; i++){
			*r2++ = *buf_ptr++;
			*g2++ = *buf_ptr++;
			*b2++ = *buf_ptr++;
		}
		if ((fd1=fopen(fname,palio_cmd))==NULL){
			err_ret = 3;
			goto error;
		}
		num_rec=fwrite(buff,sizeof(buff),ONE,fd1);
		if (num_rec<ONE){
			err_ret = 4;
			fclose(fd1);
			goto error;
		}
		fclose(fd1);
	}
	else if (!strcmp(palio_cmd,"r")){
		int ir,ig,ib;
		if ((fd1=fopen(fname,palio_cmd))==NULL){
			err_ret = 5;
			goto error;
		}
		for(i=0; i<256; i++){
			num_vals = fscanf(fd1,"%d %d %d\n",&ir,&ig,&ib);
			if (ferror(fd1)){
				err_ret = 6;
				fclose(fd1);
				goto error;
			}
			if (num_vals != 3){
				err_ret = 6;
				fclose(fd1);
				goto error;
			}
			palette[i][0] = (unsigned char)ir;
			palette[i][1] = (unsigned char)ig;
			palette[i][2] = (unsigned char)ib;
		}
		fclose(fd1);
	}
	else if (!strcmp(palio_cmd,"w")){
		int ir,ig,ib;
		if ((fd1=fopen(fname,palio_cmd))==NULL){
			err_ret = 7;
			goto error;
		}
		for(i=0; i<256; i++){
			ir = (int)palette[i][0];
			ig = (int)palette[i][1];
			ib = (int)palette[i][2];
			num_vals = fprintf(fd1,"%d %d %d\n",ir,ig,ib);
			if (ferror(fd1)){
				err_ret = 8;
				fclose(fd1);
				goto error;
			}
		}
		fclose(fd1);
	}
	else{
		err_ret = 9;
		goto error;
	}

error:
#if RTE_PRERR
	err_msg("seqpal_io",err_ret);
#endif
	return(err_ret);

}

