/*
  File: Intersect.c
  Author: David Meyers
  LastModified: 2 July 1990
  Purpose: Intersection in 2 dimensions
 */

#include <math.h>
#include <TypeDefinitions.h>

/**************************************************************************/
/*
  Computes the cross product of the vectors defined by P1-->P2 and P1-->P3.
  If P3 is to the LEFT of P1-->P2, the result is positive. If to the RIGHT,
  the result is negative. If the points are colinear, the result is 0.
 */
double CrossProduct(x1, y1, x2, y2, x3, y3)
 double x1, y1, x2, y2, x3, y3;
{
 double dx1, dy1, dx2, dy2;

 dx1 = x2-x1; dy1 = y2-y1;
 dx2 = x3-x1; dy2 = y3-y1;
 return(dx1*dy2 - dx2*dy1);
}

/**************************************************************************/
/*
  If SameSide returns a negative value, then the points P3 and P4 are on
  opposite sides of the line P1-P2. If 0 then one or both of the points are
  colinear with P1-P2. If positive, then both points are on the same side of
  P1-P2.
 */
double SameSide(x1, y1, x2, y2, x3, y3, x4, y4)
 double x1, x2, x3, y1, y2, y3, x4, y4;
{
 double dx, dy, dx1, dy1, dx2, dy2;

 dx = x2-x1; dy = y2-y1;
 dx1 = x3-x1; dy1 = y3-y1;
 dx2 = x4-x2; dy2 = y4-y2;
 return((dx*dy1 - dy*dx1)*(dx*dy2 - dy*dx2));
}

/**************************************************************************/

extern int Intersect(X1, Y1, X2, Y2, X3, Y3, X4, Y4)
 double X1, Y1, X2, Y2, X3, Y3, X4, Y4;
{
 double Same12, Same34;

 Same34 = SameSide(X1, Y1, X2, Y2, X3, Y3, X4, Y4);
 if(Same34 > EPSILON)
  return(FALSE); /* P3, P4 are both on the same side of line P1-P2 */

 Same12 = SameSide(X3, Y3, X4, Y4, X1, Y1, X2, Y2);
 if (Same12 > EPSILON)
  return(FALSE); /* P1, P2 are both on the same side of line P3-P4 */

 if ((fabs(Same12) > EPSILON) && (fabs(Same34) > EPSILON))
  return(TRUE);  /* The segments intersect between both sets of endpoints */

 if ((X1 == X3) && (Y1 == Y3))
  {
   if(fabs(CrossProduct(X1, Y1, X2, Y2, X4, Y4)) > EPSILON)
    return(FALSE); /* share an endpoint but not colinear */
  }
 if ((X1 == X4) && (Y1 == Y4))
  {
   if(fabs(CrossProduct(X1, Y1, X2, Y2, X3, Y3)) > EPSILON)
    return(FALSE); /* share an endpoint but not colinear */
  }
 if ((X2 == X3) && (Y2 == Y3))
  {
   if(fabs(CrossProduct(X2, Y2, X1, Y1, X4, Y4)) > EPSILON)
    return(FALSE); /* share an endpoint but not colinear */
  }
 if ((X2 == X4) && (Y2 == Y4))
  {
   if(fabs(CrossProduct(X2, Y2, X3, Y3, X1, Y1)) > EPSILON)
    return(FALSE); /* share an endpoint but not colinear */
  }

 if ((((X1 < X3) && (X1 > X4)) || ((X1 > X3) && (X1 < X4))) ||
     (((X2 < X3) && (X2 > X4)) || ((X2 > X3) && (X2 < X4))) ||
     (((X3 < X1) && (X3 > X2)) || ((X3 > X1) && (X3 < X2))) ||
     (((X4 < X1) && (X4 > X2)) || ((X4 > X1) && (X4 < X2)))    )
  return(TRUE);  /* colinear with overlapping X extents */

 if ((((Y1 < Y3) && (Y1 > Y4)) || ((Y1 > Y3) && (Y1 < Y4))) ||
     (((Y2 < Y3) && (Y2 > Y4)) || ((Y2 > Y3) && (Y2 < Y4))) ||
     (((Y3 < Y1) && (Y3 > Y2)) || ((Y3 > Y1) && (Y3 < Y2))) ||
     (((Y4 < Y1) && (Y4 > Y2)) || ((Y4 > Y1) && (Y4 < Y2)))    )
  return(TRUE);  /* colinear with overlapping Y extents */
 /* none of the intersection tests have succeeded, so we return FALSE */
 return(FALSE);
}

