/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*	sexp.c: S-expression support */
/* Message bus sexp manipulation functions */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/lib/RCS/sexp.c,v $ */

static char rcsid[] = "sexp.c $Revision: 2.1.1.2 $ $Date: 91/11/15 13:35:29 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include <stdio.h>

#include "mbus.h"
#include "api.h"

/* ------------------------------------------------------------------------- */
int
MBconsp(e) struct mb_object *e;
{
  return NULL != e && MB_CONS == e->type;
}

int
MBnamep(e) struct mb_object *e;
{
  return NULL != e && MB_NAME == e->type;
}

int
MBstringp(e) struct mb_object *e;
{
  return NULL != e && MB_STRING == e->type;
}

int
MBrawp(e) struct mb_object *e;
{
  return NULL != e && MB_RAW == e->type;
}

int
MBchunkp(e) struct mb_object *e;
{
  return MB_CHUNKP(e);
}
/* ------------------------------------------------------------------------- */
struct mb_object *
MBcar(e) struct mb_object *e;
{
  return NULL != e && MB_CONS == e->type ? e->object.cons.car : NULL;
}

struct mb_object *
MBcdr(e) struct mb_object *e;
{
  return NULL != e && MB_CONS == e->type ? e->object.cons.cdr : NULL;
}

struct mb_object *
MBnth(e,n)
     struct mb_object *e;
     int n;
{
  if (n < 0 || NULL == e) return NULL;
  while (n-- > 0 && NULL != e) e = MB_CDR(e);
  return MB_CONSP(e) ? MB_CAR(e) : e;
}

struct mb_object *
MBnthcdr(e,n)
     struct mb_object *e;
     int n;
{
  if (n < 0 || NULL == e) return NULL;
  while (n-- > 0 && NULL != e) e = MB_CDR(e);
  return e;
}

struct mb_object *
MBsetcar(e,new) struct mb_object *e,*new;
{
  if (NULL != e && e->type == MB_CONS)
    e->object.cons.car = new;
  return new;
}

struct mb_object *
MBsetcdr(e,new) struct mb_object *e,*new;
{
  if (NULL != e && e->type == MB_CONS)
    e->object.cons.cdr = new;
  return new;
}
/* ------------------------------------------------------------------------- */
int
MBemptyp(e) t_sexp e;
{
  return MB_CHUNKP(e) ? MBisChunkEmpty(&(e->object.chunk)) : 1;
}

void
MBempty(e) t_sexp e;
{
  if (MB_CHUNKP(e)) MBChunkEmpty(&(e->object.chunk));
}
/* ------------------------------------------------------------------------- */
/* Determine the length of an expression */
int
MBlength(e) struct mb_object *e;
{
  int length = 0;

  if (NULL != e)
    {
      switch (e->type)
	{
	case MB_NAME:
	case MB_STRING:
	case MB_RAW:
	  length = e->object.chunk.count;
	  break;
	case MB_CONS:
	  while (NULL != e)
	    {
	      length += 1;
	      e = MB_CDR(e);
	    }
	  break;
	}
    }
  return length;
}
      
/* ------------------------------------------------------------------------- */
int
MBequal(s1,s2) struct mb_object *s1,*s2;
{
  int equal = 0;
  /* same pointer, must be equal */
  if (s1 == s2) equal = 1;
  /* pointers not the same, if either is NULL then not equal */
  else if (NULL == s1 || NULL == s2) equal = 0;
  /* so, they are not the same pointer, and both are non-NULL */
  /* if not the same type, do special checks, default not equal */
  else if (s1->type != s2->type)
    {
      if (MB_REGEXP == s1->type)
	equal = MBRegexpCompare(&(s1->object.re), s2);
      else if (MB_REGEXP == s2->type)
	equal = MBRegexpCompare(&(s2->object.re), s1);
    }
  else
    switch (s1->type)			/* either, since the same */
      {
      case MB_CONS:
	equal = MBequal(MBcar(s1),MBcar(s2)) && MBequal(MBcdr(s1),MBcdr(s2));
	break;
      case MB_STRING:
      case MB_NAME:
      case MB_RAW:
	equal = MBChunkEqual(&(s1->object.chunk),&(s2->object.chunk));
	break;
      case MB_CHUNK_POINTER:
	equal = MBCPEqual(&(s1->object.cp),&(s2->object.cp));
	break;
      case MB_REGEXP:
	equal = !strcmp(s1->object.re.program->data,
			s2->object.re.program->data);
	break;
      }
  return equal;
}

/* ------------------------------------------------------------------------- */
void
MBfree(s) struct mb_object *s;
{
  if (NULL == s) return;
  else if (s->check_word != MB_CHECK_WORD_VALUE)
    {
      fprintf(stderr,
	      "MBus Library: Attempt to MBfree non-MB object at %x\n", s);
      return;
    }
  else if (MB_FREEP(s))
    {
      if (MBLogLevel > 1) fprintf(stderr,"Freed element in sexp free\n");
      return;
    }

  /* free the actual object */
  MBFreeObject(s);

  /* free subsidiary data. The freeing of an object doesn't affect any of
   * the non-generic data.
   */
  switch (s->type)
    {
    case MB_CONS:
      MBfree(s->object.cons.car);
      MBfree(s->object.cons.cdr);
      break;
    case MB_STRING:
    case MB_NAME:
    case MB_RAW:
      MBChunkEmpty(&(s->object.chunk));
      break;
    case MB_CHUNK_POINTER:
      /* nothing */
      break;
    case MB_REGEXP:
      if (NULL != s->object.re.program) MBFreeBlock(s->object.re.program);
      break;
    default:
      if (MBLogLevel)
	fprintf(stderr,"Bad type during free sexp (%d)\n",s->type);
      break;
    }
}
/* ------------------------------------------------------------------------- */
void
MBmark(sexp) t_sexp sexp;
{
  if (NULL == sexp || sexp->flags & MB_MARKED) return;

  sexp->flags |= MB_MARKED;

  switch (sexp->type)
    {
    case MB_NAME:
    case MB_STRING:
    case MB_RAW:
      MBChunkMark(&(sexp->object.chunk));
      break;
    case MB_CONS:
      MBmark(sexp->object.cons.car);
      MBmark(sexp->object.cons.cdr);
      break;
    case MB_REGEXP :
      sexp->object.re.program->flags |= MB_MARKED;
      break;
    }
}
/* ------------------------------------------------------------------------- */
t_sexp
MBduplicate(sexp) t_sexp sexp;
{
  t_sexp new = NULL;

  if (NULL == sexp) return NULL;

  switch (sexp->type)
    {
    case MB_CONS:
      new = MBGetCons();
      new->object.cons.car = MBduplicate(sexp->object.cons.car);
      new->object.cons.cdr = MBduplicate(sexp->object.cons.cdr);
      break;
    case MB_NAME:
    case MB_STRING:
    case MB_RAW:
      new = MBGetName();
      MBChunkPutChunk(&(new->object.chunk), &(sexp->object.chunk));
      new->type = sexp->type;		/* preserve the type */
      break;
    case MB_REGEXP:
      /* this needs to be fixed someday */
      break;
    }
  return new;
}
/* ------------------------------------------------------------------------- */
static void
MBPrintChunk(obj,dst)
     struct mb_object *obj;
     struct mb_chunk *dst;
{
  int c;
  struct mb_chunk_pointer cp;

  MBCPSet(&cp, obj);

  while ( (c = MBCPGetChar(&cp)) != CHUNK_EMPTY)
    switch (c)
      {
      case '"':
      case '(':
      case ')':
      case '\\':
	MBChunkPutChar(dst,'\\');
	/* fall through */
      default:
	MBChunkPutChar(dst,c);
      }
}
/* ------------------------------------------------------------------------- */
static void
MBPrintString(obj,dst)
     struct mb_object *obj;
     struct mb_chunk *dst;
{
  int c;
  struct mb_chunk_pointer cp;

  MBCPSet(&cp, obj);

  MBChunkPutChar(dst,'"');
  while ( (c = MBCPGetChar(&cp)) != CHUNK_EMPTY)
    switch (c)
      {
      case '"':
      case '\\':
	MBChunkPutChar(dst,'\\');
      default:
	MBChunkPutChar(dst,c);
      }
  MBChunkPutChar(dst,'"');
}
/* ------------------------------------------------------------------------- */
static void
MBPrintRaw(obj, dst)
     struct mb_object *obj;
     struct mb_chunk *dst;
{
  char buff[64];

  MBChunkPutChar(dst, MB_MACRO_CHAR);
  sprintf(buff, "%d", obj->object.chunk.count);
  MBChunkPutBuffer(dst, buff, strlen(buff));
  MBChunkPutChar(dst, MB_MACRO_RAW_CHAR);
  MBChunkPutChunk(dst, &(obj->object.chunk));
}
/* ------------------------------------------------------------------------- */
static void
MBPrintList(obj,dst)
     struct mb_object *obj;
     struct mb_chunk *dst;
{
#ifdef __STDC__
  static void MBPrintSexp(t_sexp, struct mb_chunk *);
#else
  static void MBPrintSexp();
#endif

  while (NULL != obj)
    {
      if (MB_CONS == obj->type)
	{
	  MBPrintSexp(obj->object.cons.car,dst);
	  obj = obj->object.cons.cdr;
	  if (NULL != obj) MBChunkPutChar(dst,' ');
	}
      else
	{
	  MBChunkPutChar(dst,' ');
	  MBChunkPutChar(dst,'.');
	  MBChunkPutChar(dst,' ');
	  MBPrintSexp(obj,dst);
	  obj = NULL;
	}
    }
}
/* ------------------------------------------------------------------------- */
static void
MBPrintSexp(e,dst)
     struct mb_object *e;
     struct mb_chunk *dst;
{
  if (NULL == e)
    MBChunkPutBuffer(dst,"()",2);
  else
    switch (e->type)
      {
      case MB_CONS:
	MBChunkPutChar(dst,'(');
	MBPrintList(e,dst);
	MBChunkPutChar(dst,')');
	break;
      case MB_NAME:
	MBPrintChunk(e,dst);
	break;
      case MB_STRING:
	MBPrintString(e,dst);
	break;
      case MB_RAW:
	MBPrintRaw(e, dst);
	break;
      }
}
/* ------------------------------------------------------------------------- */
t_sexp
MBprint(e, dst)
     t_sexp e,dst;
{
  if (NULL == dst) dst = MBGetName();

  if (MB_CHUNKP(dst)) MBPrintSexp(e, &(dst->object.chunk));
  return dst;
}
/* ------------------------------------------------------------------------- */
char *
MBCstring(obj)
     struct mb_object *obj;
{
  char *result = NULL;

  if (MB_CHUNKP(obj))
    {
      result = MBmalloc_func(obj->object.chunk.count + 1);
      if (NULL != result)
	result[MBChunkCopyBuffer(&(obj->object.chunk), result,
				 obj->object.chunk.count)] = 0;
    }
  return result;
}
  
/* ------------------------------------------------------------------------- */
char *
MBprint_Cstring(sexp)
     t_sexp sexp;
{
  char *s;
  t_sexp tmp = MBprint(sexp, NULL);

  s = MBCstring(tmp);
  MBfree(tmp);
  return s;
}
/* ------------------------------------------------------------------------- */
char *
MBCsubstring(obj, first, last)
     t_sexp obj;
     int first, last;
{
  char *result = NULL;
  int length;
  int count;

  if (MB_CHUNKP(obj))
    {
      length = obj->object.chunk.count;
      /* adjust for negative values, which mean from end */
      if (first < 0) first += length;
      if (last < 0) last += length;
      /* Now verify the values */
      if (first >= last || first < 0 || last > length)
	result = NULL;
      else
	{
	  count = last - first;		/* length of substring */
	  result = MBmalloc_func(count + 1); /* include terminating null */
	  if (NULL != result)
	    result[MBChunkCopySubbuffer(&(obj->object.chunk),
					result, count, first)] = 0;
	}
    }
  return result;
}
/* ------------------------------------------------------------------------- */
struct mb_object *
MBput_Cstring(obj,s)
     struct mb_object *obj;
     char *s;
{
  if (NULL == obj) obj = MBGetString();
  if (MB_CHUNKP(obj) && s)
      MBChunkPutBuffer(&(obj->object.chunk), s, strlen(s));
  return obj;
}
/* ------------------------------------------------------------------------- */
struct mb_object *
MBput_buffer(obj, buff, count)
     struct mb_object *obj;
     char *buff;
     int count;
{
  if (MB_CHUNKP(obj)) MBChunkPutBuffer(&(obj->object.chunk), buff, count);
  return obj;
}
/* ------------------------------------------------------------------------- */
/* Copy the contents of the chunk into the buffer */
int
MBbuffer(obj,s,n)
     struct mb_object *obj;
     char *s;
     int n;
{
  if (NULL != s && n > 0) *s = 0;	/* guarantee terminated string */

  return MB_CHUNKP(obj) ? MBChunkCopyBuffer(&(obj->object.chunk), s, n) : 0;
}
/* ------------------------------------------------------------------------- */
struct mb_object *
MBput(dst,src) struct mb_object *dst,*src;
{
  if (MB_CHUNKP(dst) && MB_CHUNKP(src))
    MBChunkPutChunk(&(dst->object.chunk), &(src->object.chunk));
  return dst;
}
/* ------------------------------------------------------------------------- */
struct mb_object *
#ifdef __STDC__
MBput_char(struct mb_object *obj, char c)
#else
MBput_char(obj,c)
     struct mb_object *obj;
     char c;
#endif
{
  if (MB_CHUNKP(obj)) MBChunkPutBuffer(&(obj->object.chunk), &c, 1);
  return obj;
}
/* ------------------------------------------------------------------------- */
int
MBcompare_Cstring(obj,s)
     struct mb_object *obj;
     char *s;
{
  int n;

  if (NULL == s || !MB_CHUNKP(obj)) return 1;

  n = strlen(s);
  if (n > obj->object.chunk.count) return 1;
  else if (n < obj->object.chunk.count) return -1;

  return MBChunkBufferCompare(&(obj->object.chunk), s, n);
}
/* ------------------------------------------------------------------------- */
int
MBpeek_char(obj) struct mb_object *obj;
{
  return MB_CHUNKP(obj) ? MBChunkPeekChar(&(obj->object.chunk)) : CHUNK_EMPTY;
}
/* ------------------------------------------------------------------------- */
int
MBwrite(fd, obj)
     int fd;
     struct mb_object *obj;
{
  return MB_CHUNKP(obj) ? MBChunkWrite(fd, &(obj->object.chunk)) : 0;
}
/* ------------------------------------------------------------------------- */
int
MBint(e, count)
     t_sexp e;				/* expression to convert */
     int *count;			/* place to put # chars processed */
{
  int neg = 0;
  int c;
  int result = 0;
  int n = 0;
  struct mb_chunk_pointer cp;

  if (NULL != count) *count = 0;
  if (!MB_CHUNKP(e)) return 0;
  MBCPSet(&cp, &(e->object.chunk));

  if ('-' == MBCPPeek(&cp))
    {
      neg = 1;
      n += 1;
      MBCPGetChar(&cp);
    }

  for (c = MBCPGetChar(&cp) ; '0' <= c && c <= '9' ; c = MBCPGetChar(&cp))
    {
      result = result * 10 + c - '0';
      n += 1;
    }

  if (neg && 1 == n) n = 0;		/* negative sign, but no digits */

  if (NULL != count) *count = n;
  return neg ? -result : result;
}
/* ------------------------------------------------------------------------- */
