/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*	chunk.c: Chunk manipulation functions */
/* Chunk (data block strings) support */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/lib/RCS/chunk.c,v $ */

static char rcsid[] = "chunk.c $Revision: 2.1.1.2 $ $Date: 91/11/15 13:35:23 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include <stdio.h>

#include "mbus.h"
#include "api.h"

/* ------------------------------------------------------------------------- */
#define FIX_CHUNK_TAIL(chunk)\
{\
  struct mb_chunk *c = (chunk);\
  if (c->tindex >= c->tail->count)\
    {\
      struct mb_block_struct *b = c->tail;\
      c->tindex = 0;\
      c->tail = c->tail->next;\
      MBFreeBlock(b);\
      if (NULL == c->tail)\
	{\
	  if (c->head != b && MBLogLevel > 0)\
	    fprintf(stderr, "Mangled Chunk - incoherent count / head\n");\
	  c->head = NULL;\
	  c->count = 0;\
	  c->hindex = 0;\
	}\
    }\
}
      
/* ------------------------------------------------------------------------- */
int
MBisChunkEmpty(s) struct mb_chunk *s;
{
  t_mb_block b = s->tail;

  return NULL == b || (b == s->head && s->tindex >= b->count);
}
/* ------------------------------------------------------------------------- */
struct mb_chunk *
MBChunkReset(s) struct mb_chunk *s;
{
  s->hindex = s->tindex = s->count = 0;
  s->tail = s->head = NULL;
  return s;
}

/* ------------------------------------------------------------------------- */
void
MBChunkEmpty(s) struct mb_chunk *s;
{
  t_mb_block b = s->tail;

  while (NULL != b)
    {
      t_mb_block next = b->next;
      MBFreeBlock(b);
      b = next;
    }
  MBChunkReset(s);
}
/* ------------------------------------------------------------------------- */
/* Return the set of contiguous characters in the first block. This can be
 * used to be more efficient about moving groups of characters around.
 */
t_block_char *
MBChunkFindSpan(chunk, count)
     struct mb_chunk *chunk;
     int *count;
{
  int n = 0;				/* store count in here */
  t_block_char *spot = NULL;		/* addr goes in here */

  if (!MBisChunkEmpty(chunk))
    {
      spot = chunk->tail->data + chunk->tindex;
      n = chunk->tail->count - chunk->tindex;
    }

  if (NULL != count) *count = n;
  return spot;
}
/* ------------------------------------------------------------------------- */
/* returns the number of characters that can be stored */
int
MBPrepareChunkForInput(s) struct mb_chunk *s;
{
  int n;				/* the amount we can read */
  t_mb_block b = s->head;	/* place to put the data */

  /* what are the cases?
   * . There are no data blocks
   * . There is space in the current block
   * . the full block is not the maximum size
   * . the full block is maximum size
   */

  if (NULL == b)			/* no data blocks */
    {
      s->head = s->tail = MBGetBlock(0); /* get smallest size */
      s->count = s->tindex = s->hindex = 0;
      n = s->head->size;
    }
  else if (s->hindex < b->size)		/* space left in block */
    n = b->size - s->hindex;
#if 0
  else if (! MBisLargestBlock(b) && s->tail == s->head)
    {
      t_mb_block new = MBUpgradeBlock(b); /* try for a bigger one */
      if (NULL == new)			/* blow off, try again next time */
	{
	  if (MBLogLevel)
	    fprintf(stderr,"PrepareChunkForInput: block upgrade failed\n");
	  return 0;
	}
      s->tail = s->head = b = new;
      /* assume that block sizes are monotonically increasing */
      n = b->size - s->hindex;
    }
#endif
  else					/* have to add another block */
    {
      t_mb_block new;
      new = MBGetBlock(b->size + !MBisLargestBlock(b));

      if (NULL == new)
	{
	  if (MBLogLevel)
	    fprintf(stderr,
		    "PrepareChunkForInput: unable to extend block list\n");
	  return 0;
	}
      s->head->next = new;
      s->head = new;
      s->hindex = 0;
      n = s->head->size;
    }
  return n;
}
/* ------------------------------------------------------------------------- */
int
MBChunkGetChar(s) struct mb_chunk *s;
{
  int c;
  t_mb_block b = s->tail;

  /* this verifies that (tindex < hindex OR head != tail) */
  if (MBisChunkEmpty(s)) return CHUNK_EMPTY;

  c = b->data[s->tindex++];
  s->count -= 1;
  if (s->tindex >= b->size)		/* ran out of block, move on */
    {
      t_mb_block next = b->next;

      /* if only 1 block, update head too */
      if (s->tail == s->head) s->tail = s->head = NULL;
      else s->tail = next;
      s->tindex = 0;
      MBFreeBlock(b);
    }
  return c;
}
/* ------------------------------------------------------------------------- */
int
#ifdef __STDC__
MBChunkPutChar(struct mb_chunk *s, char c)
#else
MBChunkPutChar(s,c)
     struct mb_chunk *s;
     char c;
#endif
{
  int n = MBPrepareChunkForInput(s);
  t_mb_block b = s->head;

  if (n > 0)
    {
      b->data[s->hindex++] = c;
      s->count += 1;
      b->count += 1;
    }
  else if (MBLogLevel) fprintf(stderr,"ChunkPutChar: dropped character\n");
  return n > 0;
}
/* ------------------------------------------------------------------------- */
int MBChunkPeekChar(s) struct mb_chunk *s;
{
  int c;
  t_mb_block b = s->tail;

  return MBisChunkEmpty(s) ? CHUNK_EMPTY : b->data[s->tindex];
}
/* ------------------------------------------------------------------------- */
int
MBChunkRead(fd,s)
     int fd;				/* file descriptor */
     struct mb_chunk *s;			/* place to put it */
{
  int n;			/* the amount we can read */
  int count = 0;			/* actual amount read */
  char buff[4096];			/* read buffer size */

  count = read(fd, buff, 4096);

  if (count > 0) MBChunkPutBuffer(s, buff, count);
  else if ((count < 0) && MBLogLevel) perror("MBChunkRead:");

  return count;
}

/* ------------------------------------------------------------------------- */
/* This will write out parts of the chunk until the chunk is empty or a write
 * fails. For blocking fd's, this should mean that the entire chunk will be
 * written (except for errors).
 */
int
MBChunkWrite(fd,chunk)
     int fd;				/* file descriptor */
     struct mb_chunk *chunk;			/* the Chunk */
{
  t_mb_block b = chunk->tail;
  int n = 0;				/* number of byte available */
  int count = 0;			/* number written in last write*/
  int total = 0;			/* total bytes written */

  if (NULL == b) return 0;		/* check for empty chunk */

  /* Loop, writing to the fd until either the chunk runs out or
   * a write doesn't fully succeed.
   */
  do
    {
      n = b->count - chunk->tindex;	/* # of characters in the block */
      if (n > 0)
	{
	  count = write(fd, (char *)&(b->data[chunk->tindex]), n);
	  if (count > 0)
	    {
	      chunk->count -= count;
	      chunk->tindex += count;
	      total += count;		/* track total successfully sent */
	      /* if this block is used up, discard it */
	      if (chunk->tindex >= b->size)
		{
		  chunk->tail = chunk->tail->next;
		  chunk->tindex = 0;
		  MBFreeBlock(b);
		  if (NULL == chunk->tail)
		    {
		      if ((chunk->head != b) && MBLogLevel)
			fprintf(stderr,"Mangled Chunk\n");
		      chunk->head = NULL;
		      chunk->hindex = 0;
		    }
		  b = chunk->tail;	/* update current block */
		}
	    }
	}
      else if (n < 0)
	{
	  if (MBLogLevel)
	    fprintf(stderr,"Excess count in block (count=%d,tail=%d)\n",
		    b->count,chunk->tail);
	  chunk->tindex = b->count;
	}
    } while (count >= n && n > 0 && NULL != b);
  return total;
}
/* ------------------------------------------------------------------------- */
struct mb_chunk *
MBChunkPutBuffer(dst,src,n)
     struct mb_chunk *dst;
     char *src;
     int n;				/* number of characters from src */
{
  while (n > 0)
    {
      int count = MBPrepareChunkForInput(dst);
      
      if (count < 1) n = 0;
      else
	{
	  count = count > n ? n : count; /* clip by remaining input */
	  memcpy((char *)&(dst->head->data[dst->hindex]), src, count);
	  dst->hindex += count;
	  dst->count += count;
	  dst->head->count += count;
	  src += count;
	  n -= count;
	}
    }
  return dst;
}
/* ------------------------------------------------------------------------- */
struct mb_chunk *
MBChunkPutChunk(dst,src)
     struct mb_chunk *dst,*src;
{
  t_mb_block b = src->tail;
  int start = src->tindex;

  while (NULL != b)
    {
      MBChunkPutBuffer(dst, (char *)&(b->data[start]), b->count - start);
      start = 0;
      b = b->next;
    }
  return dst;
}
/* ------------------------------------------------------------------------- */
/* move up count characters from the source to the destination and return
 * the number of characters moved
 */
int
MBChunkMoveN(dst, src, count)
     struct mb_chunk *dst,*src;
     int count;
{
  t_mb_block b;
  int size, total = 0;

  while (count > 0 && NULL != (b = src->tail))
    {
      size = b->count - src->tindex;
      if (size > count) size = count;
      MBChunkPutBuffer(dst, (char *)&(b->data[src->tindex]), size);
      total += size;
      count -= size;
      src->tindex += size;
      src->count -= size;
      FIX_CHUNK_TAIL(src);
    }
  return total;
}
/* ------------------------------------------------------------------------- */
int
MBChunkEqual(s,p) struct mb_chunk *s,*p;
{
  t_mb_block pb, sb;
  int si,pi;
  unsigned int n = s->count;		/* doesn't matter which one */

  if (s->count != p->count) return 0;

  sb = s->tail; si = s->tindex;
  pb = p->tail; pi = p->tindex;

  while (n--)
    {
      if (sb == NULL || pb == NULL || sb->data[si++] != pb->data[pi++])
	return 0;

      if (si >= sb->count)
	{
	  sb = sb->next;
	  si = 0;
	}
      if (pi >= pb->count)
	{
	  pb = pb->next;
	  pi = 0;
	}
    }
  return 1;
}
/* ------------------------------------------------------------------------- */
int MBChunkCopyBuffer(chunk, buff, n)
     struct mb_chunk *chunk;
     char *buff;
     int n;
{
  return MBChunkCopySubbuffer(chunk, buff, n, 0);
}
/* ------------------------------------------------------------------------- */
/* This does _not_ null terminate the result. */
int
MBChunkCopySubbuffer(chunk, buff, n, offset)
     struct mb_chunk *chunk;		/* the Chunk */
     char *buff;			/* the buffer to put it in */
     int n;				/* size of the buffer */
     int offset;			/* how far into the chunk to start */
{
  t_mb_block sb = chunk->tail;
  t_block_int si = chunk->tindex;
  t_block_int sn = chunk->count;
  int size;
  int count = 0;

  while (sn && n && NULL != sb)
    {
      size = sb->count - si;
      if (offset > 0)
	{
	  if (size > offset) size = offset;
	  offset -= size;
	}
      else				/* no offset */
	{
	  if (size > n) size = n;
	  memcpy(buff, &(sb->data[si]), size);
	  buff += size;
	  n -= size;
	  count += size;
	}
      si += size;
      sn -= size;
      if (si >= sb->count)
	{
	  si = 0;
	  sb = sb->next;
	}
    }

  return count;
}
/* ------------------------------------------------------------------------- */
int
MBChunkDropChars(chunk, n)
     struct mb_chunk *chunk;
     int n;
{
  int size;
  t_mb_block sb;
  t_block_int si = chunk->tindex;

  while (n > 0 && chunk->count && NULL != (sb = chunk->tail))
    {
      size = sb->count - chunk->tindex;
      if (size > n) size = n;
      chunk->count -= size;
      chunk->tindex += size;
      n -= size;
      FIX_CHUNK_TAIL(chunk);
    }
}
/* ------------------------------------------------------------------------- */
#define MBCP_ADJUST(cp)							\
  {									\
    if ((cp)->index >= (cp)->block->count)				\
      {									\
        (cp)->index = 0;						\
        (cp)->block = (cp)->block->next;				\
      }									\
    if (NULL != (cp)->block)						\
      {									\
        (cp)->c = (cp)->block->data[(cp)->index];			\
      }									\
    else								\
      {									\
        (cp)->c = CHUNK_EMPTY;						\
      }									\
  }
/* ------------------------------------------------------------------------- */
void
MBCPCopy(dst,src) struct mb_chunk_pointer *dst,*src;
{
  memcpy((char *)dst, (char *)src, sizeof(struct mb_chunk_pointer));
}
/* ------------------------------------------------------------------------- */
int
MBCPLength(cp) struct mb_chunk_pointer *cp;
{
  int count = 0;
  int index;
  t_mb_block b;

  if (NULL == cp) return 0;
  b = cp->block;
  index = cp->index;
  while (NULL != b)
    {
      count += b->count - index;
      index = 0;
      b = b->next;
    }
  return count;
}
/* ------------------------------------------------------------------------- */
void
MBCPEmpty(cp) struct mb_chunk_pointer *cp;
{
  if (NULL != cp)
    {
      cp->c = CHUNK_EMPTY;
      cp->block = NULL;
      cp->index = 0;
    }
}
/* ------------------------------------------------------------------------- */
int
MBCPEqual(cp1,cp2) struct mb_chunk_pointer *cp1,*cp2;
{
  if (cp1 == cp2) return 1;
  else if (NULL == cp1 || NULL == cp2) return 0;
  else return cp1->block == cp2->block && cp1->index == cp2->index;
}
/* ------------------------------------------------------------------------- */
void
MBCPSet(cp,obj)
     struct mb_chunk_pointer *cp;
     struct mb_object *obj;
{
  if (NULL == cp) return;

  if (NULL == obj)
    {
      cp->block = NULL;
      cp->index = 0;
      cp->c = CHUNK_EMPTY;
    }
  else switch (obj->type)
    {
    case MB_NAME:
    case MB_STRING:
      cp->block = obj->object.chunk.tail;
      cp->index = obj->object.chunk.tindex;
      cp->c = (NULL == cp->block ? CHUNK_EMPTY :  cp->block->data[cp->index]);
      break;
    case MB_CHUNK_POINTER:
      cp->block = obj->object.cp.block;
      cp->index = obj->object.cp.index;
      cp->c = obj->object.cp.c;
      break;
    default:
      cp->block = NULL;
      cp->index = 0;
      cp->c = CHUNK_EMPTY;
      break;
    }
}
/* ------------------------------------------------------------------------- */
/* return the current character and advance to the next one */
int
MBCPGetChar(cp) struct mb_chunk_pointer *cp;
{
  int cc;

  if (NULL == cp || NULL == cp->block)
    cc = CHUNK_EMPTY;
  else
    {
      cc = cp->c;
      /* move to the next character */
      cp->index += 1;
      MBCP_ADJUST(cp);
    }
  return cc;
}
/* ------------------------------------------------------------------------- */
int
MBCPSkipChars(cp,n)
     struct mb_chunk_pointer *cp;
     int n;				/* number of chars to skip */
{
  if (NULL == cp || NULL == cp->block) return CHUNK_EMPTY;
  if (n < 1) return cp->c;

  while (n > 0 && NULL != cp->block)
    {
      int count = cp->block->count - cp->index;
      
      if (count > n) count = n;		/* limit distance */
      cp->index += count;
      n -= count;
      MBCP_ADJUST(cp);			/* fix up block overflow */
    }
  return cp->c;
}
/* ------------------------------------------------------------------------- */
int
MBCPAdvanceToChar(cp,c)
     struct mb_chunk_pointer *cp;
     int c;
{
  if (NULL == cp || CHUNK_EMPTY == c) return 0;

  while (NULL != cp->block)
    {
      extern char *memchr();
      int count = cp->block->count - cp->index;
      char *spot = memchr(cp->block->data + cp->index, c, count);

      if (NULL != spot)
	{
	  cp->index = spot - (char *)cp->block->data;
	  cp->c = cp->block->data[cp->index];
	  return 1;
	}
      cp->index = 0;
      cp->block = cp->block->next;
    }
  return 0;
}
/* ------------------------------------------------------------------------- */
int
MBCPFindChar(src,dst,c)
     struct mb_chunk_pointer *src,*dst;
     int c;
{
  if (NULL == src || NULL == dst || CHUNK_EMPTY == c) return 0;

  memcpy((char *)dst, (char *)src, sizeof(struct mb_chunk_pointer));
  return MBCPAdvanceToChar(dst,c);
}

/* ------------------------------------------------------------------------- */
int
MBCPPeek(cp)
     struct mb_chunk_pointer *cp;
{
  int cc;

  if (NULL == cp || cp->block == NULL) cc = CHUNK_EMPTY;
  else if (cp->index < cp->block->count - 1) cc = cp->block->data[cp->index+1];
  else if (NULL == cp->block->next) cc = CHUNK_EMPTY;
  else cc = cp->block->next->data[0];

  return cc;
}
/* ------------------------------------------------------------------------- */
/* Peek n chars ahead */
int
MBCPPeekN(cp,n)
     struct mb_chunk_pointer *cp;
     int n;
{
  int cc = CHUNK_EMPTY;

  if (NULL == cp) cc = CHUNK_EMPTY;
  else if (0 >= n) cc = cp->c;
  else
    {
      t_mb_block b = cp->block;
      t_block_int i = cp->index;
      int count;

      while (n > 0 && NULL != b)
	{
	  count = b->count - i;
	  if (count > n) count = n;
	  i += count;
	  n -= count;
	  if (i > b->count)
	    {
	      i = 0;
	      b = b->next;
	    }
	}
      if (NULL != b) cc = b->data[i];
    }
  return cc;
}
/* ------------------------------------------------------------------------- */
int
MBCPBufferCompare(cp,s,n)
     struct mb_chunk_pointer *cp;
     char *s;
     int n;
{
  t_mb_block b;
  int index;
  int ret = 0;

  if (NULL == cp || NULL == s || 1 > n) return 0;

  index = cp->index;
  b = cp->block;

  while (n > 0 && NULL != b)
    {
      int count = b->count - index;

      if (count > n) count = n;
      if (ret = strncmp(s,b->data + index,count)) return ret;
      n -= count;
      b = b->next;
      index = 0;
    }
  return 0;
}
/* ------------------------------------------------------------------------- */
int
MBChunkBufferCompare(c,s,n)
     struct mb_chunk *c;
     char *s;
     int n;
{
  t_mb_block b;
  int index;
  int ret = 0;

  if (NULL == c || NULL == s) return 1;
  if (c->count < n) return 1;		/* not enough chars in chunk */

  b = c->tail;
  index = c->tindex;

  while (n > 0 && NULL != b)
    {
      int count = b->count - index;

      if (count > n) count = n;
      if (ret = strncmp(s,b->data + index,count)) return ret;
      n -= count;
      b = b->next;
      index = 0;
    }
  return 0;
}
/* ------------------------------------------------------------------------- */
void
MBChunkMark(c) struct mb_chunk *c;
{
  t_mb_block b;

  if (NULL == c) return;

  for ( b = c->tail ; NULL != b ; b = b->next )
    b->flags |= MB_MARKED;
}
/* ------------------------------------------------------------------------- */
