/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/* Run a program and connect it to the message bus. */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/commands/RCS/client.c,v $ */

static char rcsid[] = "$Revision: 2.1.1.1 $ $Date: 91/11/15 13:37:55 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include "header.h"

#include <netdb.h>
#include <sys/signal.h>
#include "getopt.h"

#if POLL
#include <poll.h>
#include <stropts.h>
#endif

#ifdef NeXT
#include <sys/fcntl.h>
#endif
/* ------------------------------------------------------------------------- */
int service_pid;
int service_in, service_out;
char *service_command;
char **service_args;

int trace_fd = -1;

static char enable_messages [] = { "(accept \".*\" \".\")" };

/* ------------------------------------------------------------------------- */
void
bomb(s) char *s;
{ perror(s); exit(1); }
/* ------------------------------------------------------------------------- */
int
ConnectSocket(host,port)
     char *host;
     int port;
{
  int sock;
  int flags;
  struct sockaddr_in saddr;
  struct hostent *h;

  sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (sock < 0) bomb("Cannot open socket");

  h = gethostbyname(host);

  if (NULL == h) bomb("Cannot find host");
  if (h->h_addr_list == 0) bomb("No addresses for host");

  saddr.sin_family = AF_INET;
  saddr.sin_addr.s_addr = INADDR_ANY;
  saddr.sin_port = 0;			/* anywhere */

  saddr.sin_family = AF_INET;
  memcpy((char *) &saddr.sin_addr, h->h_addr_list[0], h->h_length);
  saddr.sin_port = htons(port);

  if (connect(sock, (struct sockaddr *) &saddr, sizeof(saddr)))
    bomb("Couldn't connect to MBus");

  /* enable all messages */
  write(sock, enable_messages, sizeof(enable_messages)-1);
  return sock;
}
/* ------------------------------------------------------------------------- */
void
SignalHandler(sig)
     int sig;
{
  switch (sig)
    {
    case SIGCLD:
      fprintf(stderr,"Child process finished\n");
      exit(1);
      break;
    default:
      if (service_pid) kill(service_pid,sig);
      else exit(1);
      break;
    }
  return;
}
/* ------------------------------------------------------------------------- */
int StartService()
{
  int input[2];
  int output[2];
  FILE *infile, *outfile;

#if SPIPE
  if (spipe(input) < 0) bomb("Pipe");
  if (spipe(output) < 0) bomb("Pipe");
#else
  if (pipe(input) < 0) bomb("Pipe");
  if (pipe(output) < 0) bomb("Pipe");
#endif

  signal(SIGCLD, SignalHandler);
  signal(SIGHUP, SignalHandler);
  signal(SIGTERM, SignalHandler);

  /* we now have the pipes that we need. Let's fork off the service. */
  if (!(service_pid = fork()))
    {
      /* child process */
      close(0); close(1);		/* stdin/stdout */
      dup(input[0]);			/* read end of input */
      dup(output[1]);			/* write end of output */
      close(input[1]); close(input[0]);
      close(output[0]); close(output[1]);
      execvp(service_command,service_args);
      bomb("exec");
    }
  else if (service_pid < 0) bomb("Fork");

  service_out = input[1]; close(input[0]);
  service_in = output[0]; close(output[1]);
}
/* ------------------------------------------------------------------------- */
#define BUFFSIZE 32768
#define MAX_WRITE_SIZE 1024

#define SET_NDELAY(f)  { int ff; fcntl(f, F_GETFL, &ff); ff |= O_NDELAY; fcntl(f, F_SETFL, ff); }

#if POLL
#define SOCKET_READ_OK	(p_fd[2].revents & POLLIN)
#define SOCKET_WRITE_OK (p_fd[2].revents & POLLOUT)
#define CLIENT_READ_OK  (p_fd[0].revents & POLLIN)
#define CLIENT_WRITE_OK (p_fd[1].revents & POLLOUT)
#else
#define SOCKET_READ_OK  FD_ISSET(s, &read_fds)
#define SOCKET_WRITE_OK FD_ISSET(s, &write_fds)
#define CLIENT_READ_OK  FD_ISSET(service_in, &read_fds)
#define CLIENT_WRITE_OK FD_ISSET(service_out, &write_fds)
#endif

void
Run(s) int s;
{
#if POLL
  struct pollfd p_fd[3];
#else
  fd_set read_fds, write_fds;
#endif

  int ff;				/* file flags var */
  int n;				/* n of ready fd's */
  int count;				/* result of read/write */
  struct
    {
      int head,tail;
      int count;
      char buff[BUFFSIZE];
    } client, socket;

#if POLL
  p_fd[0].fd = service_in;
  p_fd[1].fd = service_out;
  p_fd[2].fd = s;
#endif

  client.head = client.tail = socket.head = socket.tail = 0;
  client.count = socket.count = 0;

  SET_NDELAY(s);
  SET_NDELAY(service_in);
  SET_NDELAY(service_out);

  while (1)
    {
#if POLL
      p_fd[2].events = (client.count < BUFFSIZE ? POLLIN : 0)
	| (socket.count ? POLLOUT : 0);
      p_fd[0].events = socket.count < BUFFSIZE ? POLLIN : 0;
      p_fd[1].events = client.count ? POLLOUT : 0;
      n = poll(p_fd, 3, -1);
#else
      FD_ZERO(&read_fds); FD_ZERO(&write_fds);
      if (socket.count < BUFFSIZE) FD_SET(service_in, &read_fds);
      if (client.count) FD_SET(service_out, &write_fds);
      if (client.count < BUFFSIZE) FD_SET(s, &read_fds);
      if (socket.count) FD_SET(s, &write_fds);
      n = select(16, &read_fds, &write_fds, 0, 0);
#endif

      if (n <= 0) continue;		/* nothing to do */

#if POLL
      if (p_fd[0].revents & POLLHUP || p_fd[1].revents & POLLHUP)
	{
	  fprintf(stderr,"Lost connection to child process\n");
	  kill(service_pid,SIGTERM);		/* toast child too */
	  exit(1);
	}
      else if (p_fd[2].revents & POLLHUP)
	{
	  fprintf(stderr,"Lost connection to MBus\n");
	  exit(1);
	}
#endif

      /* deal with the write's first */

      if (SOCKET_WRITE_OK)
	{
	  count = socket.head - socket.tail;
	  if (count <= 0) count = BUFFSIZE - socket.tail;
	  if (count >= MAX_WRITE_SIZE) count = MAX_WRITE_SIZE;
	  count = write(s, socket.buff + socket.tail, count);
	  if (count > 0)
	    {
	      socket.tail = (socket.tail + count) % BUFFSIZE;
	      socket.count -= count;
	    }
	}
      if (CLIENT_WRITE_OK)
	{
	  count = client.head - client.tail;
	  if (count <= 0) count = BUFFSIZE - client.tail;
	  if (count > MAX_WRITE_SIZE) count = MAX_WRITE_SIZE;
	  count = write(service_out, client.buff + client.tail, count);
	  if (count > 0)
	    {
	      client.tail = (client.tail + count) % BUFFSIZE;
	      client.count -= count;
	    }
	}

      /* now, read */
      if (CLIENT_READ_OK)
	{
	  count = socket.tail - socket.head;
	  if (count <= 0) count = BUFFSIZE - socket.head;
	  count = read(service_in, socket.buff + socket.head, count);
	  if (count > 0)
	    {
	      if (trace_fd > 0) write(trace_fd,socket.buff+socket.head,count);
	      socket.head = (socket.head + count) % BUFFSIZE;
	      socket.count += count;
	    }
	  else if (count == 0)
	    {
	      fprintf(stderr, "Lost connection to Mbus\n");
	      exit(0);
	    }
	}
      if (SOCKET_READ_OK)
	{
	  count = client.tail - client.head;
	  if (count <= 0) count = BUFFSIZE - client.head;
	  count = read(s, client.buff + client.head, count);
	  if (count > 0)
	    {
	      if (trace_fd > 0) write(trace_fd,client.buff+client.head,count);
	      client.head = (client.head + count) % BUFFSIZE;
	      client.count += count;
	    }
	  else if (count == 0)
	    {
	      fprintf(stderr, "Lost connection to MBus\n");
	      exit(0);
	    }
	}
    }
}
/* ------------------------------------------------------------------------- */
int DealWithCommandLine(argc,argv)
     int argc;
     char **argv;
{
  int opt;
  char useage = 0;

  while ( (opt = getopt(argc,argv,"t;")) != EOF)
    switch (opt)
      {
      case '?' :
	fprintf(stderr,"%c\tBad option\n",opt_err_char);
	useage = 1;
	break;
      case 't' :
	if (optarg == NULL || !strcmp(optarg,"=")) trace_fd = fileno(stdout);
	else
	  {
	    trace_fd = open(optarg,
#ifdef O_SYNC 
		O_WRONLY|O_SYNC|O_APPEND|O_CREAT,
#else
		O_WRONLY|O_APPEND|O_CREAT,
#endif
		0666);
	    if (trace_fd < 0)
	      {
		fprintf(stderr,"Trace file %s : " , optarg);
		perror("open");
	      }
	  }
	break;
      }

  if (useage)
    {
      /*
	PrintUseMessage(argv);
	*/
      exit(1);
    }
  return optind;
}
/* ------------------------------------------------------------------------- */
main(argc,argv)
     int argc;
     char *argv[];
{
  int port;
  int s;
  int skip;
  char id[256];

  skip = DealWithCommandLine(argc,argv);

  if ((argc - skip) < 3)
    {
      fprintf(stderr,"Usage: %s host port command...\n",argv[0]);
      exit(1);
    }

  port = atoi(argv[1+skip]);
  if (port <= 0)
    {
      fprintf(stderr, "%s is a bad port number\n", argv[1+skip]);
      exit(1);
    }

  service_command = argv[2+skip];
  service_args = argv + 2 + skip;

  /* this succeeds or doesn't return */
  s = ConnectSocket(argv[skip],port);
  /* a little dangerous... */
  sprintf(id, "(id \"%s\")", argv[2+skip]);
  write(s, id, strlen(id));
  StartService();
  Run(s);
}
/* ------------------------------------------------------------------------- */
