/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */
/*
     Nature -- a force-directed graph-drawing program
     author: Thomas Fruchterman
     copyright 1990 Thomas Fruchterman
*/
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <sys/types.h>
#include "extern.h"
#include "graph.h"
#include "config.h"

/*
 *  some simple list manipulation routines to add nodes and edges
 *  to NodeList and Edgelists.  These are straighforward.
*/
void
AddNodetoList(list, node)
     NodeList **list;
     Vertex *node;
{
  NodeList *n_element;

  n_element = (NodeList *)malloc(sizeof(NodeList));
  n_element->node = node;
  n_element->next = *list;
  *list = n_element;
}    

void
AddEdgetoList(list, edge)
     EdgeList **list;
     Edge *edge;
{
  EdgeList *n_element;

  n_element = (EdgeList *)malloc(sizeof(EdgeList));
  n_element->edge = edge;
  n_element->next = *list;
  *list = n_element;
}    

void
DupNodeList(node_list, new_node_list)
  NodeList *node_list, **new_node_list;
{
  NodeList *nodes;
  for (nodes = node_list; nodes; nodes = nodes->next)
      AddNodetoList(new_node_list, nodes->node);
}

/*
 *  Two routines to remove nodes and edges from NodeLists and EdgeLists.
 *  These should be fairly straighforward as well.
*/

void
RemoveEdgefromList(edge_list, edge)
     EdgeList **edge_list;
     Edge *edge;
{
  EdgeList *current, *previous;

/*
 *  Be sure to keep track of the previous NodeList location visted.
*/
  previous = NULL;
  current = *edge_list;
  while (current != NULL) {
    if (current->edge == edge)
/*
 *  Found the right edge to remove
*/
      break;
    else {
      previous = current;
      current = current->next;
    }
  }
/*
 *  If we get here, and we haven't found a node, then we should just
 *  exit.  Trying to remove a node from a list shouldn't crash the
 *  system.
*/
  if (current == NULL)
    return;
/*
 *  Now, move pointers around the edgelist element getting removed
*/
  if (previous == NULL)
    *edge_list = current->next;
  else
    previous->next = current->next;
  free(current);			/* be sure to free the space. */
}

void
RemoveNodefromList(node_list, node)
     NodeList **node_list;
     Vertex *node;
{
  NodeList *current, *previous;

/*
 *  Be sure to keep track of the previous NodeList location visted.
*/
  previous = NULL;
  current = *node_list;
  while (current != NULL) {
    if (current->node == node)
/*
 *  Found the right node to remove
*/
      break;
    else {
      previous = current;
      current = current->next;
    }
  }
/*
 *  If we get here, and we haven't found a node, then we should just
 *  exit.  Trying to remove a node from a list shouldn't crash the
 *  system.
*/
  if (current == NULL)
    return;
/*
 *  Now, move pointers around the nodelist element getting removed
*/
  if (previous == NULL)
    *node_list = current->next;
  else
    previous->next = current->next;
  free(current);			/* be sure to free the space. */
}

void
FreeNodeList(node_list)
  NodeList **node_list;
{
  NodeList *tmp;
  while (*node_list)
  {
    tmp = *node_list;
    *node_list = (*node_list)->next;
    free(tmp);
  }
}

void
FreeEdgeList(edge_list)
  EdgeList **edge_list;
{
  EdgeList *tmp;
  while (*edge_list)
  {
    tmp = *edge_list;
    *edge_list = (*edge_list)->next;
    free(tmp);
  }
}

/*
 * This routine attempts to find the node on the node list.
 * Return TRUE if found, FALSE otherwise.
 */
int
FindNodeonList(node_list, node)
     NodeList *node_list;
     Vertex *node;
{
  for (; node_list; node_list = node_list->next)
  {
    if (node_list->node == node) return TRUE;
  }
  return FALSE;
}

/*
 * This routine adds the node to the list if it isn't already on the list.
 * Returns TRUE if the node was added and FALSE otherwise.
 */
int
NewNodetoList(list, node)
     NodeList **list;
     Vertex *node;
{
  if (!FindNodeonList(*list, node))
  {
    AddNodetoList(list, node);
    return TRUE;
  }
  return FALSE;
}

/*
 *  This routine merges two edgelists.  The second list is appended
 *  onto the end of the first list.
*/
void
MergeEdgeLists(edgelist, addlist)
     EdgeList **edgelist, *addlist;
{
  EdgeList *edges, *prev;

/*
 *  Keep this simple and do nothing if nothing to add.
*/
  if (addlist == NULL)
    return;
  prev = NULL;
/*
 *  Now, check to see if the list we are adding to is NULL.
*/
  if (edgelist == NULL) {
    edgelist = &addlist;
    return;
  }

  edges = *edgelist;
/*
 *  Keep track of the previous edgelist pointer, and skip to the end
 *  of the list
*/
  while (edges != NULL) {
    prev = edges;
    edges = edges->next;
  }
/*
 *  Now, just attach the addlist onto either the edgelist pointer itself
 *  if there are no entries in the edgelist, or onto the end of the
 *  edgelist if there is at least one edge on the edgelist.
*/
  if (prev == NULL)
    *edgelist = addlist;
  else
    prev->next = addlist;
}

/*
 *  This routine merges two nodelists.  The second list is appended
 *  onto the end of the first list.
*/
void
MergeNodeLists(nodelist, addlist)
     NodeList **nodelist, *addlist;
{
  NodeList *nodes, *prev;

/*
 *  Keep this simple and do nothing if nothing to add.
*/
  if (addlist == NULL)
    return;
  prev = NULL;
  nodes = *nodelist;
/*
 *  Keep track of the previous nodelist pointer, and skip to the end
 *  of the list
*/
  while (nodes != NULL) {
    prev = nodes;
    nodes = nodes->next;
  }
/*
 *  Now, just attach the addlist onto either the nodelist pointer itself
 *  if there are no entries in the nodelist, or onto the end of the
 *  nodelist if there is at least one node on the nodelist.
*/
  if (prev == NULL)
    *nodelist = addlist;
  else
    prev->next = addlist;
}

/*
 *  CompareNodes will take two NodeLists and compare them to see if
 *  they have the same element in the list.
*/
int
CompareNodes(list1, list2)
     NodeList *list1, *list2;
{
  NodeList *ptr1, *ptr2;

  for (ptr1 = list1; ptr1 != NULL; ptr1 = ptr1->next) {
    for (ptr2 = list2; ptr2 != NULL; ptr2 = ptr2->next) {
      if (ptr1->node == ptr2->node)
	break;
    }
    if (ptr2 == NULL)
      return FALSE;
  }
  return TRUE;
}

/*
 *  The routine FindSameEdges take an edge, and creates a list of
 *  edges that contain the exact same nodes along the edge.  This is used
 *  for drawing the edges between nodes properly.  The number of edges
 *  that have the exact same nodes along them is returned in count.
*/
void
FindSameEdges(edge, elist, count)
     Edge *edge;
     EdgeList **elist;
     int *count;
{
  Edge *n_edge;

  *count = 0;
  *elist = NULL;
  if (edge == NULL)
    return;
  for (n_edge = edge->next; n_edge != NULL; n_edge = n_edge->next) {
    if (edge->node_count == n_edge->node_count) {
      if (CompareNodes(edge->edge_nodes, n_edge->edge_nodes)) {
	AddEdgetoList(elist, n_edge);
	(*count)++;
      }
    }
  }
  /* Make sure that the edge passed in is at the front */
  AddEdgetoList(elist, edge);
  (*count)++;
}
