/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */
/*
     Nature -- a force-directed graph-drawing program
     author: Thomas Fruchterman
     copyright 1990 Thomas Fruchterman
*/
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <sys/types.h>
#include "extern.h"
#include "graph.h"
#include "config.h"
#include "elision.h"

/*
 * If SCALE is defined, it means to allow the nodes to go whereever
 * they want during the time the force functions are being run and then
 * in the end we will scale them all back down to the 0, 0 to 1, 1
 * coordinates.  If SCALE is not defined, then the nodes will be
 * pushed to the edges and when they try to go past, they will stop
 * at the edge.  There are of course compuational trade offs.  When
 * using scaling we introduce more division (on the order of 2 * n),
 * but we avoid lots of comparisons (on the order of 80 * n).
 *
 * If SCALE_UP is defined then we will take any graph and scale it
 * to the 0, 0 to 1, 1 size.  This means that graphs smaller than
 * 0, 0 to 1, 1 will be brought up to larger dimensions to make them
 * fill the entire graph space.  When used with SCALE, this produces
 * no more calculations; however, when used with SCALE undefined you
 * get the worst of both worlds (i.e., you have the 400*n comparisions
 * AND 2*n divisions.
 */

#define SCALE
#define SCALE_UP
#define SQR(x) ((x) * (x))

/*
 * Only want to add the graph if it doesn't already exist in the table.
 */
void
AddGraphtoList(list, graph, mask)
  struct TableStruct **list;
  Graph *graph;
  int mask;
{
  struct TableStruct *g_element, *ptr;

  for (ptr = *list; ptr; ptr = ptr->next)
      if (ptr->graph == graph)
	  break;
  
  if (ptr)
  {
    ptr->status |= mask;
    return;
  }
  
/* We don't care about the id in this structure right now*/
  g_element = (struct TableStruct *)malloc(sizeof(struct TableStruct));
  g_element->graph = graph;
  g_element->status = mask;
  g_element->next = *list;
  *list = g_element;
}    

/* EdgeGoesTo was written by Doug Bogia on 08/06/91.
 * This function will take an edge and a node and will return TRUE
 * if the edge goes to the node and FALSE if it doesn't.
 */
int
EdgeGoesTo (edge, node)
  Edge *edge;
  Vertex *node;
{
  Link *link;
  for (link = edge->links; link != NULL; link = link->n_edge_adj_node) {
    if (link->tnode == node)
    {
      return (TRUE);
    }
  }
  return (FALSE);
}

void
GraphReassign(cap)
     coord_t cap;
{
  Vertex *node, *node1;
  Link *link;
  EdgeIdList *edge_id;
  Edge *edge;
  coord_t x, y;
  coord_t capsq, distsq;
  double ratio;
  int connected, found1, found2;

/*
 * All the nodes' new positions must be 0.0 here.  We don't check
 * since we create all nodes with newpos set to zero and then
 * later in this function we reset them for the next pass.
 *    
 * Some might worry about the lack of a check here to make sure
 * there is a node in the graph before grabbing the next field; however,
 * since graphDetermine calls this function and it makes sure there
 * are nodes, we don't have to check.  If somebody decides to change
 * this, things will break here.
*/

  for (node = graph->v; node->next != NULL; node = node->next) {
/*
 *  Make a quick check to see if the node is hidden in the graph.  Hidden
 *  nodes don't figure into the force-functions, and shouldn't figure
 *  into any graph layout algorithm (probably).
*/
    if (NodeHidden(node))
	continue;
    for (node1 = node->next; node1 != NULL; node1 = node1->next) {
/*
 *  see comment above
*/
      if (NodeHidden(node1))
	  continue;
/* 
 * What we want to know is whether the node is attached to node1.  By
 * attached we mean is there an edge that in some way joins the
 * two nodes.  Notice that two nodes can be joined by an edge even
 * if the edge only points to the nodes (i.e. there is not a path
 * from one node to the other.  If the two nodes are joined, we will
 * compute an Attractive Force.  Either way, we will compute a
 * Repulsive force to attempt to keep them from being on top of
 * one another.
 * Notice also that we only want one Attractive force computed regardless
 * of the number of edge connections between the nodes.  This is
 * done since computing the force function multiple times can
 * get us into trouble since we can effectively apply forces
 * greater than one which will cause us to flip past the node
 * we are attracted to.
*/
      connected = 0;
      /*
       * First do the quick and easy computations.  See if there
       * is a direct link from node to node1.
       */
      for (link = node->links; !connected && link != NULL; 
	   link = link->n_adj_node)
	  connected = (link->tnode == node1);

      /*
       * Then check if there is a direct link from node1 to node
       */
      for (link = node1->links; !connected && link != NULL; 
	   link = link->n_adj_node)
	  connected = (link->tnode == node);
/*
 *  Now, we will have a check to see if the graph contains hyper-edges.
 *  If the graph does contain hyper edges, then we want to do this
 *  special code that will check for an edge connecting these two
 *  nodes.  The main problem occurs in a graph like:
 *                b
 *                ^
 *                |
 *                |
 *                |
 *                |
 *         a------+------->c
 *
 *  In this case, the above checks would not find that b is connect to
 *  c, but for our purposes, they are connected.  This following code
 *  attempts to remidy this problem.  It is only necessary if the
 *  graph has hyperedged.
*/
      if (graph->have_hyper_edge) {
	/* Run down all the edges that claim to be pointing into the node */
	for (edge_id=node->edge_id_list; !connected && edge_id;
	     edge_id=edge_id->next) {
	  edge = EdgeLookup(edge_id->edge_id, graph->edges);
/*
 * If the edge was not found or it is not a hyper edge, just skip it.
*/
	  if (!edge || edge->node_count < 3)
	      continue;
	  connected = (FindNodeonList(edge->edge_nodes, node) &&
		       FindNodeonList(edge->edge_nodes, node1));
	}
      }
      if (connected)
	  appAFunc(node, node1);
      appRFunc(node, node1);
    }
  }
  
  capsq = cap * cap;

  for(node = graph->v; node != NULL; node = node->next) {
    if (NodeHidden(node))	/* simple check for hidden nodes */
	continue;
/*
 *  check to see if the node is fixed.  if isn't fixed, then do all the
 *  computations.
*/
    if (!NodeFixed(node)
#ifndef SCALE
       && (node->pos[0] != LEFT && node->pos[0] != RIGHT &&
	   node->pos[1] != TOP && node->pos[1] != BOTTOM)
#endif /* SCALE */
	) {
      x = node->newpos[0];
      y = node->newpos[1];
/*
 *  Above assignment saves on expensive lookups.  Next, check and see if
 *  the total distance the node can move is greater that the cap placed
 *  on that node.  If so, scale back the distance so that it only moves
 *  cap distance away, and no more..
*/
      distsq = SQR(x) + SQR(y);
      if (distsq > capsq) {
	ratio = cap / sqrt((double)distsq);
	x = x * (coord_t)ratio;
	y = y * (coord_t)ratio;
      }

      node->pos[0] = node->pos[0] + x;
      node->pos[1] = node->pos[1] + y;

#ifndef SCALE
/*
 *  Keep the nodes on the border, and don't let them go past at all
 */ 
      if (node->pos[0] > RIGHT)	node->pos[0] = RIGHT;
      if (node->pos[0] < LEFT) node->pos[0] = LEFT;
      if (node->pos[1] > BOTTOM) node->pos[1] = BOTTOM;
      if (node->pos[1] < TOP) node->pos[1] = TOP;
#endif /* SCALE */
    }
    node->newpos[0] = node->newpos[1] = 0.0;
  }

#ifdef DRAWSTEP
  TempGraphScale();
  DrawGraph(VIRT_NODES | VIEW_NODES);
  getc(stdin);
#endif

}

GraphScale ()
{
  Vertex *node;
  coord_t min_x, max_x, min_y, max_y, x, y, virtx_scale, virty_scale;

  /*
   * We should check to see if there any nodes here, but since
   * we know that graphDetermine already checked that, we won't.
   */

  /*
   * Run all the nodes to find out where they are.  What we want is
   * the bounding box that will encompass all the nodes.  With this
   * we want to force this to be equivalent to the 0 to 1 box that
   * reflects our virtual space.
   *
   * If SCALE_UP is defined, then if the bounding area is actually
   * inside of the 0, 0 to 1, 1 area, we will scale up the graph
   * layout to take up most (if not all) of the screen size.
   * Something to think about is what happens when we have fixed
   * nodes?  The fixed nodes won't be scaled and thus won't fall
   * in their "proper" position (according to the graph layout),
   * but they will fail where placed.  This could look really bad.
   */
#if defined SCALE_UP || defined SCALE
#ifdef SCALE_UP
  min_x = min_y = 1000.0;	/* Something to start us off */
  max_x = max_y = -1000.0;	/* Something to start us off */
#else
  min_x = min_y = 0.0;		/* Something to start us off */
  max_x = max_y = 1.0;		/* Something to start us off */
#endif /* SCALE_UP */
  for(node = graph->v; node != NULL; node = node->next) {
    if (NodeHidden(node)) continue;
    if ((x = node->pos[0] - node->label_x_offset) < min_x) min_x = x;
    if ((x = node->pos[0] + node->label_x_offset) > max_x) max_x = x;
    if ((y = node->pos[1] - node->label_y_offset) < min_y) min_y = y;
    if ((y = node->pos[1] + node->label_y_offset) > max_y) max_y = y;
  }
  /* Compute the scale */
  x = max_x - min_x;
  y = max_y - min_y;
  /* Make sure we don't divide by zero */
  if (x == 0.0) x = 0.000001;
  if (y == 0.0) y = 0.000001;
#endif /* SCALE_UP || SCALE */
  virtx_scale = graph->virtx_max - graph->virtx_min;
  virty_scale = graph->virty_max - graph->virty_min;
  for(node = graph->v; node != NULL; node = node->next) {
#if defined SCALE || defined SCALE_UP
    if (!NodeFixed(node) && !NodeHidden(node))
    {
      /*
       * Now rescale their virtual coordinates back to
       * 0 to 1.
       */
      node->pos[0] = (node->pos[0] - min_x) / x;
      node->pos[1] = (node->pos[1] - min_y) / y;
      /*
       * Now the problem is that the scaling works fine, except that
       * the text size of the labels doesn't get scaled down.  Thus,
       * it is possible to have a node that hangs off the edges.
       * The code below corrects for this.  First pull back on the right
       * side, then the left.  It is done in this order in case we
       * have a node that is longer than the screen.  If this is the
       * case, we favor seeing the first of the node label instead of
       * the end.  Likewise, we do the bottom check then the top.
       */
      if (node->pos[0] + node->label_x_offset > RIGHTSIDE)
	  node->pos[0] = RIGHTSIDE - node->label_x_offset;
      if (node->pos[0] - node->label_x_offset < LEFTSIDE)
	  node->pos[0] = LEFTSIDE + node->label_x_offset;
      if (node->pos[1] + node->label_y_offset > BOTTOMSIDE)
	  node->pos[1] = BOTTOMSIDE - node->label_y_offset;
      if (node->pos[1] - node->label_y_offset < TOPSIDE)
	  node->pos[1] = TOPSIDE + node->label_y_offset;
    }
#endif /* SCALE || SCALE_UP */
    node->x = (node->pos[0] - graph->virtx_min) / virtx_scale;
    node->y = (node->pos[1] - graph->virty_min) / virty_scale;
  }
}  

#ifdef DRAWSTEP
TempGraphScale ()
{
  Vertex *node;
  coord_t min_x, max_x, min_y, max_y, x, y, virtx_scale, virty_scale;

  /*
   * We should check to see if there any nodes here, but since
   * we know that graphDetermine already checked that, we won't.
   */

  /*
   * Run all the nodes to find out where they are.  What we want is
   * the bounding box that will encompass all the nodes.  With this
   * we want to force this to be equivalent to the 0 to 1 box that
   * reflects our virtual space.
   *
   * If SCALE_UP is defined, then if the bounding area is actually
   * inside of the 0, 0 to 1, 1 area, we will scale up the graph
   * layout to take up most (if not all) of the screen size.
   * Something to think about is what happens when we have fixed
   * nodes?  The fixed nodes won't be scaled and thus won't fall
   * in their "proper" position (according to the graph layout),
   * but they will fail where placed.  This could look really bad.
   */
  min_x = min_y = 0.0;		/* Something to start us off */
  max_x = max_y = 1.0;		/* Something to start us off */
  for(node = graph->v; node != NULL; node = node->next) {
    if ((x = node->pos[0] - node->label_x_offset) < min_x) min_x = x;
    if ((x = node->pos[0] + node->label_x_offset) > max_x) max_x = x;
    if ((y = node->pos[1] - node->label_y_offset) < min_y) min_y = y;
    if ((y = node->pos[1] + node->label_y_offset) > max_y) max_y = y;
  }
  /* Compute the scale */
  x = max_x - min_x;
  y = max_y - min_y;
  /* Make sure we don't divide by zero */
  if (x == 0.0) x = 0.000001;
  if (y == 0.0) y = 0.000001;
  for(node = graph->v; node != NULL; node = node->next) {
    node->x = (node->pos[0] - min_x) / x;
    node->y = (node->pos[1] - min_y) / y;
  }
}  
#endif

void
UpdateGraph(mask)
  int mask;
{
  if (mask & CHANGED_NODES) AssignNodeLabels();
  if (mask & CHANGED_EDGES) RecomputeEdgePos();
  if (mask & REDETERMINE)
      graphDetermine();
  else if (mask & (REDRAW | CHANGED_NODES | CHANGED_EDGES))
      DrawGraph(VIRT_NODES | VIEW_NODES, TRUE);
  return;
}
