#!/usr/bin/env node
/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import esbuild from "esbuild";
import path from "node:path";
import fs from "node:fs";
import linaria from '@linaria/esbuild'

// eslint-disable-next-line no-undef
const BASE = process.cwd();

const preact = path.join(
  BASE,
  "node_modules",
  "preact",
  "compat",
  "dist",
  "compat.module.js",
);

const preactCompatPlugin = {
  name: "preact-compat",
  setup(build) {
    build.onResolve({ filter: /^(react-dom|react)$/ }, (args) => {
      //console.log("onresolve", JSON.stringify(args, undefined, 2));
      return {
        path: preact,
      };
    });
  },
};

const pages = ["OfferRefund", "RequestPayment", "ShowOrderDetails"]
const langs = ["en", "de", "es"]
const entryPoints = pages.map(p => `src/pages/${p}.tsx`);

let GIT_ROOT = BASE;
while (!fs.existsSync(path.join(GIT_ROOT, ".git")) && GIT_ROOT !== "/") {
  GIT_ROOT = path.join(GIT_ROOT, "../");
}
const GIT_HASH = GIT_ROOT === "/" ? 'not defined' : git_hash();

let PACKAGE_VERSION = get_version()
function get_version() {
  try {
    return JSON.parse(fs.readFileSync(path.join(BASE, "package.json"))).version;
  } catch {
    return 'not defined'
  }
}

function git_hash() {
  const rev = fs
    .readFileSync(path.join(GIT_ROOT, ".git", "HEAD"))
    .toString()
    .trim()
    .split(/.*[: ]/)
    .slice(-1)[0];
  if (rev.indexOf("/") === -1) {
    return rev;
  } else {
    return fs.readFileSync(path.join(GIT_ROOT, ".git", rev)).toString().trim();
  }
}
function toCamelCaseName(name, lang) {
  return name
    .replace(/^[A-Z]/, letter => `${letter.toLowerCase()}`) //first letter lowercase
    .replace(/[A-Z]/g, letter => `_${letter.toLowerCase()}`) //snake case
    .concat(`.${lang}.html`); //extension
}

function templatePlugin(options) {
  return {
    name: "template-backend",
    setup(build) {
      build.onEnd(() => {
        for (const pageName of options.pages) {
          for (const langName of options.langs) {
            const css = fs.readFileSync(path.join(build.initialOptions.outdir, `${pageName}.css`), "utf8").toString()
            const js = fs.readFileSync(path.join(build.initialOptions.outdir, `${pageName}.js`), "utf8").toString()
            const location = path.join(build.initialOptions.outdir, toCamelCaseName(pageName, langName))
            const render = new Function(`${js}; return page.buildTimeRendering("${langName}");`)()
            const html = `
          <!doctype html>
          <html lang="${langName}">
            <head>
              ${render.head}
              <style>${css}</style>
            </head>
            <script id="built_time_data">
            </script>
            <body>
              ${render.body}
              <script>${js}</script>
              <script>page.mount("${langName}")</script>
            </body>
          </html>`
            fs.writeFileSync(location, html);
          }
        }
      });
    },
  };
}

export const buildConfig = {
  entryPoints: [...entryPoints],
  bundle: true,
  outdir: "dist/pages",
  /*
   * Doing a minified version will replace templatestring to common strings
   * This app is building mustache template with placeholders that will be replaced
   * with string in runtime by the merchant-backend
   *
   * To the date, merchant backend is replacing with multiline string so 
   * doing minified version will brake at runtime
   * */
  minify: false,
  loader: {
    ".svg": "file",
    ".png": "dataurl",
    ".jpeg": "dataurl",
    '.ttf': 'file',
    '.woff': 'file',
    '.woff2': 'file',
    '.eot': 'file',
  },
  target: ["es2020"],
  format: "iife",
  platform: "browser",
  sourcemap: false,
  globalName: "page",
  jsxFactory: "h",
  jsxFragment: "Fragment",
  define: {
    __VERSION__: `"${PACKAGE_VERSION}"`,
    __GIT_HASH__: `"${GIT_HASH}"`,
  },
  plugins: [
    linaria.default({
      babelOptions: {
        babelrc: false,
        configFile: './babel.config-linaria.json',
      },
      sourceMap: true,
    }),
    preactCompatPlugin,
    templatePlugin({ pages, langs })
  ],
};

await esbuild.build(buildConfig)

export const testingConfig = {
  entryPoints: ["src/render-examples.ts"],
  bundle: true,
  outdir: "dist/test",
  minify: false,
  loader: {
    ".svg": "file",
    ".png": "dataurl",
    ".jpeg": "dataurl",
    '.ttf': 'file',
    '.woff': 'file',
    '.woff2': 'file',
    '.eot': 'file',
  },
  target: ["es2020"],
  format: "iife",
  platform: "node",
  sourcemap: true,
  define: {
    __VERSION__: `"${PACKAGE_VERSION}"`,
    __GIT_HASH__: `"${GIT_HASH}"`,
  },
  plugins: [
  ],
};

await esbuild.build(testingConfig)
