/*
 * signal, raise : handle signals.
 *                 "default" handling for all signals is
 *                 to kill the program quickly by calling _exit.
 */

#include "lib.h"
#include <signal.h>

static void (*vectab[NSIG])();	/* array of functions to catch signals */

void (*signal(signr, func))()
int signr;			/* which signal is being set */
void (*func)();			/* pointer to function that catches signal */
{
  int r;
  void (*old)();

  /*
   * bounds-check signr
   */
  if ((signr <= 0) || (signr >= NSIG))
    {
      errno = ERANGE;
      return SIG_ERR;
    }
  old = vectab[signr - 1];
  vectab[signr - 1] = func;
  return(old);
}

/*
 * raise a signal.
 */
int
  raise(int signr)
{
  void (*func)();

  /*
   * bounds-check the signal.
   */
  if ((signr<=0) || (signr>=NSIG))
    return -1;
  
  func = vectab[signr-1];
  if (func == SIG_DFL)
    _exit(1);
  else if (func == SIG_IGN)
    return 0;		/* just ignore the signal */
  else
    {
      /*
       * prevent recursion by defaulting this signal.
       */
      vectab[signr-1] = SIG_DFL;
      (*func)(signr);
    }
  
  /*
   * if the handler returns, restore table
   * and return 0.
   */

  vectab[signr-1] = func;
  return 0;
}


/*
 * called from crt0 to reset the signal table.
 */

void
  _reset_signals()
{
  int i;

  for (i=NSIG-1; i>=0; i--)
    vectab[i] = SIG_DFL;
}
