/*
 *	Matrix algebra, part of
 *
 *	Nonlinear least squares fit according to the
 *	Marquardt-Levenberg-algorithm
 *
 *	added as Patch to Gnuplot 3.2
 *	by Carsten Grammes
 *	Experimental Physics, University of Saarbruecken, Germany
 *
 *	Internet address: ph12hucg@rz.uni-sb.de
 *
 *	Copyright of this module:   Carsten Grammes, 1993
 *
 *	Permission to use, copy, and distribute this software and its
 *	documentation for any purpose with or without fee is hereby granted,
 *	provided that the above copyright notice appear in all copies and
 *	that both that copyright notice and this permission notice appear
 *	in supporting documentation.
 *
 *	This software is provided "as is" without express or implied warranty.
 *  
 *
 *	Acknowledgment:
 *
 *	The matrix algebra routines herein were developped after
 *	algorithms presented in the book "Numerical recipes in C"
 *	by W. Press, B. Flannery, S. Teukolsky and W. Vetterling. Their
 *	collection of useful mathematical algorithms is recommended to
 *	anyone writing software involving mathematics.
 *
 *	As concerns copyright there are some remarks. The original source
 *	code given in N.R. is copyrighted. It is however (according to the
 *	explicitly given statements by the authors in the preface of the book)
 *	everybodys right to analyze the ideas of a program (including
 *	the sequence of processes adopted by the programmer) and to
 *	express those in one's own distinct implementation. The authors
 *	of the book have done so also, of course. The routines given here
 *	are using the same algorithm as given in N.R. but differ substantially
 *	in the details of the implementation. Therefore I have the copyright
 *	of this implementation and also the right to publish its source
 *	code.
 */


#define MATRIX_MAIN

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "type.h"
#include "fit.h"
#include "matrix.h"


/*****************************************************************/

#define Swap(a,b)   {double temp=(a); (a)=(b); (b)=temp;}
#define WINZIG	      1e-30


/*****************************************************************
    internal prototypes
*****************************************************************/
static void lu_decomp (double **a, int n, int *indx, double *d);
static void lu_backsubst (double **a, int n, int *indx, double b[]);


/*****************************************************************
    first straightforward vector and matrix allocation functions
*****************************************************************/
double *vector (int n)
{
    /* allocates a double vector with n elements */

    if( n < 1 )
	return (double *) NULL;
    return (double *) malloc ( n * sizeof(double) );
}


int *ivector (int n)
{
    /* allocates a int vector with n elements */
    if( n < 1 )
	return (int *) NULL;
    return (int *) malloc ( n * sizeof(int) );
}

double **matrix (int rows, int cols)
{
    /* allocates a double matrix */

    register int i;
    register double **m;

    if ( rows < 1  ||  cols < 1 )
        return NULL;
    /* allocate pointers to rows */
    m = (double **) malloc ( rows * sizeof(double *) );
    /* allocate rows and set pointers to them */
    for ( i=0 ; i<rows ; i++ )
	if ( (m[i] = (double *) malloc (cols * sizeof(double))) == NULL )
	    return (double **) NULL;
    return m;
}


void free_matrix (double **m, int rows)
{
    register int i;
    for ( i=0 ; i<rows ; i++ )
	free ( m[i] );
    free (m);
}


void redim_vector (double **v, int n)
{
    if ( n < 1 ) {
	*v = NULL;
	return;
    }
    *v = (double *) realloc (*v, n * sizeof(double) );
}

void redim_ivector (int **v, int n)
{
    if ( n < 1 ) {
	*v = NULL;
	return;
    }
    *v = (int *) realloc (*v, n * sizeof(int) );
}



/*****************************************************************
    Linear equation solution by Gauss-Jordan elimination
*****************************************************************/
void solve (double **a, int n, double **b, int m)
{
    int     *c_ix, *r_ix, *pivot_ix, *ipj, *ipk,
	    i, ic, ir, j, k, l, s;

    double  large, dummy, tmp, recpiv,
	    **ar, **rp,
	    *ac, *cp, *aic, *bic;

    c_ix	= ivector (n);
    r_ix	= ivector (n);
    pivot_ix	= ivector (n);
    memset (pivot_ix, 0, n*sizeof(int));

    for ( i=0 ; i<n ; i++ ) {
	large = 0.0;
	ipj = pivot_ix;
	ar = a;
	for ( j=0  ;  j<n  ;  j++, ipj++, ar++ )
	    if (*ipj != 1) {
		ipk = pivot_ix;
		ac = *ar;
		for ( k=0  ;  k<n  ;  k++, ipk++, ac++ )
		    if ( *ipk ) {
			if ( *ipk > 1 )
                            error_ex ("Singular matrix");
		    }
		    else {
			if ( (tmp = fabs(*ac)) >= large ) {
			    large = tmp;
			    ir = j;
			    ic = k;
			}
		    }
	    }
	++(pivot_ix[ic]);

	if ( ic != ir ) {
	    ac = b[ir];
	    cp = b[ic];
	    for ( l=0  ;  l<m  ;  l++, ac++, cp++ )
		Swap(*ac, *cp)
	    ac = a[ir];
            cp = a[ic];
            for ( l=0  ;  l<n  ;  l++, ac++, cp++ )
                Swap(*ac, *cp)
        }

	c_ix[i] = ic;
        r_ix[i] = ir;
	if ( *(cp = &a[ic][ic]) == 0.0 )
	    error_ex ("Singular matrix");
	recpiv = 1/(*cp);
	*cp = 1;

	cp = b[ic];
        for ( l=0 ; l<m ; l++ )
            *cp++ *= recpiv;
        cp = a[ic];
	for ( l=0 ; l<n ; l++ )
	    *cp++ *= recpiv;

	ar = a;
	rp = b;
	for ( s=0 ; s<n ; s++, ar++, rp++ )
	    if ( s != ic ) {
		dummy = (*ar)[ic];
		(*ar)[ic] = 0;
		ac = *ar;
		aic = a[ic];
		for ( l=0 ; l<n ; l++ )
		    *ac++ -= *aic++ * dummy;
		cp = *rp;
		bic = b[ic];
		for ( l=0 ; l<m ; l++ )
		    *cp++ -= *bic++ * dummy;
	    }
    }

    for ( l=n-1 ; l>=0 ; l-- )
	if ( r_ix[l] != c_ix[l] )
	    for ( ar=a, k=0  ;	k<n  ;	k++, ar++ )
		Swap ((*ar)[r_ix[l]], (*ar)[c_ix[l]])

    free (pivot_ix);
    free (r_ix);
    free (c_ix);
}


/*****************************************************************
    LU-Decomposition of a quadratic matrix
*****************************************************************/
static void lu_decomp (double **a, int n, int *indx, double *d)
{
    int     i, imax, j, k;

    double  large, dummy, temp,
	    **ar, **lim,
	    *limc, *ac, *dp, *vscal;

    dp = vscal = vector (n);
    *d = 1.0;
    for ( ar=a, lim=&a[n] ; ar<lim ; ar++ ) {	/* loop over rows */
	large = 0.0;
	for ( ac=*ar, limc=&ac[n] ; ac<limc ; )
	    if ( (temp = fabs (*ac++)) > large )
		large = temp;
	if ( large == 0.0 )
	    error_ex ("Singular matrix in LU-DECOMP");
	*dp++ = 1/large;			/* save scaling */
    }
    ar = a;
    for ( j=0 ; j<n ; j++, ar++ ) {		/* loop over columns */
	for ( i=0 ; i<j ; i++ ) {
	    ac = &a[i][j];
	    for ( k=0 ; k<i ; k++ )
		*ac -= a[i][k] * a[k][j];
	}
	large = 0.0;
	dp = &vscal[j];
	for ( i=j ; i<n ; i++ ) {
	    ac = &a[i][j];
            for ( k=0 ; k<j ; k++ )
		*ac -= a[i][k] * a[k][j];
	    if ( (dummy = *dp++ * fabs(*ac)) >= large ) {   /* better pivot? */
		large = dummy;
		imax = i;
	    }
	}
	if ( j != imax ) {		    /* interchange rows ? */
	    ac = a[imax];
	    dp = *ar;
	    for ( k=0 ; k<n ; k++, ac++, dp++ ) 	  /* Yeah */
		Swap (*ac, *dp);
	    *d = -(*d); 		    /* change sign of d   */
	    vscal[imax] = vscal[j];	    /* and scale factor   */
	}
	indx[j] = imax;
	if ( *(dp = &(*ar)[j]) == 0 )
	    *dp = WINZIG;

	if ( j != n-1 ) {		    /* finally divide by pivot */
	    dummy = 1/(*ar)[j];
	    for ( i=j+1 ; i<n ; i++ )
		a[i][j] *= dummy;
	}
    }					    /* next column */
    free (vscal);
}


/*****************************************************************
    Routine for backsubstitution
*****************************************************************/
static void lu_backsubst (double **a, int n, int *indx, double b[])
{
    int     i, memi = -1, ip, j;

    double  sum, *bp, *bip, **ar, *ac;

    ar = a;
    for ( i=0 ; i<n ; i++, ar++ ) {
	ip = indx[i];
	sum = b[ip];
	b[ip] = b[i];
	if (memi >= 0) {
	    ac = &(*ar)[memi];
	    bp = &b[memi];
	    for ( j=memi ; j<=i-1 ; j++ )
		sum -= *ac++ * *bp++;
	}
        else
	    if ( sum )
		memi = i;
	b[i] = sum;
    }
    ar--;
    for ( i=n-1 ; i>=0 ; i-- ) {     /* now backsubstitution */
	ac = &(*ar)[i+1];
	bp = &b[i+1];
	bip = &b[i];
	for ( j=i+1 ; j<n ; j++ )
	    *bip -= *ac-- * *bp--;
	*bip /= (*ar--)[i];
    }
}


/*****************************************************************
    matrix inversion
*****************************************************************/
void inverse (double **src, double **dst, int n)
{
    int i,j;
    int *indx;
    double d, *col, **tmp;

    indx = ivector (n);
    col = vector (n);
    tmp = matrix (n, n);
    for ( i=0 ; i<n ; i++ )
	memcpy (tmp[i], src[i], n*sizeof(double));

    lu_decomp (tmp, n, indx, &d);

    for ( j=0 ; j<n ; j++ ) {
	for ( i=0 ; i<n ; i++ )
	    col[i] = 0;
	col[j] = 1;
	lu_backsubst (tmp, n, indx, col);
	for ( i=0 ; i<n ; i++ )
	    dst[i][j] = col[i];
    }
    free (indx);
    free (col);
    free_matrix (tmp, n);
}


