/*
FUNCTION
<<log1p>>, <<log1pf>>---log of <<1 + <[x]>>>

INDEX
	log1p
INDEX
	log1pf

ANSI_SYNOPSIS
	#include <math.h>
	double log1p(double <[x]>);
	float log1pf(float <[x]>);

TRAD_SYNOPSIS
	#include <math.h>
	double log1p(<[x]>)
	double <[x]>;

	float log1pf(<[x]>)
	float <[x]>;

DESCRIPTION
<<log1p>> calculates 
@tex
$ln(1+x)$, 
@end tex
the natural logarithm of <<1+<[x]>>>.  You can use <<log1p>> rather
than `<<log(1+<[x]>)>>' for greater precision when <[x]> is very
small.

<<log1pf>> calculates the same thing, but accepts and returns
<<float>> values rather than <<double>>.

RETURNS
<<log1p>> returns a <<double>>, the natural log of <<1+<[x]>>>.
<<log1pf>> returns a <<float>>, the natural log of <<1+<[x]>>>.

You can control the error behavior via <<matherr>>.  By default, 
<<log1p>> sets <<errno>> to <<ERANGE>> and returns an IEEE infinity
when <[x]> is <<-1>>, and sets <<errno>> to <<EDOM>> and returns an IEEE
NaN (not a number) when <[x]> is less than <<-1>>.

PORTABILITY
Neither <<log1p>> nor <<log1pf>> is required by ANSI C or by the System V
Interface Definition (Issue 2).

*/
 

#include "mathimpl.h" 

#ifdef DOUBLE 
#define SNAME "log1p"
#else
#define SNAME "log1pf"
#endif


TYPE_RET									
_DEFUN(log1p,(xa),							
      TYPE_ARG xa)								
{ 									
  TYPE z,s,t,c;
  TYPE x = xa;
  
  int k; 								

  BITSTYPE temp;
	  
  double p;
 									
  if(!finite(x) || x <= -1.0) 
  {
    if ( x == -1.0 )
    { 						
      return __matherror(SNAME, x, 0.0, OVERFLOW, infinity());
    }
    return __matherror(SNAME, x, 0.0, DOMAIN, nan());
  }
  
	  
  /* argument reduction */ 					
  if (-1e20 < x && x < 1e-20 ) 					
   return x; 							
	
  p = 1.0 + x;
  temp.value = p;
  k= temp.number.exponent - IEEE_BIAS;

  z=scalbn(x,-k);						
  t=scalbn(1.0,-k); 						
									
  if(z+t >= M_SQRT2 )  						
  { 								
    k += 1 ;							
    z *= 0.5; 							
    t *= 0.5; 							
  } 								
  t -= 1.0;							
  x = z + t; 							
  c = (t-x)+z ;			/* correction term for x */ 		
 									
  /* compute log(1+x)  */ 					
  s = x/(2+x);							
  t = x*x*0.5; 							
  c += ((double)k* M_LN2LO-c*x); 					
  z = c+s*(t+log__L(s*s)); 					
  x += (z - t) ; 						
 									
  return ((double)k*M_LN2HI+x); 					
} 									
									 




