/*
 | Copyright (c) 1989 The Regents of the University of California.
 | All rights reserved.
 |
 | Redistribution and use in source and binary forms, with or without
 | modification, are permitted provided that the following conditions
 | are met:
 | 1. Redistributions of source code must retain the above copyright
 |    notice, this list of conditions and the following disclaimer.
 | 2. Redistributions in binary form must reproduce the above copyright
 |    notice, this list of conditions and the following disclaimer in the
 |    documentation and/or other materials provided with the distribution.
 | 3. All advertising materials mentioning features or use of this software
 |    must display the following acknowledgement:
 |	This product includes software developed by the University of
 |	California, Berkeley and its contributors.
 | 4. Neither the name of the University nor the names of its contributors
 |    may be used to endorse or promote products derived from this software
 |    without specific prior written permission.
 |
 | THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 | ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 | IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 | ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 | FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 | DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 | OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 | HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 | LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 | OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 | SUCH DAMAGE.
 */


/* 
FUNCTION 
<<fmod>>, <<fmodf>>---floating-point remainder (modulo)

INDEX
fmod
INDEX
fmodf

ANSI_SYNOPSIS
#include <math.h>
double fmod(double <[x]>, double <[y]>)
float fmodf(float <[x]>, float <[y]>)

TRAD_SYNOPSIS
#include <math.h>
double fmod(<[x]>, <[y]>)
double (<[x]>, <[y]>);

float fmodf(<[x]>, <[y]>)
float (<[x]>, <[y]>);

DESCRIPTION
The <<fmod>> and <<fmodf>> functions compute the floating-point
remainder of <[x]>/<[y]> (<[x]> modulo <[y]>).

RETURNS
The <<fmod>> function returns the value 
@ifinfo
<[x]>-<[i]>*<[y]>, 
@end ifinfo
@tex
$x-i\times y$,
@end tex
for the largest integer <[i]> such that, if <[y]> is nonzero, the
result has the same sign as <[x]> and magnitude less than the
magnitude of <[y]>. 

<<fmod(INF,<[y]>)>> is an invalid operation and NaN (not a number) is
returned. <<fmod(<[x]>,0)>> returns 0.0;

You can modify error treatment for these functions using <<matherr>>.

PORTABILITY
<<fmod>> is ANSI C. <<fmodf>> is an extension.
*/

#include "mathimpl.h"

#ifdef DOUBLE
#define SNAME "fmod"
#else
#define SNAME "fmodf"
#endif

TYPE_RET					       
_DEFUN(fmod, (xa,ya),			       
       TYPE_ARG xa _AND			       
       TYPE_ARG ya)				       
{					       
  int ir,iy;		
/* Careful, otherwise might be passed a double which can't be
   represented in a float */
		       
  TYPE r,w;	 
  TYPE x;
  TYPE y;

  y = ya;
  x = xa;

  if (y == 0)
   return __matherror(SNAME, x, y, DOMAIN, x);
  
  if (!finite(y) || !finite(x)) 
   return infinity();
  r = fabs(x);				       
  y = fabs(y);				       

  frexp(y,&iy);		       

  while (r >= y) 			       
  {					       
    frexp(r,&ir);	      
    w = ldexp (y ,ir-iy);		       
    r -= w <= r ? w : w*(double)0.5;	       
  }					       
  return x >= 0 ? r : -r;		       
}
