/*
FUNCTION
	<<exp>>, <<expf>>---exponential
INDEX
	exp
INDEX
	expf

ANSI_SYNOPSIS
	#include <math.h>
	double exp(double <[x]>);
	float expf(float <[x]>);

TRAD_SYNOPSIS
	#include <math.h>
	double exp(<[x]>);
	double <[x]>;

	float expf(<[x]>);
	float <[x]>;

DESCRIPTION
	<<exp>> and <<expf>> calculate the exponential of <[x]>, that is, 
	@ifinfo
	e raised to the power <[x]> (where e
	@end ifinfo
	@tex
	$e^x$ (where $e$
	@end tex
	is the base of the natural system of logarithms, approximately 2.71828).

	You can use the (non-ANSI) function <<matherr>> to specify
	error handling for these functions.

RETURNS
	On success, <<exp>> and <<expf>> return the calculated value.
	Otherwise, on underflow (when <[x]> is a negative number whose
	magnitude is too large for the result to be representable),
	the returned value is <<0>>.  When <[x]> is NaN (not a
	number), the returned value is the same as <[x]>, and the
	global <<errno>> is set to <<EDOM>>.  When <[x]> is too large,
	the global <<errno>> is set to <<ERANGE>>.  The returned value
	is <<HUGE_VAL>> (the largest representable <<double>>) for
	<<exp>>, or <<FLT_MAX>>, the largest representable
	floating-point value for <<expf>>. 

PORTABILITY
	<<exp>> is ANSI C.  <<expf>> is an extension.

*/

#include "mathimpl.h"
#ifdef FLOAT
#define SNAME "expf"
#else
#define SNAME "exp"
#endif


/************************************************************************
 *									*
 *				N O T I C E				*
 *									*
 *			Copyright Abandoned, 1987, Fred Fish		*
 *									*
 *	This previously copyrighted work has been placed into the	*
 *	public domain by the author (Fred Fish) and may be freely used	*
 *	for any purpose, private or commercial.  I would appreciate	*
 *	it, as a courtesy, if this notice is left in all copies and	*
 *	derivative works.  Thank you, and enjoy...			*
 *									*
 *	The author makes no warranty of any kind with respect to this	*
 *	product and explicitly disclaims any implied warranties of	*
 *	merchantability or fitness for any particular purpose.		*
 *									*
 ************************************************************************
 */

/*
 |  FUNCTION
 |
 |	exp   double precision exponential
 |
 |  KEY WORDS
 |
 |	exp
 |	machine independent routines
 |	math libraries
 |
 |  DESCRIPTION
 |
 |	Returns double precision exponential of double precision
 |	floating point number.
 |
 |  USAGE
 |
 |	double exp (double x)
 |
 |  REFERENCES
 |
 |	Fortran IV plus users guide, Digital Equipment Corp. pp B-3
 |
 |	Computer Approximations, J.F. Hart et al, John Wiley & Sons,
 |	1968, pp. 96-104.
 |
 |  RESTRICTIONS
 |
 |	Inputs greater than log(DBL_MAX) result in overflow.
 |	Inputs less than log(DBL_MIN) result in underflow.
 |
 |	The maximum relative error for the approximating polynomial
 |	is 10**(-16.4).  However, this assumes exact arithmetic
 |	in the polynomial evaluation.  Additional rounding and
 |	truncation errors may occur as the argument is reduced
 |	to the range over which the polynomial approximation
 |	is valid, and as the polynomial is evaluated using
 |	finite precision arithmetic.
 |	
 |  PROGRAMMER
 |
 |	Fred Fish
 |
 |  INTERNALS
 |
 |	Computes exponential from:
 |
 |		exp(x)	=	2**y  *  2**z  *  2**w
 |
 |	Where:
 |
 |		y	=	int ( x * log2(e) )
 |
 |		v	=	16 * frac ( x * log2(e))
 |
 |		z	=	(1/16) * int (v)
 |
 |		w	=	(1/16) * frac (v)
 |
 |	Note that:
 |
 |		0 =< v < 16
 |
 |		z = {0, 1/16, 2/16, ...15/16}
 |
 |		0 =< w < 1/16
 |
 |	Then:
 |
 |		2**z is looked up in a table of 2**0, 2**1/16, ...
 |
 |		2**w is computed from an approximation:
 |
 |			2**w	=  (A + B) / (A - B)
 |
 |			A	=  C + (D * w * w)
 |
 |			B	=  w * (E + (F * w * w))
 |
 |			C	=  20.8137711965230361973
 |
 |			D	=  1.0
 |
 |			E	=  7.2135034108448192083
 |
 |			F	=  0.057761135831801928
 |
 |		Coefficients are from HART, table #1121, pg 206.
 |
 |		Effective multiplication by 2**y is done by a
 |		floating point scale with y as scale argument.
 |
 */
 

# define C  20.8137711965230361973	/* Polynomial approx coeff.	*/
# define D  1.0				/* Polynomial approx coeff.	*/
# define E  7.2135034108448192083	/* Polynomial approx coeff.	*/
# define F  0.057761135831801928	/* Polynomial approx coeff.	*/
 
/************************************************************************
 *									*
 *	This table is fixed in size and reasonably hardware		*
 *	independent.  The given constants were generated on a 		*
 *	DECSYSTEM 20 using double precision FORTRAN.			*
 *									*
 ************************************************************************
 */


static TYPE TABLE[] = {
    1.00000000000000000000,		/*    2 ** 0/16		*/
    1.04427378242741384020,		/*    2 ** 1/16		*/
    1.09050773266525765930,		/*    2 ** 2/16		*/
    1.13878863475669165390,		/*    2 ** 3/16		*/
    1.18920711500272106640,		/*    2 ** 4/16		*/
    1.24185781207348404890,		/*    2 ** 5/16		*/
    1.29683955465100966610,		/*    2 ** 6/16		*/
    1.35425554693689272850,		/*    2 ** 7/16		*/
    1.41421356237309504880,		/*    2 ** 8/16		*/
    1.47682614593949931110,		/*    2 ** 9/16		*/
    1.54221082540794082350,		/*    2 ** 10/16	*/
    1.61049033194925430820,		/*    2 ** 11/16	*/
    1.68179283050742908600,		/*    2 ** 12/16	*/
    1.75625216037329948340,		/*    2 ** 13/16	*/
    1.83400808640934246360,		/*    2 ** 14/16	*/
    1.91520656139714729380		/*    2 ** 15/16	*/
};



#define LOGHUGE 308.254715559
#define LOGTINY -308.347344429

/* This is a very important one to get right, perform intermediate
   calcs in as high a precision as possible */

TYPE_RET
_DEFUN(exp, (xa), TYPE_ARG xa)

{
  register int y;
  register int index;
  TYPE x = xa;
  double  w;
  double v;
  double a;
  double b;
  double t;
  TYPE temp;
  double wpof2;
  double zpof2;

  if(!finite(x))
  {								  
    return __matherror(SNAME, x, 0.0, DOMAIN, x);		  
  }								  
        							  
  if( x > LOGHUGE) 						  
  {		        					  
    return __matherror(SNAME, x, 0.0, OVERFLOW, TYPE_MAX);	  
  }		        					  
  		        					  
		        					  
  if( x < LOGTINY) 						  
  {		        					  
    return __matherror(SNAME, x, 0.0 , UNDERFLOW, 0.0);		  
  }	


  
  t = x * M_LOG2E;
  v = 16.0 * modf (t, &temp);
  y = temp;
  w = modf (v, &temp) / 16.0;
  index = temp;

  if (x < 0.0) 
  {
    zpof2 = 1.0 / TABLE[-index];
  } 
  else

  {
    zpof2 = TABLE[index];
  }
  a = C + (D * w * w);
  b = w * (E + (F * w * w));
  wpof2 = (a + b) / (a - b);
  x = ldexp ((wpof2 * zpof2), y);

  return x;
  
}

