/*
FUNCTION
        <<erf>>, <<erff>>, <<erfc>>, <<erfcf>>---error function 
INDEX
	erf
INDEX
	erff
INDEX
	erfc
INDEX
	erfcf

ANSI_SYNOPSIS
	#include <math.h>
	double erf(double <[x]>);
	float erff(float <[x]>);
	double erfc(double <[x]>);
	float erfcf(float <[x]>);
TRAD_SYNOPSIS
	#include <math.h>

	double erf(<[x]>)
	double <[x]>;

	float erff(<[x]>)
	float <[x]>;

	double erfc(<[x]>)
	double <[x]>;

	float erfcf(<[x]>)
	float <[x]>;

DESCRIPTION
	<<erf>> calculates an approximation to the ``error function'',
	which estimates the probability that an observation will fall within
	<[x]> standard deviations of the mean (assuming a normal
	distribution).
	@tex
	The error function is defined as
	$${2\over\sqrt\pi}\times\int_0^x e^{-t^2}dt$$
	 @end tex

	<<erfc>> calculates the complementary probability; that is,
	<<erfc(<[x]>)>> is <<1 - erf(<[x]>)>>.  <<erfc>> is computed directly,
	so that you can use it to avoid the loss of precision that would
	result from subtracting large probabilities (on large <[x]>) from 1.

	<<erff>> and <<erfcf>> differ from <<erf>> and <<erfc>> only in the
	argument and result types.

RETURNS
	For positive arguments, <<erf>> and all its variants return a
	probability---a number between 0 and 1.

PORTABILITY
	None of the variants of <<erf>> are ANSI C.
*/

#include "mathimpl.h"

/*   The main computation evaluates near-minimax approximations 
   from "Rational Chebyshev approximations for the error function" 
   by W. J. Cody, Math. Comp., 1969, PP. 631-638.  This 
   transportable program uses rational functions that theoretically 
   approximate  erf(x)  and  erfc(x)  to at least 18 significant 
   decimal digits.  The accuracy achieved depends on the arithmetic 
   system, the compiler, the intrinsic functions, and proper 
   selection of the machine-dependent constants.
*/

/* Explanation of machine-dependent constants */

/*|  XMIN   = the smallest positive floating-point number. */
/*|  XINF   = the largest positive finite floating-point number. */
/*|  XNEG   = the largest negative argument acceptable to ERFCX; */
/*|           the negative of the solution to the equation */
/*|           2*exp(x*x) = XINF. */
/*|  XSMALL = argument below which erf(x) may be represented by */
/*|           2*x/sqrt(pi)  and above which  x*x  will not underflow. */
/*|           A conservative value is the largest machine number X */
/*|           such that   1.0 + X = 1.0   to machine precision. */
/*|  XBIG   = largest argument acceptable to ERFC;  solution to */
/*|           the equation:  W(x) * (1-0.5/x**2) = XMIN,  where */
/*|           W(x) = exp(-x*x)/[x*sqrt(pi)]. */
/*|  XHUGE  = argument above which  1.0 - 1/(2*x*x) = 1.0  to */
/*|           machine precision.  A conservative value is */
/*|           1/[2*sqrt(XSMALL)] */
/*|  XMAX   = largest acceptable argument to ERFCX; the minimum */
/*|           of XINF and 1/[sqrt(pi)*XMIN]. */
  
/*   Approximate values for some important machines are: */
  
/*|                         XMIN       XINF        XNEG     XSMALL */
  


/*| IEEE (IBM/XT, */
/*|   SUN, etc.)  (S.P.)  1.18E-38    3.40E+38     -9.382  5.96E-8 */
/*| IEEE (IBM/XT, */
/*|   SUN, etc.)  (D.P.)  2.23D-308   1.79D+308   -26.628  1.11D-16 */

  
  
/*|                         XBIG       XHUGE       XMAX */
  
/*| IEEE (IBM/XT, */
/*|   SUN, etc.)  (S.P.)   9.194      2.90E+3     4.79E+37 */
/*| IEEE (IBM/XT, */
/*|   SUN, etc.)  (D.P.)  26.543      6.71D+7     2.53D+307 */
/*| IBM 195       (D.P.)  13.306      1.90D+8     7.23E+75 */
/*| UNIVAC 1108   (D.P.)  26.582      5.37D+8     8.98D+307 */
/*| VAX D-Format  (D.P.)   9.269      1.90D+8     1.70D+38 */
/*| VAX G-Format  (D.P.)  26.569      6.71D+7     8.98D+307 */
  
#define d_int(x) ((x > 0) ? floor(x) : -floor(-x))

static double a[5] = 
{
  3.1611237438705656,
  113.864154151050156,
  377.485237685302021,
  3209.37758913846947,
  .185777706184603153 
 };

static double b[4] =
{
  23.6012909523441209,
  244.024637934444173,
  1282.61652607737228,
  2844.23683343917062 
};
static double c[9] = 
{
  .564188496988670089,
  8.88314979438837594,
  66.1191906371416295,
  298.635138197400131,
  881.95222124176909,
  1712.04761263407058,
  2051.07837782607147,
  1230.33935479799725,
  2.15311535474403846e-8
 };
static double d[8] = 
{
  15.7449261107098347,
  117.693950891312499,
  537.181101862009858,
  1621.38957456669019,
  3290.79923573345963,
  4362.61909014324716,
  3439.36767414372164,
  1230.33935480374942 
};

static double p[6] = 
{
  .305326634961232344,
  .360344899949804439,
  .125781726111229246,
  .0160837851487422766,
  6.58749161529837803e-4,
  .0163153871373020978
};

static double q[5] = 
{
  2.56852019228982242,
  1.87295284992346047,
  .527905102951428412,
  .0605183413124413191,
  .00233520497626869185 
};

#define XSMALL  1.1e-16 
#define XNEG    -26.628 
#define XMAX     2.53e307
#define XBIG    26.543  
#define XHUGE   6.71e7 

#define NAME calerf
#define DOUBLE


#include "erf.h"
#undef DOUBLE
#define FLOAT
#undef NAME
#undef XSMALL
#undef XNEG
#undef XMAX
#undef XBIG
#undef XHUGE
#define XSMALL  5.9e-8
#define XNEG    -9.382
#define XMAX	 4.79e37 
#define XBIG	 9.194
#define XHUGE	 2.90e3


#define NAME calerff
#include "erf.h"


double 
_DEFUN(erf,(x),
      double x)
{
  return  calerf(x,0);
} 

double 
_DEFUN(erfc,(x),
      double x)
{
  return   calerf(x,1);

} 



_FLOAT_RET
_DEFUN(erff,(x),
      _FLOAT_ARG x)
{
  return  calerff(x,0);
} 

_FLOAT_RET
_DEFUN(erfcf,(x),
      _FLOAT_ARG x)
{
  return   calerff(x,1);
} 


