
/*
FUNCTION
        <<atan>>, <<atanf>>---Arctangent

INDEX
   atan
INDEX
   atanf

ANSI_SYNOPSIS
        #include <math.h>
        double atan(double <[x]>);
        float atanf(float <[x]>);

TRAD_SYNOPSIS
        #include <math.h>
        double atan(<[x]>);
        double <[x]>;

        float atanf(<[x]>);
        float <[x]>;

DESCRIPTION

<<atan>> computes the inverse tangent (Arctan) of the input value.
Arguments to <<atan>> must be in the range @minus{}1 to 1. 

<<atanf>> is identical to <<atan>>, save that it operates on <<floats>>.

RETURNS
@ifinfo
<<atan>> returns a value in radians, in the range of -pi/2 to pi/2.
@end ifinfo
@tex
<<atan>> returns a value in radians, in the range of $-\pi/2$ to $\pi/2$.
@end tex

When the argument <[x]> is not in the range @minus{}1 to 1, the
result is NaN (not a number), the global variable <<errno>> is set
to <<EDOM>>, and <<DOMAIN error>> is sent to standard error.

You can modify error handling for these routines using <<matherr>>.

PORTABILITY
<<atan>> is ANSI C.  <<atanf>> is an extension.

*/



#include "mathimpl.h"

/*
	floating-point arctangent

	atan returns the value of the arctangent of its
	argument in the range [-pi/2,pi/2].

	atan2 returns the arctangent of arg1/arg2
	in the range [-pi,pi].

	there are no error returns.

	coefficients are #5077 from Hart & Cheney. (19.56D)
*/

#define pio2	 1.570796326794896619
#define sq2p1	 2.414213562373095048802e0
#define sq2m1	  .414213562373095048802e0
#define pio4	  .785398163397448309615e0
#define p4	  .161536412982230228262e2
#define p3	  .26842548195503973794141e3
#define p2	  .11530293515404850115428136e4
#define p1	  .178040631643319697105464587e4
#define p0	  .89678597403663861959987488e3
#define q4	  .5895697050844462222791e2
#define q3	  .536265374031215315104235e3
#define q2	  .16667838148816337184521798e4
#define q1	  .207933497444540981287275926e4
#define q0	  .89678597403663861962481162e3



/*
	xatan evaluates a series valid in the
	range [-0.414...,+0.414...].
*/

static 
TYPE_RET
_DEFUN(_xatan,(arg),
       TYPE_ARG arg)

{
	TYPE argsq;
	TYPE value;

	argsq = arg*arg;
	value = ((((p4*argsq + p3)*argsq + p2)*argsq + p1)*argsq + p0);
	value = value/(((((argsq + q4)*argsq + q3)*argsq + q2)*argsq + q1)*argsq + q0);
	return(value*arg);
}



TYPE_RET 

_DEFUN(_satan,(arg),
       TYPE_ARG arg)
{
  if(arg < sq2m1)
   return(_xatan(arg));
  else if(arg > sq2p1)
   return(pio2 - _xatan(1.0/arg));
  return(pio4 + _xatan((arg-1.0)/(arg+1.0)));
}



TYPE_RET 
_DEFUN(atan,(arg), 
       TYPE_ARG  arg)
{
	if(arg>0)
		return(_satan(arg));
	else
		return(-_satan(-arg));
}




