/*
 | Copyright (c) 1985 Regents of the University of California.
 | All rights reserved.
 |
 | Redistribution and use in source and binary forms, with or without
 | modification, are permitted provided that the following conditions
 | are met:
 | 1. Redistributions of source code must retain the above copyright
 |    notice, this list of conditions and the following disclaimer.
 | 2. Redistributions in binary form must reproduce the above copyright
 |    notice, this list of conditions and the following disclaimer in the
 |    documentation and/or other materials provided with the distribution.
 | 3. All advertising materials mentioning features or use of this software
 |    must display the following acknowledgement:
 |	This product includes software developed by the University of
 |	California, Berkeley and its contributors.
 | 4. Neither the name of the University nor the names of its contributors
 |    may be used to endorse or promote products derived from this software
 |    without specific prior written permission.
 |
 | THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 | ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 | IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 | ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 | FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 | DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 | OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 | HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 | LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 | OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 | SUCH DAMAGE.
 */



/*
FUNCTION
	<<asinh>>, <<asinhf>>---inverse hyperbolic sine 

INDEX
	asinh
INDEX
	asinhf

ANSI_SYNOPSIS
	#include <math.h>
	double asinh(double <[x]>);
	float asinhf(float <[x]>);

TRAD_SYNOPSIS
	#include <math.h>
	double asinh(<[x]>)
	double <[x]>;

	float asinhf(<[x]>)
	float <[x]>;

DESCRIPTION
<<asinh>> calculates the inverse hyperbolic sine of <[x]>.
<<asinh>> is defined as 
@ifinfo
. sgn(<[x]>) * log(abs(<[x]>) + sqrt(1+<[x]>*<[x]>))
@end ifinfo
@tex
$$sign(x) \times ln\Bigl(|x| + \sqrt{1+x^2}\Bigr)$$
@end tex

<<asinhf>> is identical, other than taking and returning floats.

RETURNS
<<asinh>> and <<asinhf>> return the calculated value, save when <[x]>
is NaN (not a number).  In that case, the result is also NaN, the
global <<errno>> is set to <<EDOM>>, and a <<DOMAIN error>> message is
sent to standard error output.

You can modify error handling using <<matherr>>.

PORTABILITY
Neither <<asinh>> nor <<asinhf>> are ANSI C.

*/
/* ASINH(X)
 | RETURN THE INVERSE HYPERBOLIC SINE OF X
 | DOUBLE PRECISION (VAX D format 56 bits, IEEE DOUBLE 53 BITS)
 | CODED IN C BY K.C. NG, 2/16/85;
 | REVISED BY K.C. NG on 3/7/85, 3/24/85, 4/16/85.
 |
 | Required system supported functions :
 |	copysign(x,y)
 |	sqrt(x)
 |
 | Required kernel function:
 |	log1p(x) 		...return log(1+x)
 |
 | Method :
 |	Based on 
 |		asinh(x) = sign(x) * log [ |x| + sqrt(x*x+1) ]
 |	we have
 |	asinh(x) := x  if  1+x*x=1,
 |		 := sign(x)*(log1p(x)+ln2))	 if sqrt(1+x*x)=x, else
 |		 := sign(x)*log1p(|x| + |x|/(1/|x| + sqrt(1+(1/|x|)^2)) )  
 |
 | Accuracy:
 |	asinh(x) returns the exact inverse hyperbolic sine of x nearly rounded.
 |	In a test run with 52,000 random arguments on a VAX, the maximum 
 |	observed error was 1.58 ulps (units in the last place).
 |
 | Constants:
 | The hexadecimal values are the intended ones for the following constants.
 | The decimal values may be used, provided that the compiler will convert
 | from decimal to binary accurately enough to produce the hexadecimal values
 | shown.
 */

#include "mathimpl.h"

#define DOUBLE
#define SNAME "asinh"
#include "asinh.h"

#undef SNAME
#define FLOAT
#define SNAME "asinhf"
#include "asinh.h"

