/*
 * Epoch 4.0 - Raw code for interval support.
 *
 * Buttons will be anchored by intervals; interval left edges correspond to
 * anchor positions in the buffer.  This implementation is dramatically
 * less expensive for updating positions upon insertion/deletion into buffer.
 * References to get position of an anchor point will be more expensive than
 * with markers.
 *
 * This implementation of intervals is a simplified version of an intervals
 * prototype developed by Dan LaLiberte.
 *
 * $Revision: 1.1 $
 * $Source: /import/kaplan/stable/distrib/epoch-4.0b0/src/RCS/interval.c,v $
 * $Date: 91/11/21 12:45:52 $ 
 * $Author: love $
 */

#ifndef LINT
static char rcsid[] = "$Author: love $ $Date: 91/11/21 12:45:52 $ $Source: /import/kaplan/stable/distrib/epoch-4.0b0/src/RCS/interval.c,v $ $Revision: 1.1 $";
#endif

/* #include "config.h" */
/* #include "lisp.h" */
/* #include "buffer.h" */
#include <stdio.h>
#include "interval.h"

#define min(a,b) ((a)<(b) ? (a) : (b))

struct Lisp_Interval *interval_tree;
struct Lisp_Interval *a0,*a1,*a2,*a3,*a4,*a5,*a6,*a7,*a8,*a9;
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* Allocation functions
 */
struct Lisp_Interval *interval_free_list;

int
interval_pool()
{
  struct Lisp_Interval *t;
  int i;

  t = (struct Lisp_Interval *) malloc (50*sizeof(struct Lisp_Interval));
  if (!t) return 1;
  bzero(t,50*sizeof(struct Lisp_Interval));
  for (i = 0; i < 50; i++)
    {
      t->data = interval_free_list;
      interval_free_list = t;
      t++;
    }
  return 0;
}

int num = 0;
struct Lisp_Interval *
get_interval_node()
{
  struct Lisp_Interval *i;
  if (!interval_free_list && interval_pool())
    fprintf(stderr,"Couldn't allocate space"), abort();

  i = interval_free_list;
  interval_free_list = interval_free_list->data;
  i->data = 0;

  i->num = num++;
  return i;
}

free_interval_node(node)
     struct Lisp_Interval *node;
{
  bzero (node,sizeof(struct Lisp_Interval));

  node->data = interval_free_list;
  interval_free_list = node;
}

free_intervals(node)
     struct Lisp_Interval *node;
{
  if (node->left)
    free_intervals(node->left);
  if (node->right)
    free_intervals(node->right);
  if (node->data)
    free_intervals(node->data);
  free_interval_node(node);
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* Utility functions
 */
int
interval_start(interval)
     struct Lisp_Interval *interval;
{
  int left_pos = interval->bump
    - (interval->right ? interval->right->bump : 0);

  /* Find interval's starting point by following parent ptrs up the tree */

  while (interval->parent)
    {
      struct Lisp_Interval *parent = interval->parent;
      int end_left, start_right;

      end_left = (parent->left ? parent->left->bump : 0);
      
      start_right = (parent->bump -
		     (parent->right ? parent->right->bump : 0));

      if (interval == parent->left)
	{
/*	  left_pos += end_left; */
	  ;
	}
      else
	{
	  left_pos += start_right;
	}
      interval = interval->parent;
    }
  
  return left_pos;
}

int
interval_size (interval)
     struct Lisp_Interval *interval;
{
  int s;

  s = interval->bump -
    (interval->left ? interval->left->bump : 0) -
      (interval->right ? interval->right->bump : 0);
  return s;
}

/* return 0 - found, 1 - left, 2 - right */
interval_which_way(interval, pos, left_pos, start)
     struct Lisp_Interval *interval;
     int pos, left_pos, *start;
{
  int end_left, start_right;

  end_left = (interval->left ? interval->left->bump : 0);
  if (pos < end_left)
    {
      *start = left_pos;
      return 1;
    }
  start_right = (interval->bump -
		 (interval->right ? interval->right->bump : 0));
  if (pos >= start_right)
    {
      *start = start_right;
      return 2;
    }
  *start = end_left;
  return 0;
}

struct Lisp_Interval *
find_interval(interval, pos, left_pos, start)
     struct Lisp_Interval *interval;
     int pos, left_pos, *start;
{
  int which_way,found = 0;
  int start2;

  while (!found)
    {
      which_way = interval_which_way(interval, pos, left_pos, &start2);
      switch (which_way)
	{
	case 1:
	  interval = interval->left;
	  break;
	case 2:
	  left_pos = start2;
	  interval = interval->right;
	  break;
	case 0:
	  found++;
	  left_pos = start2;
	  break;
	default:
	  break;
	}
    }
  
  *start = left_pos;
  return interval;
}

struct Lisp_Interval *
find_inner_interval_tree (pos,start)
     int pos,*start;
{
/*  struct Lisp_Interval *interval = XINTERVAL(current_buffer->interval_tree); */
  struct Lisp_Interval *interval = interval_tree;
  struct Lisp_Interval *super_interval_tree = 0,*subintervals = interval;
  int left_pos = 0,super_left_pos = 0,start2;

  while (subintervals)
    {
      interval = find_interval(interval,pos,left_pos,&start2);
      left_pos = *start;
      subintervals = interval->data;
      if (subintervals)
	{
	  super_interval_tree = interval;
	  super_left_pos = left_pos;
	  interval = subintervals;
	}
    }
  start2 = super_left_pos;
  return super_interval_tree;
}

struct Lisp_Interval *
split_interval_here (pos)
     int pos;
{
  struct Lisp_Interval *new_subs,*new_int;

  interval_tree = split_interval(interval_tree,pos,0,&new_int);
  return new_int;
}

struct Lisp_Interval *
split_interval (interval,pos,left_pos,new_int)
     struct Lisp_Interval *interval;
     int pos,left_pos;
     struct Lisp_Interval **new_int;
{
  int start,found = 0;
  int which;

  while (!found)
    {
      which = interval_which_way(interval,pos,left_pos,&start);
      left_pos = start;
      
/*      if (pos == 767) snapshot(); */
      switch(which)
	{
	case 1:
	  interval = splay_interval_left(interval);
	  break;
	case 2:
	  interval = splay_interval_right(interval);
	  break;
	case 0:
	  found++;
	  if ((pos - left_pos) > 0)
	    {
	      struct Lisp_Interval *left = interval->left;
	      struct Lisp_Interval *new = get_interval_node();

	      *new_int = new;	/* New node being added */
/*	      new->buffer = current_buffer; */
	      new->left = left;
	      new->bump = (pos - left_pos) + (left ? left->bump : 0);
	      new->instances = 1;
	      if (left)
		left->parent = new;
	      interval->left = new;
	      new->parent = interval;
	    }
	  else
	    {
	      *new_int = interval->left;
	      interval->left->instances++;
	    }
	  break;
	}
    }
  return interval;
}

struct Lisp_Interval *
splay_interval_left (interval)
     struct Lisp_Interval *interval;
{
  struct Lisp_Interval *left, *left_right;
  int left_right_bump;

/*  printf("Splay left\n"); */

  left = interval->left;
  left_right = left ? left->right : 0;
  left_right_bump = left_right ? left_right->bump : 0;

  interval->bump = (interval->bump - (left ? left->bump : 0))
    + left_right_bump;
  left->bump = ((left ? left->bump : 0) - left_right_bump)
    + interval->bump;
  left->right = interval;
  left->parent = interval->parent;
  interval->parent = left;
  interval->left = left_right;
  if (left_right)
    left_right->parent = interval;

  return left;
}

struct Lisp_Interval *
splay_interval_right (interval)
     struct Lisp_Interval *interval;
{
  struct Lisp_Interval *right, *right_left;
  int right_left_bump;

/*  printf("Splay right\n"); */
  
  right = interval->right;
  right_left = right ? right->left : 0;
  right_left_bump = right_left ? right_left_bump : 0;

  interval->bump = (interval->bump - (right ? right->bump : 0))
    + right_left_bump;
  right->bump = ((right ? right->bump : 0) - right_left_bump)
    + interval->bump;
  right->left = interval;
  right->parent = interval->parent;
  interval->parent = right;
  interval->right = right_left;
  if (right_left)
    right_left->parent = interval;

  return right;
}
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* Balance tree starting at INTERVAL
 */
#define IMBALANCE_THRESH	100

struct Lisp_Interval *
balance_interval_tree(interval)
     struct Lisp_Interval *interval;
{
  if (interval)
    {
      interval = balance_interval(interval);
      interval->left = balance_interval_tree(interval->left);
      interval->right = balance_interval_tree(interval->right);
      return interval;
    }
  else return interval;
}

struct Lisp_Interval *
balance_interval(interval)
     struct Lisp_Interval *interval;
{
  struct Lisp_Interval *left = interval ? interval->left : 0;
  struct Lisp_Interval *right = interval ? interval->right : 0;
  int left_bump = left ? left->bump : 0;
  int right_bump = right ? right->bump : 0;

  if (right_bump - left_bump > IMBALANCE_THRESH)
    {
      /* Right side heavy; splay right */
      return splay_interval_right(interval);
    }
  else if (left_bump - right_bump > IMBALANCE_THRESH)
    {
      /* Left side heavy; splay left */
      return splay_interval_left(interval);
    }
  else return interval;		/* No change */
}
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* Update functions for insertions and deletions of LENGTH at position POS.
 */
interval_insertion_update (pos,len)
     int pos,len;
{
/*  struct Lisp_Interval *interval = XINTERVAL(current_buffer->interval_tree); */
  struct Lisp_Interval *interval = interval_tree;
  int which,found = 0,start,left_pos = 0;
  int count = 0;

  while (!found)
    {
      which = interval_which_way(interval,pos,left_pos,&start);
      interval->bump += len;
      count++;
      switch (which)
	{
	case 1:
	  interval = interval->left;
	  break;
	case 2:
	  left_pos = start;
	  interval = interval->right;
	  break;
	case 0:
	  left_pos = start;
	  found = 1;
	  break;
	default:
	  break;
	}
    }
  
  printf("Updated %d\n",count);
}

int del_num;

interval_deletion_update (pos,len)
     int pos,len;
{
/*  interval_deletion1(XINTERVAL(current_buffer->interval_tree),
		     pos,len,0); */

  del_num = 0;
  interval_deletion1(interval_tree,pos,len,0);
  printf("Updated %d\n",del_num);
}

interval_deletion1(interval,pos,len,left_pos)
     struct Lisp_Interval *interval;
     int pos,len,left_pos;
{
  int which,start;
  int delete_more = 0, found = 0;
  int len_remaining;
  int bump;

  if (!interval) return len;
  del_num++;
  which = interval_which_way(interval,pos,left_pos,&start);
  bump = interval->bump;

  if (which == 1)
    {
      found++;
      len_remaining = interval_deletion1(interval->left,
					 pos,len,left_pos);

      bump = (bump - (len - len_remaining));
      len = len_remaining;
      delete_more = (len > 0);
      start = pos;
    }

  if ((found && delete_more) ||
      (which == 0))
    {
      int amount = min(len,(interval_size(interval) - (pos - start)));

      found++;
      bump -= amount;
      len -= amount;
      delete_more = (len > 0);
      start = pos;
    }
  if ((found && delete_more) ||
      (which == 2))
    {
      len_remaining = interval_deletion1 (interval->right,
					  pos,len,start);
      bump -= (len - len_remaining);
      len = len_remaining;
    }
  interval->bump = bump;
  return len;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* DEBUGGING FUNCTIONS
 */
print_node(node)
     struct Lisp_Interval *node;
{
  if (node->left)
    print_node(node->left);
/*  printf("[Node size %d]\n",interval_size(node)); */
  printf("Number %d\n",node->num);
  if (node->right)
    print_node(node->right);
}

snapshot()
{
  printf("--------\n");
  printf("Junk end %d\n",interval_start(a0));
  printf("Marker1 end %d\n",interval_start(a1));
  printf("Marker2 end %d\n",interval_start(a2));
  printf("Marker3 end %d\n",interval_start(a3));
  printf("Marker4 end %d\n",interval_start(a4));
  printf("Marker5 end %d\n",interval_start(a5));
  printf("Marker6 end %d\n",interval_start(a6));
  printf("Marker7 end %d\n",interval_start(a7));
  printf("Marker8 end %d\n",interval_start(a8));
  printf("Marker9 end %d\n",interval_start(a9));
}

void
main()
{
  int start;

  interval_free_list = 0;
  
  interval_tree = get_interval_node();

  interval_tree->bump = 10000;
  a0 = interval_tree;

  a1 = split_interval_here(240);
  a2 = split_interval_here(120);
  a3 = split_interval_here(170);
  a4 = split_interval_here(20);
  a5 = split_interval_here(55);
  a6 = split_interval_here(767);
  a7 = split_interval_here(430);   
  a8 = split_interval_here(10);
  a9 = split_interval_here(10);

  snapshot();
  printf("Insert 50 chars at 10\n");
  interval_insertion_update(1,50);
  snapshot();

  printf("Deleted 50 chars at 10\n");
  interval_deletion_update(1,50);
  snapshot();

  printf("Balance \n");
  interval_tree = balance_interval_tree(interval_tree);
  snapshot();

  printf("Insert 50 chars at 10\n");
  interval_insertion_update(1,50);
  snapshot();
  
}
  

  
