/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *      Modified:  Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/* widget server prompt support */
/* $Source: /export/kaplan/stable/sun4.os4.1/cb-2.0/src/data/node118.text,v $ */

static char rcsid[] = "prompt.c $Revision: 1.1 $ $Date: 92/05/08 15:25:31 $ $State: Exp $ $Author: CBmgr $";

/* ------------------------------------------------------------------------ */
#include "header.h"
#include <Xm/PushBG.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>
#include <Xm/RowColumn.h>
/* ------------------------------------------------------------------------ */
extern void
  PromptRemove(), PromptUpdate(), PromptReply(), PromptRealized(),
  PromptTransmit(), PromptEditable();

extern t_generic_widget PromptHandle();

struct widget_class_struct PromptClassRecord =
{
  WS_PROMPT, "prompt",
  PromptHandle, PromptRemove, NULL, PromptUpdate, PromptReply,
  PromptRealized, PromptTransmit, PromptEditable,
} ;
/* ------------------------------------------------------------------------ */
typedef struct prompt_lock_struct
{
  t_sexp lock;				/* the lock */
  int armed;				/* currently armed flag */
} * prompt_lock_ptr;

typedef struct prompt_item_struct
{
  Widget widget;			/* the button */
  t_sexp id;				/* identifier */
  struct prompt_widget_struct *prompt;	/* the prompt object */
  t_sexp action;			/* action list */
  struct prompt_item_struct *next;	/* next button */
  int local;				/* local actions flag */
  int n_locks;				/* number of locks */
  int n_armed;				/* number of armed locks */
  prompt_lock_ptr lock;			/* array of locks */
} * t_prompt_item;
  
typedef struct prompt_widget_struct
{
  DECLARE_STANDARD_WIDGET_SLOTS;
  t_prompt_item items;			/* list of defined buttons */
  int n_items;				/* # of items */
  int item_width, item_height;		/* width and height of items */
  int total_width;			/* total width of items */
  Widget active;			/* active (default) item */
  char *mbus_header;			/* header for messages */
  t_sexp context;			/* shell context */
  char *tag;				/* tag for utterance */
  char mbus_wrap;			/* wrap the sexp in parens? */
} * t_prompt_widget;

/* ------------------------------------------------------------------------ */
void
PromptButtonCallback(w, item, data)
     Widget w;
     t_prompt_item item;
     XmAnyCallbackStruct *data;
{
  Act(item->prompt, item->action, item->local);
}
/* ------------------------------------------------------------------------ */
void
PromptItemArmCallback(w, item, data)
     Widget w;
     t_prompt_item item;
     XmAnyCallbackStruct *data;
{
  if (NULL != item->prompt->active)
    XtVaSetValues(item->prompt->active, XmNshowAsDefault, 0, NULL);
  XtVaSetValues(w, XmNshowAsDefault, 1, NULL);
  item->prompt->active = w;
  XtVaSetValues(w, XmNwidth, item->prompt->item_width, NULL);
}
/* ------------------------------------------------------------------------ */
/* A prompt button looks like ( id label &key action) */
static struct keyword_entry_struct pbkw[] =
{
  { ":action", NULL, KEYWORD_GET_SEXP },
  { ":local", (void*)KEYWORD_FALSE, KEYWORD_FLAG },
  { ":locks", NULL, KEYWORD_SEXP },
};

int
PromptAddButton(prompt, button, sexp)
     t_prompt_widget prompt;
     int button;			/* button # */
     t_sexp sexp;
{
  t_sexp id, s_label, s_locks;
  t_prompt_item item;
  char *label;
  XmString xm_label;
  int n;
  Arg argl[6];

  id = MB_CAR(sexp);
  s_label = MBnth(sexp, 1);
  if (!((MB_NAMEP(id) || MB_STRINGP(id)) && MB_STRINGP(s_label))) return 0;

  item = NEW_STRUCT(prompt_item_struct);
  item->id = MBduplicate(id);
  item->prompt = prompt;
  item->next = prompt->items;
  item->n_locks = item->n_armed = 0;
  item->lock = NULL;

  MBparse_keywords(MBnthcdr(sexp,2), pbkw, ARRAY_SIZE(pbkw));
  item->action = (t_sexp) pbkw[0].result;
  item->local = (t_keyword_flag)pbkw[1].result == KEYWORD_TRUE;
  s_locks = (t_sexp) pbkw[2].result;

  label = MBCstring(s_label);
  xm_label = CtoXmString(label);

  prompt->items = item;

  if (MB_CONSP(s_locks))
    {
      int i;

      /* get a count of the locks, and allocate the array */
      item->n_locks = MBlength(s_locks);
      item->lock = (prompt_lock_ptr) XtMalloc(item->n_locks * sizeof(struct prompt_lock_struct));
      for ( i = 0 ; MB_CONSP(s_locks) ; ++i, s_locks = MB_CDR(s_locks))
	{
	  item->lock[i].lock = MBduplicate(MB_CAR(s_locks));
	  item->lock[i].armed = 0;
	}
    }
  else if (MB_NAMEP(s_locks) || MB_STRINGP(s_locks))
    {
      item->n_locks = 1;
      item->lock = NEW_STRUCT(prompt_lock_struct);
      item->lock[0].lock = MBduplicate(s_locks);
      item->lock[0].armed = 0;
    }

  n = 0;
  XtSetArg(argl[n], XmNsensitive, item->n_armed == item->n_locks), ++n;
  XtSetArg(argl[n], XmNlabelString, xm_label), ++n;
  XtSetArg(argl[n], XmNdefaultButtonShadowThickness, 2), ++n;
  item->widget = XtCreateWidget("promptButton", xmPushButtonGadgetClass,
				prompt->widget, argl, n);
  XtAddCallback(item->widget, XmNactivateCallback, PromptButtonCallback, item);
  XtAddCallback(item->widget, XmNarmCallback, PromptItemArmCallback, item);
  XtFree(label); XtFree(xm_label);

  return 1;
}
/* ------------------------------------------------------------------------ */
void
PromptArmItem(item, lock, flag)
     t_prompt_item item;
     t_sexp lock;
     int flag;
{
  int i;
  Arg arg;

  /* normalize flag */
  flag = flag ? 1 : 0;

  /* special values */
  if (!MBcompare_Cstring(lock, ":all"))
    {
      for ( i = 0 ; i < item->n_locks ; ++i )
	item->lock[i].armed = flag;;
      item->n_armed = flag ? item->n_locks : 0;
      XtSetArg(arg, XmNsensitive, flag ? True : False);
      XtSetValues(item->widget, &arg, 1);
    }
  else /* Default cause, look for the lock */
    {
      for ( i = 0 ; i < item->n_locks ; ++i )
	{
	  if (MBequal(item->lock[i].lock, lock))
	    {
	      item->n_armed -= item->lock[i].armed; /* adjust count */
	      item->lock[i].armed = flag;
	      item->n_armed += item->lock[i].armed; /* adjust count again */
	      XtSetArg(arg, XmNsensitive, item->n_armed == item->n_locks);
	      XtSetValues(item->widget, &arg, 1);
	      break;
	    }
	}
    }
}
/* ------------------------------------------------------------------------ */
/* Free up memory from a prompt button */
void
PromptRemoveItems(item) t_prompt_item item;
{
  t_prompt_item next;
  int i;

  while (NULL != item)
    {
      next = item->next;
      MBfree(item->id);
      MBfree(item->action);
      for ( i = 0 ; i < item->n_locks ; ++i )
	MBfree(item->lock[i].lock);
      XtFree(item->lock);
      XtFree(item);
      item = next;
    }
}
/* ------------------------------------------------------------------------ */
void
PromptAdjustSpacing(prompt, width)
     t_prompt_widget prompt;
     int width;
{
  int spacing, margin;

  width -= prompt->total_width;
  spacing = width / (prompt->n_items + 1);
  width -= spacing * (prompt->n_items - 1);
  margin = width / 2;

  if (margin < 3) margin = 3;
  if (spacing < 3) spacing = 3;
  /*
  printf("Set spacing to %d, margin to %d\n", spacing, margin);
  */
  XtVaSetValues(prompt->widget, XmNspacing, spacing, XmNmarginWidth, margin,
		NULL);
}
/* ------------------------------------------------------------------------ */
void
PromptResizeHandler(w, prompt, event, f_continue)
     Widget w;
     t_prompt_widget prompt;
     XEvent *event;
     Boolean *f_continue;
{
  if (event->type == ConfigureNotify)
    PromptAdjustSpacing(prompt, event->xconfigure.width);
}
/* ------------------------------------------------------------------------ */
/* A prompt is just a set of buttons, meant to perform actions. In general
 * these actions will be of the nature of transmitting a message or clearing
 * out the other widgets in the shell.
 */

static struct keyword_entry_struct PromptKeywords[] =
{
  { ":header", NULL, KEYWORD_COOKED },
  { ":context", NULL, KEYWORD_GET_SEXP },
  { ":wrap", (void*)KEYWORD_TRUE, KEYWORD_FLAG },
  { ":tag", NULL, KEYWORD_RAW },
  { ":values", NULL, KEYWORD_SEXP},
};

t_generic_widget
PromptHandle(parent,sexp)
     t_generic_widget parent;
     t_sexp sexp;
{
  t_prompt_widget prompt;
  t_sexp s_buttons, s_id;
  t_prompt_item item;
  int n;
  Arg argl[10];
  int bcount = 0;			/* number of buttons used  */
  Dimension width, height;

  s_id = MBnth(sexp, 1);
  if (!(MB_NAMEP(s_id) || MB_STRINGP(s_id))) return NULL;

  prompt = NEW_STRUCT(prompt_widget_struct);
  prompt->type = WS_PROMPT;
  prompt->id = MBduplicate(s_id);
  prompt->items = NULL;
  prompt->n_items = 0;
  prompt->class = &PromptClassRecord;
  prompt->next = NULL;
  prompt->parent = parent;
  prompt->top = parent->top;
  prompt->total_width = prompt->item_width = prompt->item_height = 0;
  prompt->active = NULL;

  MBparse_keywords(MBnthcdr(sexp,3),
		   PromptKeywords, ARRAY_SIZE(PromptKeywords));
  prompt->mbus_header = (char *)PromptKeywords[0].result;
  prompt->context = (t_sexp)PromptKeywords[1].result;
  prompt->mbus_wrap = (t_keyword_flag)PromptKeywords[2].result == KEYWORD_TRUE;
  prompt->tag = (char *)PromptKeywords[3].result;
  prompt->values = ContextString((t_sexp)PromptKeywords[4].result);

  n = 0;
  XtSetArg(argl[n], XmNpacking, XmPACK_TIGHT), ++n;
  XtSetArg(argl[n], XmNorientation, XmHORIZONTAL), ++n;
  XtSetArg(argl[n], XmNentryAlignment, XmALIGNMENT_CENTER), ++n;
  XtSetArg(argl[n], XmNresizeHeight, True), ++n;
  XtSetArg(argl[n], XmNadjustLast, False), ++n;
  prompt->widget = XtCreateWidget("prompt", xmRowColumnWidgetClass,
				  parent->widget, argl, n);

  /* Step through the buttons, until we run out of buttons or list elements */
  for ( s_buttons = MBnth(sexp,2)
       ; MB_CONSP(s_buttons)
       ; s_buttons = MB_CDR(s_buttons)
       )
    {
      bcount += PromptAddButton(prompt, bcount, MBcar(s_buttons));
    }

  if (bcount <= 0)
    {
      XtDestroyWidget(prompt->widget);
      return NULL;		/* no buttons, no prompt. */
    }

  prompt->n_items = bcount;

  /* Manage the items and find out how big they are */
  for ( item = prompt->items ; NULL != item ; item = item->next )
    {
      XtVaGetValues(item->widget, XmNwidth, &width, XmNheight, &height, NULL);
      if (width > prompt->item_width) prompt->item_width = width;
      if (height > prompt->item_height) prompt->item_height = height;
    }

  /* change the sizes */
  /* This doesn't actually seem to set anything, for some reason */
  for ( item = prompt->items ; NULL != item ; item = item->next )
    {
      XtVaSetValues(item->widget,
		    XmNwidth, (Dimension)prompt->item_width,
		    XmNheight, (Dimension)prompt->item_height,
		    XmNrecomputeSize, False,
		    NULL);
      XtManageChild(item->widget);
    }

  prompt->total_width = prompt->n_items * prompt->item_width;
  XtVaGetValues(prompt->widget, XmNwidth, &width, NULL);
  PromptAdjustSpacing(prompt, width);
  XtAddEventHandler(prompt->widget, StructureNotifyMask, False,
		    PromptResizeHandler, prompt);

  return (t_generic_widget)prompt;
}
/* ------------------------------------------------------------------------ */
void
PromptRemove(self) t_prompt_widget self;
{
  PromptRemoveItems(self->items);
  XtFree(self->tag);
  XtFree(self->mbus_header);
  XtFree(self->values);
  MBfree(self->context);
  XtFree(self);
}
/* ------------------------------------------------------------------------ */
/* Accept an update message and update the prompt */

/* Keywords for updating a prompt button */
static struct keyword_entry_struct piukw[] =
{
  { ":arm", NULL, KEYWORD_SEXP },
  { ":disarm", NULL, KEYWORD_SEXP },
} ;
void
PromptItemUpdate(self,sexp)
     t_prompt_item self;
     t_sexp sexp;
{
  t_sexp s_arm, s_disarm;

  if (!MBequal(MBnth(sexp,1), self->id)) return;

  MBparse_keywords(MBnthcdr(sexp, 2), piukw, ARRAY_SIZE(piukw));
  s_arm = (t_sexp)piukw[0].result;
  s_disarm = (t_sexp)piukw[1].result;

  while (MB_CONSP(s_arm))
    {
      PromptArmItem(self, MB_CAR(s_arm), 1);
      s_arm = MB_CDR(s_arm);
    }
  if (NULL != s_arm) PromptArmItem(self, s_arm, 1);

  while (MB_CONSP(s_disarm))
    {
      PromptArmItem(self, MB_CAR(s_disarm), 0);
      s_disarm = MB_CDR(s_disarm);
    }
  if (NULL != s_disarm) PromptArmItem(self, s_disarm, 0);
}

/* keywords for the prompt itself */
static struct keyword_entry_struct pukw[] =
{
  { ":label", NULL, KEYWORD_COOKED },
  { ":tag", NULL, KEYWORD_RAW },
  { ":values", NULL, KEYWORD_RAW },
};

void
PromptUpdate(self,sexp)
     t_prompt_widget self;
     t_sexp sexp;
{
  t_prompt_item item;

  /* Pass it on to all over our prompt items (buttons) */
  for (item = self->items ; NULL != item ; item = item->next )
    PromptItemUpdate(item, sexp);
}
/* ------------------------------------------------------------------------ */
void
PromptReply(self, reply)
     t_prompt_widget self;
     t_mbus_reply reply;
{
}
/* ------------------------------------------------------------------------ */
void
PromptRealized(self) t_prompt_widget self;
{
  Dimension width;
  XtVaGetValues(self->widget, XmNwidth, &width, NULL);
  PromptAdjustSpacing(self, width);
}
/* ------------------------------------------------------------------------ */
void
PromptTransmit(message, orig_reply, self)
  t_sexp message;
  t_mbus_reply orig_reply;
  t_prompt_widget self;
{
  if (NULL != self->values) MBput_Cstring(message, self->values);
}
/* ------------------------------------------------------------------------ */
void
PromptEditable(self, flag)
     t_prompt_widget self;
     int flag;
{
}
/* ------------------------------------------------------------------------ */
