/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*	socket.c: socket communications functions. */
/* Applications programmer interface routines */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/lib/RCS/socket.c,v $ */

static char rcsid[] = "api.c $Revision: 2.1.1.1 $ $Date: 91/11/15 13:35:31 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include <stdio.h>
#include "config.h"
#include "api.h"
#include "mbus.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/errno.h>

#if POLL
#include <stropts.h>
#include <poll.h>
#else
#include <sys/time.h>
#endif
/* ------------------------------------------------------------------------- */
/* t_mb_connection is defined in the header file already */
struct mb_connection_struct
{
  int fd;				/* file descriptor */
  int flags;
  struct sockaddr_in addr;
  struct mb_object *in;			/* data read from the buss */
  struct mb_object *out;		/* data to write to the buss */
  t_sexp message;			/* last message received */
  struct mb_parse_state state;
} ;

#define CONN_BLOCKING 0x1		/* connection is set to be blocking */

/* ------------------------------------------------------------------------- */
void (*mb_lost_connection_function)() = MBLostConnection;
extern int errno;

#ifdef EWOULDBLOCK
#define IO_ERROR (errno != EAGAIN && errno != EWOULDBLOCK && errno != EINTR)
#else
#define IO_ERROR (errno != EAGAIN && errno != EINTR)
#endif
/* ------------------------------------------------------------------------- */

/* This is the application's programmer interface. */
t_mb_connection
MBConnect(host,port)
     char *host;
     int port;
{
  int flags;				/* for setting NO_DELAY */
  t_mb_connection conn;
  struct hostent *h;
  char hostname[64];

  if (port <= 0) port = 0x956;		/* magic number! */
  if (NULL == host || 0 == *host)
    {
      host = hostname;
      if (0 > gethostname(host,64))
	{
	  if (MBLogLevel) perror("Can't get host name");
	  return NULL;
	}
    }

  h = gethostbyname(host);
  if (NULL == h)
    {
      if (MBLogLevel)
	{
	  char message[256];
	  sprintf(message,"Can't find host (%s)",host,port);
	  perror(message);
	}
      return NULL;
    }

  conn = (t_mb_connection) MBmalloc_func(sizeof (struct mb_connection_struct));
  if (NULL == conn) return NULL;
  conn->flags = 0;		/* clear all flags */

  conn->fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (conn->fd < 0)
    {
      if (MBLogLevel)
	{
	  char message[256];
	  sprintf(message,"MBus Connection (%s,%d) failed",host,port);
	  perror(message);
	}
      MBfree_func(conn);
      return NULL;
    }

  conn->addr.sin_family = AF_INET;

  memcpy((char *) &(conn->addr.sin_addr), h->h_addr_list[0], h->h_length);
  conn->addr.sin_port = htons(port);

  if (connect(conn->fd, (struct sockaddr *) &(conn->addr), sizeof(conn->addr)))
    {
      if (MBLogLevel)
	{
	  char message[256];
	  sprintf(message,"Cannot connect to Mbus at (%s,%d)",host,port);
	  perror(message);
	}
      MBfree_func(conn);
      return NULL;
    }

  /* Make the socket non-blocking */
  fcntl(conn->fd, F_GETFL, &flags);
  flags |= O_NDELAY;
  fcntl(conn->fd, F_SETFL, flags);

  MBInitializeParseState(&(conn->state));
  conn->in = MBGetName();
  conn->out = MBGetName();
  conn->message = NULL;			/* no message */
  return conn;
}
/* ------------------------------------------------------------------------- */
/* Control the blocking / non-blocking of the connection */
void
MBSetConnectionBlocking(conn,flag)
     t_mb_connection conn;
     int flag;
{
  int flags;
  fcntl(conn->fd, F_GETFL, &flags);
  if (flag)
    {
      flags &= ~O_NDELAY;	/* make blocking */
      conn->flags |= CONN_BLOCKING;
    }
  else
    {
      flags |= O_NDELAY;	/* make non-blocking */
      conn->flags &= ~CONN_BLOCKING;
    }
  fcntl(conn->fd, F_SETFL, flags);
}
/* ------------------------------------------------------------------------- */
void
MBClose(conn) t_mb_connection conn;
{
  close(conn->fd);
  MBChunkEmpty(&(conn->in->object.chunk));
  MBChunkEmpty(&(conn->out->object.chunk));
  MBFreeObject(conn->in);
  MBFreeObject(conn->out);
  MBfree(conn->state.base.object.cons.car);
  MBfree(conn->message);
  MBfree_func(conn);
}
/* ------------------------------------------------------------------------- */
/* See if there's data waiting in the socket */
int
MBcheck_connection(conn) t_mb_connection conn;
{
  int read_ok = 0;
#if POLL
  struct pollfd p_fd;
#else /* not POLL, use SELECT */
  fd_set read_fds;
  struct timeval timer;
#endif

  /* Check to see if there's data waiting. We do this in order to detect
   * a lost connection to the MBus
   */
#if POLL
  p_fd.fd = conn->fd;
  p_fd.events = POLLIN;
  read_ok = poll(&p_fd, 1, conn->flags & CONN_BLOCKING ? -1 : 0) > 0
    && p_fd.revents & (POLLIN | POLLHUP);
#else
  FD_ZERO(&read_fds);
  FD_SET(conn->fd,&read_fds);
  timer.tv_sec = timer.tv_usec = 0;
  read_ok =
    select(conn->fd+1, &read_fds, NULL, NULL,
	   conn->flags & CONN_BLOCKING ? NULL : &timer) > 0
      && FD_ISSET(conn->fd,&read_fds);
#endif

  return read_ok;
}
/* ------------------------------------------------------------------------- */
/* Read the connection socket. Block if the connection is set blocking.
 * Return the # of characters read
 */
int
MBread_connection(conn) t_mb_connection conn;
{
  int result = 0;

  if (MBcheck_connection(conn))
    {
      result = MBChunkRead(conn->fd, &(conn->in->object.chunk));
      if (result < 0 && IO_ERROR || result == 0)
	mb_lost_connection_function(conn);
    }
  return result;
}
/* ------------------------------------------------------------------------- */
/* Try to get a message from the connection. If a
 * complete message, then return the sexp from the state. Otherwise return
 * NULL
 */
struct mb_object *
MBparse_connection(conn) t_mb_connection conn;
{
  struct mb_object *val = NULL;

  if (MBParseChunk(&(conn->in->object.chunk),&(conn->state)))
    val = MBExtractSexp(&(conn->state));

  return val;
}
/* ------------------------------------------------------------------------- */
struct mb_object *
MBNextMessage(conn) t_mb_connection conn;
{
  struct mb_object *val = NULL;

  if (NULL == (val = MBparse_connection(conn)))
    {
      MBread_connection(conn);
      val = MBparse_connection(conn);
    }
  return val;
}
/* ------------------------------------------------------------------------- */
/* bus_write just writes what it can and return immediately. */
int
MBbus_write_buffer(conn, buff, count)
     t_mb_connection conn;
     char *buff;
     int count;
{
  count =
    (NULL == conn || NULL == buff || count < 1)
      ? 0
	: write(conn->fd, buff, count);
  if (count < 0 && IO_ERROR) mb_lost_connection_function(conn);
  return count;
}
/* ------------------------------------------------------------------------- */
/* Transmit doesn't return until the entire buffer has been sent, or an
 * error occurs.
 */
int
MBtransmit_buffer(conn, buff, count)
     t_mb_connection conn;
     char *buff;
     int count;
{
  int n;				/* number of bytes actually written */
  int write_ok;				/* ok to write on the socket */
  int initial = count;			/* initial number of byte to write */
#if POLL
  struct pollfd p_fd;
#else /* not POLL, use SELECT */
  fd_set write_fds;
#endif

  if (NULL == conn) return 0;		/* no connection, just return */

  while (count > 0)
    {
      n = 0;
#if POLL
      p_fd.fd = conn->fd;
      p_fd.events = POLLOUT;
      write_ok = poll(&p_fd, 1, -1) > 0 && p_fd.revents & (POLLOUT | POLLHUP);
#else
      FD_ZERO(&write_fds);
      FD_SET(conn->fd,&write_fds);
      write_ok = select(conn->fd+1, NULL, &write_fds, NULL, NULL) > 0
	&& FD_ISSET(conn->fd,&write_fds);
#endif
      if (write_ok)
	{
	  n = MBbus_write_buffer(conn, buff, count);
	  if (0 == n)
	    {
	      mb_lost_connection_function(conn);
	      break;			/* abort send */
	    }
	  else if (n > 0)
	    {
	      count -= n;
	      buff += n;
	    }
	}
      else if (IO_ERROR)		/* problem, let's get out of here */
	{
	  mb_lost_connection_function(conn);
	  break;
	}
    }
  return initial - count;		/* bytes actually written */
}
/* ------------------------------------------------------------------------- */
/* Just a convience function */
int
MBtransmit_Cstring(conn, str)
     t_mb_connection conn;
     char *str;
{
  return (NULL != str && NULL != conn)
    ? MBtransmit_buffer(conn, str, strlen(str))
      : 0;
}
/* ------------------------------------------------------------------------- */
void
MBLostConnection(conn) t_mb_connection conn;
{
  fprintf(stderr,"Lost connection to MBus\n");
  exit(1);
}
/* ------------------------------------------------------------------------- */
int
MBconnection_fd(conn) t_mb_connection conn;
{
  return conn->fd;
}
/* ------------------------------------------------------------------------- */
