/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*	regexp.c: Regular expression matching */
/* Regular expression matching */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/lib/RCS/regexp.c,v $ */

static char rcsid[] = "$Revision: 2.1.1.1 $ $Date: 91/11/15 13:35:27 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
/* This code is a heavily modified version of a public domain regular
 * expression matcher. The following is the original copyright notice
 */
/*
 * regcomp and regexec -- regsub and regerror are elsewhere
 * @(#)regexp.c	1.3 of 18 April 87
 *
 *	Copyright (c) 1986 by University of Toronto.
 *	Written by Henry Spencer.  Not derived from licensed software.
 *
 *	Permission is granted to anyone to use this software for any
 *	purpose on any computer system, and to redistribute it freely,
 *	subject to the following restrictions:
 *
 *	1. The author is not responsible for the consequences of use of
 *		this software, no matter how awful, even if they arise
 *		from defects in it.
 *
 *	2. The origin of this software must not be misrepresented, either
 *		by explicit claim or by omission.
 *
 *	3. Altered versions must be plainly marked as such, and must not
 *		be misrepresented as being the original software.
 *
 * Beware that some of this code is subtly aware of the way operator
 * precedence is structured in regular expressions.  Serious changes in
 * regular-expression syntax might require a total rethink.
 */
/* End of original copyright notice */
#include <stdio.h>
#include "mbus.h"
#include "api.h"
#ifdef NeXT
#include <string.h>
#endif

/*
 * The "internal use only" fields in regexp.h are present to pass info from
 * compile to execute that permits the execute phase to run lots faster on
 * simple cases.  They are:
 *
 * regstart	char that must begin a match; '\0' if none obvious
 * reganch	is the match anchored (at beginning-of-line only)?
 * regmust	string (pointer into program) that match must include, or NULL
 * regmlen	length of regmust string
 *
 * Regstart and reganch permit very fast decisions on suitable starting points
 * for a match, cutting down the work a lot.  Regmust permits fast rejection
 * of lines that cannot possibly match.  The regmust tests are costly enough
 * that regcomp() supplies a regmust only if the r.e. contains something
 * potentially expensive (at present, the only such thing detected is * or +
 * at the start of the r.e., which can involve a lot of backup).  Regmlen is
 * supplied because the test in regexec() needs it and regcomp() is computing
 * it anyway.
 */

/*
 * Structure for regexp "program".  This is essentially a linear encoding
 * of a nondeterministic finite-state machine (aka syntax charts or
 * "railroad normal form" in parsing technology).  Each node is an opcode
 * plus a "next" pointer, possibly plus an operand.  "Next" pointers of
 * all nodes except BRANCH implement concatenation; a "next" pointer with
 * a BRANCH on both ends of it is connecting two alternatives.  (Here we
 * have one of the subtle syntax dependencies:  an individual BRANCH (as
 * opposed to a collection of them) is never concatenated with anything
 * because of operator precedence.)  The operand of some types of node is
 * a literal string; for others, it is a node leading into a sub-FSM.  In
 * particular, the operand of a BRANCH node is the first node of the branch.
 * (NB this is *not* a tree structure:  the tail of the branch connects
 * to the thing following the set of BRANCHes.)  The opcodes are:
 */

/* definition	number	opnd?	meaning */
#define	END	0	/* no	End of program. */
#define	BOL	1	/* no	Match "" at beginning of line. */
#define	EOL	2	/* no	Match "" at end of line. */
#define	ANY	3	/* no	Match any one character. */
#define	ANYOF	4	/* str	Match any character in this string. */
#define	ANYBUT	5	/* str	Match any character not in this string. */
#define	BRANCH	6	/* node	Match this alternative, or the next... */
#define	BACK	7	/* no	Match "", "next" ptr points backward. */
#define	EXACTLY	8	/* str	Match this string. */
#define	NOTHING	9	/* no	Match empty string. */
#define	STAR	10	/* node	Match this (simple) thing 0 or more times. */
#define	PLUS	11	/* node	Match this (simple) thing 1 or more times. */
#define	OPEN	20	/* no	Mark this point in input as start of #n. */
			/*	OPEN+1 is number 1, etc. */
#define	CLOSE	30	/* no	Analogous to OPEN. */

/*
 * Opcode notes:
 *
 * BRANCH	The set of branches constituting a single choice are hooked
 *		together with their "next" pointers, since precedence prevents
 *		anything being concatenated to any individual branch.  The
 *		"next" pointer of the last BRANCH in a choice points to the
 *		thing following the whole choice.  This is also where the
 *		final "next" pointer of each individual branch points; each
 *		branch starts with the operand node of a BRANCH node.
 *
 * BACK		Normal "next" pointers all implicitly point forward; BACK
 *		exists to make loop structures possible.
 *
 * STAR,PLUS	'?', and complex '*' and '+', are implemented as circular
 *		BRANCH structures using BACK.  Simple cases (one character
 *		per match) are implemented with STAR and PLUS for speed
 *		and to minimize recursive plunges.
 *
 * OPEN,CLOSE	...are numbered at compile time.
 */

/*
 * A node is one char of opcode followed by two chars of "next" pointer.
 * "Next" pointers are stored as two 8-bit pieces, high order first.  The
 * value is a positive offset from the opcode of the node containing it.
 * An operand, if any, simply follows the node.  (Note that much of the
 * code generation knows about this implicit relationship.)
 *
 * Using two bytes for the "next" pointer is vast overkill for most things,
 * but allows patterns to get big without disasters.
 */
#define	OP(p)	(*(p))
#define	NEXT(p)	(((*((p)+1)&0377)<<8) + (*((p)+2)&0377))
#define	OPERAND(p)	((p) + 3)

/*
 * See regmagic.h for one further detail of program structure.
 */


/*
 * Utility definitions.
 */
#ifndef CHARBITS
#define	UCHARAT(p)	((int)*(unsigned char *)(p))
#else
#define	UCHARAT(p)	((int)*(p)&CHARBITS)
#endif

#define	FAIL(m)	{ regerror(m); return(NULL); }
#define	ISMULT(c)	((c) == '*' || (c) == '+' || (c) == '?')

/*
 * Flags to be passed up and down.
 */
#define	HASWIDTH	01	/* Known never to match null string. */
#define	SIMPLE		02	/* Simple enough to be STAR/PLUS operand. */
#define	SPSTART		04	/* Starts with * or +. */
#define	WORST		0	/* Worst case. */

/*
 * Global work variables for regcomp().
 */
static struct mb_chunk_pointer regparse; /* Input-scan pointer. */
static t_block_char regdummy;
static t_block_char *regcode;		/* Code-emit pointer; &regdummy = don't. */
static long regsize;		/* Code size. */

/*
 * Forward declarations for regcomp()'s friends.
 */
#ifndef STATIC
#define	STATIC	static
#endif
STATIC t_block_char *reg();
STATIC t_block_char *regbranch();
STATIC t_block_char *regpiece();
STATIC t_block_char *regatom();
STATIC t_block_char *regnode();
STATIC t_block_char *regnext();
STATIC void regc();
STATIC void reginsert();
STATIC void regtail();
STATIC void regoptail();
#ifdef STRCSPN
STATIC int strcspn();
#endif

void
regerror(s)
{
  extern int MBLogLevel;

  if (MBLogLevel) printf("%s\n",s);
}

/*
 - regcomp - compile a regular expression into internal code
 *
 * We can't allocate space until we know how big the compiled form will be,
 * but we can't compile it (and thus know how big it is) until we've got a
 * place to put the code.  So we cheat:  we compile it twice, once with code
 * generation turned off and size counting turned on, and once "for real".
 * This also means that we don't allocate space until we are sure that the
 * thing really will compile successfully, and we never have to move the
 * code and thus invalidate pointers into it.  (Note that it has to be in
 * one piece because free() must be able to free it all.)
 *
 * Beware that the optimization-preparation code in here knows about some
 * of the structure of the compiled regexp.
 */
struct mb_object *
MBRegexpCompile(exp)
struct mb_object *exp;
{
  struct mb_object *val;
  t_mb_block regblock;
  struct mb_regexp *r;
  register t_block_char *scan;
  register t_block_char *longest;
  register int len;
  int flags;

  if (NULL == exp) return NULL;

  /* based on the type, set up the input pointer */
  switch (exp->type)
    {
    case MB_NAME:
    case MB_STRING:
    case MB_CHUNK_POINTER:
      MBCPSet(&regparse,exp);
      break;
    default:
      return NULL;
    }

  /* First pass: determine size, legality. */
  regsize = 0L;
  regcode = &regdummy;
  regc(MB_REGEXP_MAGIC);
  if (reg(0, &flags) == NULL) return NULL;

  /* Small enough for pointer-storage convention? */
  if (regsize >= 32767L)		/* Probably could be 65535L. */
    return NULL;

  /* Allocate space. */
  regblock = MBGetBlock(regsize);
  if (NULL == regblock) return NULL;
  val = MBGetObject();
  val->type = MB_REGEXP;
  val->object.re.program = regblock;
  r = &(val->object.re);

  /* Second pass: emit code. */
  MBCPSet(&regparse,exp);
  regcode = r->program->data;
  regc(MB_REGEXP_MAGIC);
  if (reg(0, &flags) == NULL)
    {
      MBFreeBlock(regblock);
      MBFreeObject(val);
      return NULL;
    }

  /* remember the size */
  regblock->count = regsize;

  /* Dig out information for optimizations. */
  r->regstart = '\0';			/* Worst-case defaults. */
  r->reganch = 0;
  r->regmust = NULL;
  r->regmlen = 0;
  scan = &(r->program->data[1]);	/* First BRANCH. */

  if (OP(regnext(scan)) == END) {	/* Only one top-level choice. */
    scan = OPERAND(scan);

    /* Starting-point info. */
    if (OP(scan) == EXACTLY)
      r->regstart = *OPERAND(scan);
    else if (OP(scan) == BOL)
      r->reganch++;

    /*
     * If there's something expensive in the r.e., find the
     * longest literal string that must appear and make it the
     * regmust.  Resolve ties in favor of later strings, since
     * the regstart check works with the beginning of the r.e.
     * and avoiding duplication strengthens checking.  Not a
     * strong reason, but sufficient in the absence of others.
     */
    if (flags&SPSTART) {
      longest = NULL;
      len = 0;
      for (; scan != NULL; scan = regnext(scan))
	if (OP(scan) == EXACTLY && strlen((char *)OPERAND(scan)) >= len) {
	  longest = OPERAND(scan);
	  len = strlen((char *)OPERAND(scan));
	}
      r->regmust = longest;
      r->regmlen = len;
    }
  }

  return val;
}

/*
 - reg - regular expression, i.e. main body or parenthesized thing
 *
 * Caller must absorb opening parenthesis.
 *
 * Combining parenthesis handling with the base level of regular expression
 * is a trifle forced, but the need to tie the tails of the branches to what
 * follows makes it hard to avoid.
 */
static t_block_char *
reg(paren, flagp)
int paren;			/* Parenthesized? */
int *flagp;
{
	register t_block_char *ret;
	register t_block_char *br;
	register t_block_char *ender;
	register int parno;
	int flags;

	*flagp = HASWIDTH;	/* Tentatively. */

	/* Make an OPEN node, if parenthesized. */
	if (paren) {
		ret = regnode(OPEN);
	} else
		ret = NULL;

	/* Pick up the branches, linking them together. */
	br = regbranch(&flags);
	if (br == NULL)
		return(NULL);
	if (ret != NULL)
		regtail(ret, br);	/* OPEN -> first. */
	else
		ret = br;
	if (!(flags&HASWIDTH))
		*flagp &= ~HASWIDTH;
	*flagp |= flags&SPSTART;
	while (regparse.c == '|') {
		MBCPGetChar(&regparse);
		br = regbranch(&flags);
		if (br == NULL)
			return(NULL);
		regtail(ret, br);	/* BRANCH -> BRANCH. */
		if (!(flags&HASWIDTH))
			*flagp &= ~HASWIDTH;
		*flagp |= flags&SPSTART;
	}

	/* Make a closing node, and hook it on the end. */
	ender = regnode((paren) ? CLOSE : END);	
	regtail(ret, ender);

	/* Hook the tails of the branches to the closing node. */
	for (br = ret; br != NULL; br = regnext(br))
		regoptail(br, ender);

	/* Check for proper termination. */
	if (paren && MBCPGetChar(&regparse) != ')') {
		FAIL("unmatched ()");
	} else if (!paren && regparse.c != CHUNK_EMPTY) {
		if (regparse.c == ')') {
			FAIL("unmatched ()");
		} else
			FAIL("junk on end");	/* "Can't happen". */
		/* NOTREACHED */
	}

	return(ret);
}

/*
 - regbranch - one alternative of an | operator
 *
 * Implements the concatenation operator.
 */
static t_block_char *
regbranch(flagp)
int *flagp;
{
	register t_block_char *ret;
	register t_block_char *chain;
	register t_block_char *latest;
	int flags;

	*flagp = WORST;		/* Tentatively. */

	ret = regnode(BRANCH);
	chain = NULL;
	while (regparse.c != CHUNK_EMPTY && regparse.c != '|' && regparse.c != ')') {
		latest = regpiece(&flags);
		if (latest == NULL)
			return(NULL);
		*flagp |= flags&HASWIDTH;
		if (chain == NULL)	/* First piece. */
			*flagp |= flags&SPSTART;
		else
			regtail(chain, latest);
		chain = latest;
	}
	if (chain == NULL)	/* Loop ran zero times. */
		(void) regnode(NOTHING);

	return(ret);
}

/*
 - regpiece - something followed by possible [*+?]
 *
 * Note that the branching code sequences used for ? and the general cases
 * of * and + are somewhat optimized:  they use the same NOTHING node as
 * both the endmarker for their branch list and the body of the last branch.
 * It might seem that this node could be dispensed with entirely, but the
 * endmarker role is not redundant.
 */
static t_block_char *
regpiece(flagp)
int *flagp;
{
	register t_block_char *ret;
	register t_block_char op;
	register t_block_char *next;
	int flags;

	ret = regatom(&flags);
	if (ret == NULL)
		return(NULL);

	op = regparse.c;
	if (!ISMULT(op)) {
		*flagp = flags;
		return(ret);
	}

	if (!(flags&HASWIDTH) && op != '?')
		FAIL("*+ operand could be empty");
	*flagp = (op != '+') ? (WORST|SPSTART) : (WORST|HASWIDTH);

	if (op == '*' && (flags&SIMPLE))
		reginsert(STAR, ret);
	else if (op == '*') {
		/* Emit x* as (x&|), where & means "self". */
		reginsert(BRANCH, ret);			/* Either x */
		regoptail(ret, regnode(BACK));		/* and loop */
		regoptail(ret, ret);			/* back */
		regtail(ret, regnode(BRANCH));		/* or */
		regtail(ret, regnode(NOTHING));		/* null. */
	} else if (op == '+' && (flags&SIMPLE))
		reginsert(PLUS, ret);
	else if (op == '+') {
		/* Emit x+ as x(&|), where & means "self". */
		next = regnode(BRANCH);			/* Either */
		regtail(ret, next);
		regtail(regnode(BACK), ret);		/* loop back */
		regtail(next, regnode(BRANCH));		/* or */
		regtail(ret, regnode(NOTHING));		/* null. */
	} else if (op == '?') {
		/* Emit x? as (x|) */
		reginsert(BRANCH, ret);			/* Either x */
		regtail(ret, regnode(BRANCH));		/* or */
		next = regnode(NOTHING);		/* null. */
		regtail(ret, next);
		regoptail(ret, next);
	}
	MBCPGetChar(&regparse);
	if (ISMULT(regparse.c))
		FAIL("nested *?+");

	return(ret);
}

/*
 - regatom - the lowest level
 *
 * Optimization:  gobbles an entire sequence of ordinary characters so that
 * it can turn them into a single node, which is smaller to store and
 * faster to run.  Backslashed characters are exceptions, each becoming a
 * separate node; the code is simpler that way and it's not worth fixing.
 */
static t_block_char *
regatom(flagp)
int *flagp;
{
  register t_block_char *ret;
  int prevc;
  int flags;

  *flagp = WORST;			/* Tentatively. */

  switch (prevc = MBCPGetChar(&regparse))
    {
    case '^':
      ret = regnode(BOL);
      break;
    case '$':
      ret = regnode(EOL);
      break;
    case '.':
      ret = regnode(ANY);
      *flagp |= HASWIDTH|SIMPLE;
      break;
    case '[':
      {
	register int class;
	register int classend;

	if (regparse.c == '^')		/* Complement of range. */
	  {
	    ret = regnode(ANYBUT);
	    MBCPGetChar(&regparse);
	  }
	else
	  ret = regnode(ANYOF);

	class = regparse.c;		/* save this */

	if (regparse.c == ']' || regparse.c == '-')
	  regc(MBCPGetChar(&regparse));

	while (regparse.c != CHUNK_EMPTY && regparse.c != ']')
	  {
	    if (regparse.c == '-')
	      {
		MBCPGetChar(&regparse);
		if (regparse.c == ']' || regparse.c == CHUNK_EMPTY)
		  regc('-');
		else
		  {
		    classend = regparse.c;
		    if (class > classend+1)
		      FAIL("invalid [] range");
		    for (; class <= classend; class++)
		      regc(class);
		    class = regparse.c;	/* for next iteration */
		    MBCPGetChar(&regparse);
		  }
	      }
	    else
	      {
		class = regparse.c;
		regc(MBCPGetChar(&regparse));
	      }
	  }

	regc('\0');			/* close it off */
	if (regparse.c != ']') FAIL("unmatched []");
	MBCPGetChar(&regparse);
	*flagp |= HASWIDTH|SIMPLE;
      }
      break;
    case '(':
      ret = reg(1, &flags);
      if (ret == NULL)
	return(NULL);
      *flagp |= flags&(HASWIDTH|SPSTART);
      break;
    case CHUNK_EMPTY:
    case '|':
    case ')':
      FAIL("internal urp");		/* Supposed to be caught earlier. */
      break;
    case '?':
    case '+':
    case '*':
      FAIL("?+* follows nothing");
      break;
    case '\\':
      if (regparse.c == CHUNK_EMPTY)
	FAIL("trailing \\");
      ret = regnode(EXACTLY);
      regc(MBCPGetChar(&regparse));
      regc('\0');
      *flagp |= HASWIDTH|SIMPLE;
      break;
    default:
      {
	register int len = 1;

	ret = regnode(EXACTLY);
	for (  regc(prevc)
	     ; regparse.c != CHUNK_EMPTY
	     && !MB_REGEXP_IS_META(regparse.c)
	     && !ISMULT(MBCPPeek(&regparse))
	     ; len += 1
	     )
	    regc(MBCPGetChar(&regparse));
	    
	regc('\0');
	*flagp |= HASWIDTH;
	if (1 == len) *flagp |= SIMPLE;
      }
      break;
    }

  return(ret);
}

/*
 - regnode - emit a node
 */
static t_block_char *			/* Location. */
regnode(op)
t_block_char op;
{
  register t_block_char *ret;
  register t_block_char *ptr;

  ret = regcode;
  if (ret == &regdummy)
    {
      regsize += 3;
      return(ret);
    }

  ptr = ret;
  *ptr++ = op;
  *ptr++ = '\0';			/* Null "next" pointer. */
  *ptr++ = '\0';
  regcode = ptr;

  return(ret);
}

/*
 - regc - emit (if appropriate) a byte of code
 */
static void
regc(b)
t_block_char b;
{
  if (regcode != &regdummy)
    *regcode++ = b;
  else
    regsize++;
}

/*
 - reginsert - insert an operator in front of already-emitted operand
 *
 * Means relocating the operand.
 */
static void
reginsert(op, opnd)
t_block_char op;
t_block_char *opnd;
{
  register t_block_char *src;
  register t_block_char *dst;
  register t_block_char *place;

  if (regcode == &regdummy)
    {
      regsize += 3;
      return;
    }

  src = regcode;
  regcode += 3;
  dst = regcode;
  while (src > opnd)
    *--dst = *--src;

  place = opnd;			   /* Op node, where operand used to be. */
  *place++ = op;
  *place++ = '\0';
  *place++ = '\0';
}

/*
 - regtail - set the next-pointer at the end of a node chain
 */
static void
regtail(p, val)
t_block_char *p;
t_block_char *val;
{
  register t_block_char *scan;
  register t_block_char *temp;
  register int offset;

  if (p == &regdummy)
    return;

  /* Find last node. */
  scan = p;
  for (;;) {
    temp = regnext(scan);
    if (temp == NULL)
      break;
    scan = temp;
  }

  if (OP(scan) == BACK)
    offset = scan - val;
  else
    offset = val - scan;
  *(scan+1) = (offset>>8)&0377;
  *(scan+2) = offset&0377;
}

/*
 - regoptail - regtail on operand of first argument; nop if operandless
 */
static void
regoptail(p, val)
t_block_char *p;
t_block_char *val;
{
  /* "Operandless" and "op != BRANCH" are synonymous in practice. */
  if (p == NULL || p == &regdummy || OP(p) != BRANCH)
    return;
  regtail(OPERAND(p), val);
}

/*
 * regexec and friends
 */
/*
 * Global work variables for regexec().
 */
static struct mb_chunk_pointer reginput;
static struct mb_chunk_pointer regbol;

/*
 * Forwards.
 */
STATIC int regtry();
STATIC int regmatch();
STATIC int regrepeat();

#ifdef DEBUG
int regnarrate = 0;
void regdump();
STATIC char *regprop();
#endif

/*
 - regexec - match a regexp against a string
 */
int
MBRegexpCompare(prog, obj)
     struct mb_regexp *prog;
     struct mb_object *obj;
{
  struct mb_chunk_pointer s;

  /* Be paranoid... */
  if (prog == NULL || obj == NULL)
    {
      regerror("NULL parameter");
      return(0);
    }
  if (obj->type == MB_STRING
      || obj->type == MB_NAME
      || obj->type == MB_CHUNK_POINTER)
    MBCPSet(&s, obj);
  else return 0;

  /* Check validity of program. */
  if (UCHARAT(prog->program->data) != MB_REGEXP_MAGIC)
    {
      regerror("corrupted program");
      return(0);
    }

#if 0
  /* fix this later */
  /* If there is a "must appear" string, look for it. */
  if (prog->regmust != NULL)
    {
      s = string;
      while ((s = strchr(s, prog->regmust[0])) != NULL) {
	if (strncmp(s, prog->regmust, prog->regmlen) == 0)
	  break;			/* Found it. */
	s++;
      }
      if (s == NULL)			/* Not present. */
	return(0);
    }
#endif

  /* set internal chunk pointer */
  MBCPSet(&s, obj);

  /* Mark beginning of line for ^ . */
  MBCPSet(&regbol, obj);

  /* Simplest case:  anchored match need be tried only once. */
  if (prog->reganch)
    return regtry(prog, &s);

  /* Messy cases:  unanchored match. */
  if (prog->regstart != '\0')
    /* We know what char it must start with. */
    while (MBCPAdvanceToChar(&s, prog->regstart))
      {
	if (regtry(prog, &s)) return 1;
	MBCPGetChar(&s);
      }
  else
    /* We don't -- general case. */
    do
      {
	if (regtry(prog, &s)) return(1);
      } while (MBCPGetChar(&s) != CHUNK_EMPTY);

  /* Failure. */
  return 0;
}

/*
 - regtry - try match at specific point
 */
static int			/* 0 failure, 1 success */
regtry(prog, string)
     struct mb_regexp *prog;
     struct mb_chunk_pointer *string;
{
  register int i;

  MBCPCopy(&reginput, string);

  return regmatch(prog->program->data + 1);
}

/*
 - regmatch - main matching routine
 *
 * Conceptually the strategy is simple:  check to see whether the current
 * node matches, call self recursively to see whether the rest matches,
 * and then act accordingly.  In practice we make some effort to avoid
 * recursion, in particular by going through "ordinary" nodes (that don't
 * need to know whether the rest of the match failed) by a loop instead of
 * by recursion.
 */
static int			/* 0 failure, 1 success */
regmatch(prog) t_block_char *prog;
{
  t_block_char *scan;			/* Current node. */
  t_block_char *next;			/* Next node. */

  scan = prog;
#ifdef DEBUG
  if (scan != NULL && regnarrate)
    fprintf(stderr, "%s(\n", regprop(scan));
#endif
  while (scan != NULL) {
#ifdef DEBUG
    if (regnarrate)
      fprintf(stderr, "%s...\n", regprop(scan));
#endif
    next = regnext(scan);

    switch (OP(scan))
      {
      case BOL:
	if (!MBCPEqual(&reginput, &regbol)) return 0;
	break;
      case EOL:
	if (reginput.c != CHUNK_EMPTY) return 0;
	break;
      case ANY:
	if (reginput.c == CHUNK_EMPTY) return 0;
	MBCPGetChar(&reginput);
	break;
      case EXACTLY:
	{
	  register int len;
	  register t_block_char *opnd;

	  opnd = OPERAND(scan);
	  /* Inline the first character, for speed. */
	  if (*opnd != reginput.c) return 0;
	  len = strlen((char *)opnd);
	  if (len > 1 && MBCPBufferCompare(&reginput, opnd, len) != 0)
	    return(0);
	  MBCPSkipChars(&reginput,len);
	}
	break;
      case ANYOF:
	if (reginput.c == CHUNK_EMPTY
	    || strchr((char *)OPERAND(scan), reginput.c) == NULL
	    )
	  return 0;
	MBCPGetChar(&reginput);
	break;
      case ANYBUT:
	if (reginput.c == CHUNK_EMPTY
	    || strchr((char *)OPERAND(scan), reginput.c) != NULL
	    )
	  return 0;
	MBCPGetChar(&reginput);
	break;
      case NOTHING:
	break;
      case BACK:
	break;
      case OPEN:
      case CLOSE:
	return regmatch(next);
      case BRANCH:
	{
	  struct mb_chunk_pointer save;

	  if (OP(next) != BRANCH)	/* No choice. */
	    next = OPERAND(scan);	/* Avoid recursion. */
	  else
	    {
	      do
		{
		  MBCPCopy(&save, &reginput);
		  if (regmatch(OPERAND(scan))) return(1);
		  MBCPCopy(&reginput, &save);
		  scan = regnext(scan);
		} while (scan != NULL && OP(scan) == BRANCH);
	      return(0);
	      /* NOTREACHED */
	    }
	}
	break;
      case STAR:
      case PLUS:
	{
	  register char nextch;
	  register int no;
	  struct mb_chunk_pointer save;
	  register int min;

	  /*
	   * Lookahead to avoid useless match attempts
	   * when we know what character comes next.
	   */
	  nextch = '\0';
	  if (OP(next) == EXACTLY) nextch = *OPERAND(next);
	  min = (OP(scan) == STAR) ? 0 : 1;
	  MBCPCopy(&save, &reginput);
	  no = regrepeat(OPERAND(scan));
	  while (no >= min) {
	    /* If it could work, try it. */
	    if ((nextch == '\0' || reginput.c == nextch) && regmatch(next))
	      return 1;
	    /* Couldn't or didn't -- back up. */
	    no--;
	    MBCPCopy(&reginput, &save);
	    MBCPSkipChars(&reginput, no);
	  }
	  return(0);
	}
	break;
      case END: return 1;
      default:
	regerror("memory corruption");
	return(0);
	break;
      }

    scan = next;
  }

  /*
   * We get here only if there's trouble -- normally "case END" is
   * the terminating point.
   */
  regerror("corrupted pointers");
  return(0);
}

/*
 - regrepeat - repeatedly match something simple, report how many
 */
static int
regrepeat(p)
char *p;
{
  register int count = 0;
  struct mb_chunk_pointer scan;
  register char *opnd;

  MBCPCopy(&scan, &reginput);
  opnd = OPERAND(p);
  switch (OP(p))
    {
    case ANY:
      count = MBCPLength(&scan);
      MBCPEmpty(&scan);
      break;
    case EXACTLY:
      while (*opnd == scan.c)
	{
	  count++;
	  MBCPGetChar(&scan);
	}
      break;
    case ANYOF:
      while (scan.c != CHUNK_EMPTY && strchr(opnd, scan.c) != NULL)
	{
	  count++;
	  MBCPGetChar(&scan);
	}
      break;
    case ANYBUT:
      while (scan.c != CHUNK_EMPTY && strchr(opnd, scan.c) == NULL)
	{
	  count++;
	  MBCPGetChar(&scan);
	}
      break;
    default:				/* Oh dear.  Called inappropriately. */
      regerror("internal foulup");
      count = 0;			/* Best compromise. */
      break;
    }

  MBCPCopy(&reginput, &scan);
  return(count);
}

/*
 - regnext - dig the "next" pointer out of a node
 */
static t_block_char *
regnext(p) t_block_char *p;
{
  register int offset;

  if (p == &regdummy) return(NULL);

  offset = NEXT(p);
  if (offset == 0) return(NULL);

  if (OP(p) == BACK) return(p-offset);
  else return(p+offset);
}

#ifdef DEBUG

STATIC char *regprop();

/*
 - regdump - dump a regexp onto stdout in vaguely comprehensible form
 */
void
regdump(r)
regexp *r;
{
  register char *s;
  register char op = EXACTLY;		/* Arbitrary non-END op. */
  register char *next;
  extern char *strchr();


  s = r->program + 1;
  while (op != END)		  /* while that wasn't END last time... */
    {
      op = OP(s);
      printf("%2d%s", s-r->program, regprop(s)); /* Where, what. */
      next = regnext(s);
      if (next == NULL)			/* Next ptr. */
	printf("(0)");
      else 
	printf("(%d)", (s-r->program)+(next-s));
      s += 3;
      if (op == ANYOF || op == ANYBUT || op == EXACTLY)
	{
	  /* Literal string, where present. */
	  while (*s != '\0') {
	    putchar(*s);
	    s++;
	  }
	  s++;
	}
      putchar('\n');
    }

  /* Header fields of interest. */
  if (r->regstart != '\0')
    printf("start `%c' ", r->regstart);
  if (r->reganch)
    printf("anchored ");
  if (r->regmust != NULL)
    printf("must have \"%s\"", r->regmust);
  printf("\n");
}

/*
 - regprop - printable representation of opcode
 */
static char *
regprop(op)
char *op;
{
  register char *p = NULL;
  static char buf[50];

  (void) strcpy(buf, ":");

  switch (OP(op))
    {
    case BOL: p = "BOL"; break;
    case EOL: p = "EOL"; break;
    case ANY: p = "ANY"; break;
    case ANYOF: p = "ANYOF"; break;
    case ANYBUT: p = "ANYBUT"; break;
    case BRANCH: p = "BRANCH"; break;
    case EXACTLY: p = "EXACTLY"; break;
    case NOTHING: p = "NOTHING"; break;
    case BACK: p = "BACK"; break;
    case END: p = "END"; break;
    case OPEN: p = "OPEN"; break;
    case CLOSE: p = "CLOSE"; break;
    case STAR: p = "STAR"; break;
    case PLUS: p = "PLUS"; break;
    default:
      regerror("corrupted opcode");
      break;
    }
  if (p != NULL) (void) strcat(buf, p);
  return(buf);
}
#endif

/*
 * The following is provided for those people who do not have strcspn() in
 * their C libraries.  They should get off their butts and do something
 * about it; at least one public-domain implementation of those (highly
 * useful) string routines has been published on Usenet.
 */
#ifdef STRCSPN
/*
 * strcspn - find length of initial segment of s1 consisting entirely
 * of characters not from s2
 */

static int
strcspn(s1, s2)
char *s1;
char *s2;
{
	register char *scan1;
	register char *scan2;
	register int count;

	count = 0;
	for (scan1 = s1; *scan1 != '\0'; scan1++) {
		for (scan2 = s2; *scan2 != '\0';)	/* ++ moved down. */
			if (*scan1 == *scan2++)
				return(count);
		count++;
	}
	return(count);
}
#endif
