/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*
     Nature -- a force-directed graph-drawing program
     author: Thomas Fruchterman
     copyright 1990 Thomas Fruchterman
*/
#include "extern.h"
#include "cb-defs.h"
#include <stdio.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/MessageB.h>
#include <mbus/api.h>
#include "graph.h"
#include "callback.h"
#include "version.h"
#include "elision.h"

#define TRUE 1
#define FALSE 0
#define MAX(x, y) ((x > y) ? x : y)
#define MIN(x, y) ((x < y) ? x : y)

coord_t OptDist, OptDistSqr;

Widget toplevel_widget, control_panel;
Graph *graph;
coord_t kscale = 2.0;
char *cb_user = NULL;
char *cb_base_domain = NULL;
char *cb_ui_domain = NULL;
char *cb_browser_domain = NULL;
char *cb_ws_domain = NULL;

int CLOCK = 0;

void graphDetermine()
{
  int clock, iters, verts;
  coord_t cap, offset;
  Vertex *vertex;

  iters = 20;

  if (graph->v == NULL) {		/* check to see we have vertices */
/*
 *  no vertices, so we just merely draw what is there.  Cause chances
 *  are, there is nothing, and the graph will just get cleared
*/
    DrawGraph(VIRT_NODES | VIEW_NODES, TRUE);
    return;
  }

  for (verts = 0, vertex = graph->v; vertex != NULL;
       verts++, vertex = vertex->next);
  OptDist = (coord_t) (W * H) / verts;
  OptDist = (coord_t)sqrt((double)OptDist);
  OptDist = (coord_t) (OptDist * kscale);
  OptDist /= 4.0;
  OptDistSqr = OptDist * OptDist; /* Precompute for efficiency */

/*  iters = iters / 2; */
  cap = 0.1;
  offset = cap / iters;
  for(clock = 0; clock < iters; clock++, cap -= offset) {
    GraphReassign(cap);
  }
  iters /= 2;

  for(clock = 0; clock < iters; clock++)
    GraphReassign(0.015);

  for(clock = 0; clock < iters; clock++)
    GraphReassign(0.004);

  GraphScale();

  DrawGraph(VIEW_NODES | VIRT_NODES, TRUE);
}

void
Quit(widget, client_data, call_data)
     Widget widget;
     caddr_t client_data;
     caddr_t call_data;
{
  extern t_mb_connection conn;
  if (conn != NULL)
  {
    /* Tell the bus that we are out'a here. */
    MBtransmit_Cstring(conn, "(close)");
    MBClose(conn);
  }
/*
 *  We don't care here about freeing memory here.  Just kill off the top level
 *  widget, and see you later.
*/
  XtCloseDisplay(XtDisplay(widget));
  exit(0);
}

void
DestroyShells(widget, client_data, call_data)
     Widget widget;
     caddr_t client_data;
     caddr_t call_data;
{
  struct TableStruct *table_ptr;

/*
 *  Get rid of all of the graphs by skipping down the graph array, deleting
 *  the shell widget, and then deleting the graph.
*/
  for (table_ptr = GraphList; table_ptr; table_ptr = table_ptr->next)
    XtDestroyWidget(table_ptr->graph->parent); /* This will destroy the graphs too!! */
}

/* ------------------------------------------------------------------------ */
void
ConfirmDialog(w, dialog, data)
     Widget w;
     Widget dialog;
     XmAnyCallbackStruct *data;
{
  XtManageChild(dialog);
}

/*
 *  This routine taken from Alan Carroll's widget server.  It creates a
 *  panel of pushbuttons that will either quit the browsers, or destroy
 *  all of the graphs that are currently on the screen
*/
void
CreatePanel()
{
  Widget title_b,quit_b,quit_d,save_d,w,ds_d;
  XmString xm_str;
  int n;
  char buf[80];
  Arg argl[2];
  extern char *cb_user;

  control_panel = XtCreateManagedWidget("controlPanel", xmRowColumnWidgetClass,
					toplevel_widget, NULL, 0);

  sprintf (buf, "%s's Control Panel", cb_user);
  
  XtManageChild(CreateTitle(buf, control_panel));
  XtCreateManagedWidget("separator", xmSeparatorGadgetClass,
			control_panel, NULL, 0);

  xm_str = CtoXmString("Shut down the Graph Browser?");
  XtSetArg(argl[0], XmNmessageString, xm_str);
  quit_d = XmCreateWarningDialog(toplevel_widget, "Quit", argl, 1);
  XtAddCallback(quit_d, XmNokCallback, Quit, NULL);
  w = XmMessageBoxGetChild(quit_d, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(w);
  XtSetArg(argl[0], XmNtitle, "Quit Graph Browser");
  XtSetValues(XtParent(quit_d), argl, 1);
  XtFree(xm_str);

  xm_str = CtoXmString("Destroy all graphs?");
  XtSetArg(argl[0], XmNmessageString, xm_str);
  ds_d = XmCreateWarningDialog(toplevel_widget, "Destroy", argl, 1);
  XtAddCallback(ds_d, XmNokCallback, DestroyShells, NULL);
  w = XmMessageBoxGetChild(ds_d, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(w);
  XtSetArg(argl[0], XmNtitle, "Destroy Graphs");
  XtSetValues(XtParent(ds_d), argl, 1);
  XtFree(xm_str);

  CreatePushButton("Help", control_panel, present_help, NULL);
  CreatePushButton("Destroy Graphs", control_panel, ConfirmDialog, (caddr_t)ds_d);
  CreatePushButton("Quit", control_panel, ConfirmDialog, (caddr_t)quit_d);

}
int
main (argc, argv)
int argc;
char **argv;
{
  extern char *getenv(), *optarg, opt_err_char, *GetUserName();
  extern t_mb_connection conn;
  int c, Usage = 0;
  char *tmp, *host;
  char buf[80];
  int n, port;
  Arg args[10];
  struct timeval tp;
  extern char *cb_user, *cb_base_domain, *cb_ui_domain, *cb_browser_domain;
  extern char *cb_ws_domain;

  tmp = getenv(ENV_MBUS_PORT);
  if (tmp)
      port = atoi(tmp);
  else
      port = DEF_MBUS_PORT;
  host = getenv(ENV_MBUS_HOST);
  cb_base_domain = getenv(ENV_BASE_DOMAIN);
  cb_ui_domain = getenv(ENV_UI_DOMAIN);
  cb_browser_domain = getenv(ENV_BROWSER_DOMAIN);
  cb_ws_domain = getenv(ENV_WS_DOMAIN);

  toplevel_widget = XtInitialize(argv[0], "Browser", NULL, 0, &argc, argv);

  while ((c = getopt(argc, argv, "b:i:u:h:p:")) != -1)
      switch (c) {
      case 'b':
	cb_base_domain = optarg; break;
      case 'i':
	cb_ui_domain = optarg; break;
      case 'u':
	cb_user = optarg; break;
      case 'h':
	host = optarg;
	break;
      case 'p':
	port = atoi(optarg);
	break;
      case '?':
	fprintf (stderr, "Unknown option -%c.\n", opt_err_char);
	Usage = 1;
	break;
      default:
	fprintf (stderr, "Unimplemented option -%c.\n", c);
	break;
      }

  if (Usage)
  {
    printf ("Usage:  %s -b b_domain -i i_domain\n",
	    argv[0]);
    printf ("           -u u_domain -h host -p port\n");
    printf ("where:  b_domain is the base domain (default \"%s\")\n",
	    DEF_BASE_DOMAIN);
    printf ("        i_domain is the user interface domain (default \"%s\")\n",
	    DEF_UI_DOMAIN);
    printf ("        u_domain is the user domain (default $USER)\n");
    printf ("        host is the host of the message bus\n");
    printf ("        port is the port of the message bus\n");
    printf ("\nEnvironment variables used:\n");
    printf ("CB_BASE_DOMAIN     the base domain\n");
    printf ("CB_UI_DOMAIN       the user interface domain\n");
    printf ("CB_USER            the user domain\n");
    printf ("CB_BROWSER_DOMAIN  the browser domain\n");
    printf ("CB_WS_DOMAIN	the widget server domain\n");
    printf ("MBUS_HOST          the message bus host\n");
    printf ("MBUS_PORT          the message bus port\n");
    exit(1);
  }
  
  if (NULL == cb_base_domain) cb_base_domain = DEF_BASE_DOMAIN;
  if (NULL == cb_ui_domain) cb_ui_domain = DEF_UI_DOMAIN;
  if (NULL == cb_user) cb_user = GetUserName();
  if (NULL == cb_browser_domain) cb_browser_domain = DEF_BROWSER_DOMAIN;
  if (NULL == cb_ws_domain) cb_ws_domain = DEF_WS_DOMAIN;

/*
 *  The next line calls a routine in mbus-parse.c. The routine there set
 *  up the stuff for the message bus, and then sets up the stuff for
 *  the motif widgets to wait for input.  This seemed a more logical place
 *  to put this than here..
*/
  SetupBus(port, host);

/*
 *  Set up some initial variable assignment
*/
  BrowserDisplays = (struct DisplayTables *)
    malloc(sizeof(struct DisplayTables));
  BrowserDisplays->common = NULL;
  BrowserDisplays->nodes = NULL;
  BrowserDisplays->edges = NULL;
  CallbackKeys = NULL;
  CallbackFunctions = NULL;
  SuperCounter = 0;
  ElisionPolicy = 0x0;
  SuperNodeLabel = ROOT_IS_LABEL;
  graph = NULL;

  SetPixmap(toplevel_widget);
/*
 *  Seed the random number generator
*/
  if (!gettimeofday(&tp,0))
    srand48((long)tp.tv_sec);
  else
    srand48(!1);

/*
 *  Initialize the parent widget here.  Set up the input call back to
 *  be the routine that reads in the graph definition.  We will then
 *  create PopupShells from there..
*/
  n = 0;
  CreatePanel(cb_user);
  XtSetArg(args[n], XmNallowShellResize, True); n++;
  XtSetArg(args[n], XmNwidth, 130); n++;
  XtSetArg(args[n], XmNheight, 120); n++;
  sprintf (buf, "Browser %s.%s.%s.%s", MAJOR_NUM,
	   MINOR_NUM, CHANGE_NUM, COMPILATION_NUM);
  XtSetArg(args[n], XmNtitle, buf); n++;
  XtSetValues(toplevel_widget, args, n);
  XtAddInput(MBconnection_fd(conn), XtInputReadMask, ParseGraph, NULL);
  XtRealizeWidget(toplevel_widget);
/*
 *  Now, get the initial resources for browser wide stuff
*/
  GetResources(toplevel_widget);
  XtMainLoop();
}
